
/* globals GH, _ */

/**
 * Extension to FormDialog.
 *
 * Customizes success and failure handler
 */
AJS.namespace("GH.Dialog.XBoardExtension");

GH.Dialog.XBoardExtension.getActionInfo = function (form) {
    var workflowPrefix = 'workflow-transition-';
    var formId = form.options.id;
    if (formId.substring(0, workflowPrefix.length) === workflowPrefix) {
        return {
            formId: formId,
            isTransition: true,
            transitionId: formId.substring(workflowPrefix.length)
        };
    } else {
        return {
            formId: formId,
            isTransition: false
        };
    }
};

GH.Dialog.XBoardExtension.handleSuccess = function () {
    // TODO: this will break as soon as we use overrides for dialogs!
    // clear the selected issue overwrite
    GH.IssueOperationShortcuts.clearOverrideSelectedIssue();

    // hide the dialog
    this.hide(true, true);

    // ensure the spinner is hidden
    this._hideloadingIndicator();

    // fetch the form id, which tells us about the type of action that happened
    var actionInfo = GH.Dialog.XBoardExtension.getActionInfo(this);

    // fetch the issue we just changed. Do this before reloading as the selected issue might not be there anymore
    var issueId = GH.IssueOperationShortcuts.getSelectedIssueId();
    var issueKey = GH.IssueOperationShortcuts.getSelectedIssueKey();

    // Reload
    if (GH.WorkController.isActive()) {
        GH.Dialog.handleWorkModeUpdate(issueId, issueKey, actionInfo);
    } else if (GH.PlanController.isActive()) {
        GH.Dialog.handlePlanModeUpdate(issueId, issueKey, actionInfo);
    }

    AJS.$(GH).trigger("issueTransitioned", {
        issueId: issueId
    });

    // show the update message
    GH.Notification.showIssueMessage(issueId);

    // cleanup of transition target status
    GH.WorkDragAndDrop.transitionTargetStatus = undefined;
};

/**
 * Handles the plan mode update in case of a dialog success
 */
GH.Dialog.handlePlanModeUpdate = function (issueId, issueKey, actionInfo) {
    // ignore if this is a subtask edit outcome
    if (!GH.IssueOperationShortcuts.isOverrideSelectedIssueActive()) {
        var position = 0;
        var sprintId = null;
        var model = GH.BacklogModel.findModelWithIssue(issueKey);
        if (model) {
            if (GH.SprintModel.isSprintModel(model)) {
                sprintId = model.getSprintId();
            }
            position = model.getIssueList().getIssueIndex(issueKey);
        }

        GH.PlanController.reload(function () {
            var newIssueKey = GH.BacklogSelectionController.getSelectedIssueKey();
            if (newIssueKey) {
                return;
            }

            // re-get the issue list as it will have been modified underneath us (and possibly even removed)
            var issueKeyToSelect = null;
            var model = null;
            if (sprintId) {
                model = GH.BacklogModel.getSprintModel(sprintId);
            }
            // fallback to backlog
            if (!model) {
                model = GH.BacklogModel.getBacklogModel2();
            }

            GH.BacklogSelectionController.selectIssue(model.getIssueList().getIssueKeyAtIndex(position));
        });
    }
};

/**
 * Handles the work mode update in case of a dialog success
 */
GH.Dialog.handleWorkModeUpdate = function (issueId, issueKey, actionInfo) {

    // create callbacks - we do this before updating the model inline, as e.g. the selection of another issue should happen inline
    var parentCompletedCallback = GH.Dialog.createParentCompletedCheckCallback(issueKey);
    var issueSelectionCallback = GH.Dialog.createSelectMostAppropriateIssueCallback(issueKey);

    // if we transitioned, place the transitioned issue at the most likely location
    if (actionInfo.isTransition) {
        if (GH.WorkDragAndDrop.transitionTargetStatus) {
            var targetStatus = GH.WorkDragAndDrop.transitionTargetStatus;

            // get the current column of the issue
            var model = GH.GridDataController.getModel();
            var issueData = model.getIssueDataByKey(issueKey);
            var issuePosition = model.getIssuePositionByKey(issueKey);
            var columns = model.getColumns();
            var oldColumnId = issuePosition.columnId;
            //var oldColumnId = GH.GridColumnsData.getColumnIdForStatus(columns, issueData.status);
            var newColumnId = GH.GridColumnsData.getColumnIdForStatus(columns, targetStatus);
            model.moveIssueToColumn(issueKey, oldColumnId, newColumnId);

            // update the issue data
            issueData.status = targetStatus;
            model.updateIssueData(issueData);

            // update the ui
            GH.SwimlaneView.rerenderCell(issuePosition.swimlaneId, oldColumnId);
            GH.SwimlaneView.rerenderCell(issuePosition.swimlaneId, newColumnId);

            // directly do the parents check
            parentCompletedCallback();
        } else {

            // register the callback to check whether the parent should be transitioned
            GH.CallbackManager.registerCallback(GH.WorkController.CALLBACK_POOL_LOADED, "ParentWithAllSubsCompletedCheck", parentCompletedCallback);
        }
    }

    // register a callback to select the most appropriate issue after transitioning
    GH.CallbackManager.registerCallback(GH.WorkController.CALLBACK_POOL_LOADED, "SelectMostAppropriateIssueCallback", issueSelectionCallback);

    // force a reload
    GH.WorkController.reload();
};

/**
 * Create a callback that checks whether the parent should be completed
 */
GH.Dialog.createParentCompletedCheckCallback = function (issueKey) {
    return function () {
        // If the issue key we just changed is a subtask
        var model = GH.GridDataController.getModel();
        if (!_.isUndefined(model.data.issues[issueKey]) && !_.isUndefined(model.data.issues[issueKey].parentKey)) {
            // Is the subtask in the final column?
            var parentKey = model.data.issues[issueKey].parentKey;

            if (_.contains(model.data.columnIssueMapping[_.last(model.data.columns).id], issueKey)) {
                // If not, all the subtasks (might be) done! Huzzah!
                // Can we find a subtask that isn't in the final column?
                // Need to remove the last column
                // TODO This should use _.initial but AUI's version of underscore is old, old old.
                var issuesKeysInFinalColumn = model.data.columnIssueMapping[_.last(model.data.columns).id];
                var issuesNotInFinalColumn = _.toArray(model.data.issues).reduce(function (issues, issue) {
                    return issuesKeysInFinalColumn.indexOf(issue.key) > -1 ? issues : issues.concat(issue.key);
                }, []);

                var unfinishedSubtask = _.find(issuesNotInFinalColumn, function (nonFinishedIssueKey) {
                    return model.data.issues[nonFinishedIssueKey].parentKey === parentKey;
                });

                if (!unfinishedSubtask) {
                    GH.WorkController.resolveParentTask(model.data.issues[parentKey]);
                }
            }
        }
    };
};

/**
 * Create a callback that selects the most appropriate issue on the pool
 */
GH.Dialog.createSelectMostAppropriateIssueCallback = function (issueKey) {
    // fetch the current issue position of the changed issue,
    // we'll need this after the reload
    var issuePos = GH.GridDataController.getModel().getIssuePositionByKey(issueKey);

    // callback to select the most appropriate issue in case the issue disappears
    return function () {
        // check whether we still got the issue in the model
        var existingData = GH.GridDataController.getModel().getIssueDataByKey(issueKey);
        if (existingData) {
            // all good, nothing to be done
            return;
        }

        // TODO: should probably be moved into WorkKBNavigation
        // get the issue at the old position, then update the selection
        var newIssueKey = GH.GridDataController.getModel().getIssueKeyAtPosition(issuePos);
        if (newIssueKey) {
            // update the selection
            GH.WorkSelectionController.selectIssue(newIssueKey, { pushState: false, dontUpdateUI: true }); // no push state because we're correcting the URL
            GH.RapidBoard.State.replaceState();
        } else {
            // update the selection
            GH.WorkSelectionController.selectIssue(undefined, { pushState: false, dontUpdateUI: true });
            GH.RapidBoard.State.replaceState();
        }
    };
};

/** Called when the form has been canceled. */
GH.Dialog.XBoardExtension.doCancelCleanup = function () {
    GH.IssueOperationShortcuts.clearOverrideSelectedIssue();

    // clear out the transition target status
    GH.WorkDragAndDrop.transitionTargetStatus = undefined;
};