define('jira-agile/rapid/ops/collection', ['underscore'], function(_) {
    'use strict';

    const Collection = {};

    /**
     * A generic map constructor. This wraps methods around a JavaScript object to formalise the
     * interactions made with it.
     * @optional newValues
     * @return {Collection.Map} a new map instance
     * @constructor
     */
    Collection.Map = function Map(newValues) {
        if (!(this instanceof Collection.Map)) {
            return new Collection.Map(newValues);
        }
        this.clear();
        _.forEach(newValues, function (value, key) {
            this.put(key, value);
        }, this);
        return this;
    };

    /**
     * Clears the contents of the map
     * @return {Collection.Map} this
     */
    Collection.Map.prototype.clear = function () {
        this.length = 0;
        this._valuesByKey = {};
        return this;
    };

    /**
     * iterate over the values in the map, invoking the func for each key-value pair with the specified context
     * @param {Function} func
     * @optional {*} context
     * @return {Collection.Map} this
     */
    Collection.Map.prototype.forEach = function (func, context) {
        _.forEach(this._valuesByKey, func, context);
        return this;
    };

    /**
     * Returns the value for a given key
     * @param {*} key
     * @return {*} value
     */
    Collection.Map.prototype.get = function (key) {
        return this._valuesByKey[key];
    };

    /**
     * Indicates whether an entry is present for a given key
     * @param {*} key
     * @return {Boolean} true if the key is present, false otherwise
     */
    Collection.Map.prototype.has = function (key) {
        return _.has(this._valuesByKey, key);
    };

    /**
     * Returns the keys within the map
     * @return {Array}
     */
    Collection.Map.prototype.keys = function () {
        return _.keys(this._valuesByKey);
    };

    /**
     *
     * @param {*} key
     * @param {*} value
     * @return {Collection.Map} this
     */
    Collection.Map.prototype.put = function (key, value) {
        if (!this.has(key)) {
            this.length++;
        }
        this._valuesByKey[key] = value;
        return this;
    };

    /**
     *
     * @param {*} key
     * @return {Collection.Map} this
     */
    Collection.Map.prototype.remove = function (key) {
        if (this.has(key)) {
            delete this._valuesByKey[key];
            this.length--;
        }
        return this;
    };

    /**
     *
     * @return {Collection.Map} this
     */
    Collection.Map.prototype.replace = Collection.Map;

    /**
     *
     * @return {Object} a plain object representation of the Map
     */
    Collection.Map.prototype.toPlainObject = function () {
        return _.extend({}, this._valuesByKey);
    };

    /**
     *
     * @return {Array}
     */
    Collection.Map.prototype.values = function () {
        return _.values(this._valuesByKey);
    };


    /**
     *
     * @arguments
     * @return {Collection.Set}
     * @constructor
     */
    Collection.Set = function Set(/* items... */) {
        if (!(this instanceof Collection.Set)) {
            // we can't apply the arguments when we construct the set
            // so we have to apply it afterwards
            return Collection.Set.apply(new Collection.Set(), arguments);
        }
        this._map = new Collection.Map();
        this.length = 0;
        this.add.apply(this, arguments);
        return this;
    };

    /**
     * @arguments
     * @return {Collection.Set} this
     */
    Collection.Set.prototype.add = function (/* items... */) {
        var map = this._map;
        _.forEach(arguments, function (item) {
            map.put(item, true);
        }, this);
        this.length = map.length;
        return this;
    };

    /**
     *
     * @return {Collection.Set} this
     */
    Collection.Set.prototype.clear = function () {
        this.length = 0;
        this._map.clear();
        return this;
    };

    /**
     *
     * @param {Function} func
     * @optional {*} context
     * @return {Collection.Set} this
     */
    Collection.Set.prototype.forEach = function (func, context) {
        this._map.forEach(function (flag, item) {
            return func.call(context, item);
        });
        return this;
    };

    /**
     *
     * @param {*} item
     * @return {Boolean}
     */
    Collection.Set.prototype.has = function (item) {
        return this._map.has(item);
    };

    /**
     * @arguments
     * @return {Collection.Set} this
     */
    Collection.Set.prototype.remove = function (/* items... */) {
        var map = this._map;
        _.forEach(arguments, function (item) {
            map.remove(item);
        }, true);
        this.length = map.length;
        return this;
    };

    /**
     * @arguments
     * @return {Collection.Set} this
     */
    Collection.Set.prototype.replace = Collection.Set;

    /**
     *
     * @return {Array}
     */
    Collection.Set.prototype.toArray = function () {
        return this._map.keys();
    };


    /**
     * @arguments
     * @return {Collection.OrderedSet}
     * @constructor
     */
    Collection.OrderedSet = function OrderedSet(/* items... */) {
        if (!(this instanceof Collection.OrderedSet)) {
            // we can't apply the arguments when we construct the set
            // so we have to apply it afterwards
            return Collection.OrderedSet.apply(new Collection.OrderedSet(), arguments);
        }
        this.clear();
        this.push.apply(this, arguments);
        return this;
    };

    /**
     *
     * @return {Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.clear = function () {
        this.length = 0;
        this._values = [];
        this._positionByValue = new Collection.Map();
        return this;
    };

    /**
     * @arguments
     * @return {{Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.push = function (/* items... */) {
        var positionByValue = this._positionByValue;
        var values = this._values;
        _.forEach(arguments, function (item) {
            if (!positionByValue.has(item)) {
                positionByValue.put(item, values.length);
                values.push(item);
            }
        }, this);
        this._values = values;
        this.length = values.length;
        return this;
    };

    /**
     *
     * @return {*}
     */
    Collection.OrderedSet.prototype.first = function () {
        return this.get(0);
    };

    /**
     *
     * @param {Function} func
     * @optional {*} context
     * @return {Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.forEach = function (func, context) {
        _.forEach(this._values, func, context);
        return this;
    };

    /**
     *
     * @param {Number} index
     * @return {*}
     */
    Collection.OrderedSet.prototype.get = function (index) {
        return this._values[index];
    };

    /**
     *
     * @param {*} item
     * @return {Boolean}
     */
    Collection.OrderedSet.prototype.has = function (item) {
        return this._positionByValue.has(item);
    };

    /**
     *
     * @param {*} item
     * @return {Number}
     */
    Collection.OrderedSet.prototype.indexOf = function (item) {
        var index = this._positionByValue.get(item);
        return _.isUndefined(index) ? -1 : index;
    };

    /**
     *
     * @return {*}
     */
    Collection.OrderedSet.prototype.last = function () {
        return this.get(this.length - 1);
    };

    /**
     *
     * @return {*}
     */
    Collection.OrderedSet.prototype.pop = function () {
        var item = this._values.pop();
        this._positionByValue.remove(item);
        this.length = this._values.length;
        return item;
    };

    /**
     * @arguments
     * @return {Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.remove = function (/* items... */) {
        var values = this._values;
        _.forEach(arguments, function (item) {
            var index = _.indexOf(values, item);
            if (index !== -1) {
                values = values.slice(0, index).concat(values.slice(index + 1));
            }
        });
        this.replace.apply(this, values);
        return this;
    };

    /**
     * @arguments
     * @return {Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.replace = Collection.OrderedSet;

    /**
     *
     * @return {*}
     */
    Collection.OrderedSet.prototype.shift = function () {
        var values = this._values;
        var item = values.shift();
        this.replace.apply(this, values);
        return item;
    };

    /**
     *
     * @return {Array}
     */
    Collection.OrderedSet.prototype.toArray = function () {
        var arr = [];
        this.forEach(function (item) {
            arr.push(item);
        });
        return arr;
    };

    /**
     *
     * @arguments
     * @return {Collection.OrderedSet} this
     */
    Collection.OrderedSet.prototype.unshift = function (/* items... */) {
        var positionByValue = this._positionByValue;
        var values = this._values;
        // TODO: FIX ME should be inserted in reverse order
        _.forEach(arguments, function (item) {
            if (!positionByValue.has(item)) {
                values.unshift(item);
            }
        }, this);
        this.replace.apply(this, values);
        return this;
    };


    /**
     *
     * @param {Object} newValues
     * @return {Collection.OrderedMap}
     * @constructor
     */
    Collection.OrderedMap = function OrderedMap(newValues) {
        if (!(this instanceof Collection.OrderedMap)) {
            return new Collection.OrderedMap(newValues);
        }
        this.clear();
        _.forEach(newValues, function (value, key) {
            this.put(key, value);
        }, this);
        return this;
    };

    /**
     *
     * @return {Collection.OrderedMap} this
     */
    Collection.OrderedMap.prototype.clear = function () {
        this.length = 0;
        this._keys = [];
        this._values = [];
        this._indexByKey = new Collection.Map();
        this._valueByKey = new Collection.Map();
        return this;
    };

    /**
     *
     * @param {Function} func
     * @optional {*} context
     * @return {Collection.OrderedMap} this
     */
    Collection.OrderedMap.prototype.forEach = function (func, context) {
        var keys = this._keys;
        var values = this._values;
        for (var i = 0, l = keys.length; i < l; i++) {
            func.call(context, values[i], keys[i], i);
        }
        return this;
    };

    /**
     *
     * @param {*} key
     * @return {*}
     */
    Collection.OrderedMap.prototype.get = function (key) {
        return this._valueByKey.get(key);
    };

    /**
     *
     * @param {Number} index
     * @return {*}
     */
    Collection.OrderedMap.prototype.getByIndex = function (index) {
        return this._values[index];
    };

    /**
     *
     * @param {Number} index
     * @return {*}
     */
    Collection.OrderedMap.prototype.getKeyByIndex = function (index) {
        return this._keys[index];
    };

    /**
     *
     * @param {*} key
     * @return {Boolean}
     */
    Collection.OrderedMap.prototype.has = function (key) {
        return this._indexByKey.has(key);
    };

    /**
     *
     * @param {*} key
     * @return {Number}
     */
    Collection.OrderedMap.prototype.indexOf = function (key) {
        return this._indexByKey.indexOf(key);
    };

    /**
     *
     * @return {Array}
     */
    Collection.OrderedMap.prototype.keys = function () {
        return this._keys.slice(0);
    };

    /**
     *
     * @param {*} key
     * @param {*} value
     * @return {Collection.OrderedMap} this
     */
    Collection.OrderedMap.prototype.put = function (key, value) {
        if (!this.has(key)) {
            this._indexByKey.put(key, this.length);
            this._valueByKey.put(key, value);
            this._keys.push(key);
            this._values.push(value);
            this.length++;
        }
        return this;
    };

    /**
     *
     * @param {*} key
     * @return {Collection.OrderedMap} this
     */
    Collection.OrderedMap.prototype.remove = function (key) {
        var index = this._indexByKey.get(key);
        if (index !== -1) {
            this._indexByKey.remove(key);
            this._valueByKey.remove(key);
            var keys = this._keys;
            this._keys = keys.slice(0, index).concat(keys.slice(index + 1));
            var values = this._values;
            this._values = values.slice(0, index).concat(values.slice(index + 1));
            this.length--;
        }
        return this;
    };

    /**
     * @param {Object} newValues
     * @return {Collection.OrderedMap} this
     */
    Collection.OrderedMap.prototype.replace = Collection.OrderedMap;

    /**
     *
     * @return {Object}
     */
    Collection.OrderedMap.prototype.toPlainObject = function () {
        return this._valueByKey.toPlainObject();
    };

    /**
     *
     * @return {Array}
     */
    Collection.OrderedMap.prototype.values = function () {
        return this._values.slice(0);
    };

    return Collection;

});

AJS.namespace("GH.Collection", null, require("jira-agile/rapid/ops/collection"));
