/**
 * Estimation configuration tab
 */
define("jire-agile/rapid/configuration/working-days-config", ["jira-agile/rapid/ajax", "underscore", "jquery"], function (Ajax, _, $) {
    "use strict";

    var WorkingDaysConfig = {};

    WorkingDaysConfig.ISO8601Format = '%Y-%m-%d';

    /*
     * Holds the model for the configuration.
     */
    WorkingDaysConfig.model = undefined;

    /**
     * Initializes the Estimation config.
     */
    WorkingDaysConfig.init = function (data) {
        WorkingDaysConfig.model = data.workingDaysConfig;

        WorkingDaysConfig.renderTab();
    };

    /**
     * Renders the edit form using the locally stored model
     */
    WorkingDaysConfig.renderTab = function () {

        // render the shell
        $('#ghx-config-time').html(GH.tpl.workingdaysconfig.renderChartConfig());

        WorkingDaysConfig.renderTimeZoneConfig();

        WorkingDaysConfig.renderWeeklyConfig();

        WorkingDaysConfig.renderDailyConfig();
    };

    //
    // Time Zone
    //

    WorkingDaysConfig.renderTimeZoneConfig = function () {
        // put together all information we need
        var editModel = WorkingDaysConfig.model.timeZoneEditModel;

        var regions = editModel.regions;

        // figure out what the selected region is
        var selectedTimeZoneId = WorkingDaysConfig.model.timeZoneEditModel.currentTimeZoneId;
        var selectedRegionKey = WorkingDaysConfig.getRegionKeyForTimeZoneId(selectedTimeZoneId);

        var data = {
            canEdit: WorkingDaysConfig.model.canEdit,
            selectedRegionKey: selectedRegionKey,
            regions: regions,
            zones: WorkingDaysConfig.getZonesForRegion(selectedRegionKey),
            selectedTimeZoneId: selectedTimeZoneId
        };

        // render the regions
        $('#ghx-timezone-config').html(GH.tpl.workingdaysconfig.renderTimeZoneConfig(data));

        $('#ghx-region-select').change(WorkingDaysConfig.updateRegionZones);

        $('#ghx-timezone-select').change(WorkingDaysConfig.handleTimezoneUpdate);
    };

    WorkingDaysConfig.updateRegionZones = function () {
        // fetch the values for that zone
        var regionKey = $('#ghx-region-select').val();
        var zones = WorkingDaysConfig.getZonesForRegion(regionKey);

        var data = {
            zones: zones
        };

        var $timezoneSelect = $('#ghx-timezone-select');
        $timezoneSelect.html(GH.tpl.workingdaysconfig.renderTimeZoneOptions(data));

        // submit the new value
        WorkingDaysConfig.handleTimezoneUpdate();
    };

    WorkingDaysConfig.getZonesForRegion = function (regionKey) {
        return _.filter(WorkingDaysConfig.model.timeZoneEditModel.timeZones, function (zone) {
            return zone.regionKey === regionKey;
        });
    };

    WorkingDaysConfig.getRegionKeyForTimeZoneId = function (timeZoneId) {
        return _.find(WorkingDaysConfig.model.timeZoneEditModel.timeZones, function (zone) {
            return zone.timeZoneId === timeZoneId;
        }).regionKey;
    };

    WorkingDaysConfig.handleTimezoneUpdate = function () {
        var timezoneId = $('#ghx-timezone-select').val();

        WorkingDaysConfig.setTimezone(timezoneId);
    };

    //
    // Week days
    //

    WorkingDaysConfig.renderWeeklyConfig = function () {
        var $weekdaysConfig = $('#ghx-weekdays-config');
        $weekdaysConfig.html(GH.tpl.workingdaysconfig.renderWeekdaysConfig({
            canEdit: WorkingDaysConfig.model.canEdit,
            weekDays: WorkingDaysConfig.model.weekDays
        }));

        // attach change listener
        $weekdaysConfig.find('input.checkbox').change(WorkingDaysConfig.saveWeekDays);
    };

    // Daily config

    WorkingDaysConfig.renderDailyConfig = function () {
        $('#ghx-daily-config').html(GH.tpl.workingdaysconfig.renderDailyConfig({
            canEdit: WorkingDaysConfig.model.canEdit
        }));

        // The calendar unfortunately doesn't have clear events that tell us that a user selected something,
        // as opposed to just flipping through the months. Both result in onUpdate events!
        // The only hacky way to know whether this is a valid selection is to compare the time we get the update to the
        // time the calendar is closed.
        var lastSelectTimeStamp = 0;
        Calendar.setup({
            firstDay: 0,
            cache: false,
            inputField: 'ghx-add-off-day',
            button: 'ghx-add-off-day-button',
            align: 'Tr',
            singleClick: true,
            positionStyle: 'fixed',
            ifFormat: GH.ChartTimeFrames.dateFormat,
            onUpdate: function onUpdate() {
                lastSelectTimeStamp = new Date().getTime();
            },
            onClose: function onClose(cal) {
                // check whether the select was recent
                if (new Date().getTime() - lastSelectTimeStamp < 100) {
                    WorkingDaysConfig.handleAddDay();
                }
                cal.hide();
            }
        });

        WorkingDaysConfig.renderDays();
    };

    WorkingDaysConfig.renderDays = function () {
        // put together the render data
        var data = {
            days: []
        };

        var nonWorkingDays = WorkingDaysConfig.model.nonWorkingDays;

        // first sort the days
        nonWorkingDays.sort(function (a, b) {
            return a.iso8601Date.localeCompare(b.iso8601Date);
        });

        _.each(nonWorkingDays, function (nonWorkingDay) {

            var date = WorkingDaysConfig.parseIsoDateString(nonWorkingDay.iso8601Date);
            date = new Date(date); // because Date.parse doesn't return the Calendar extended Date object :/
            var renderedDate = date.print(GH.ChartTimeFrames.dateFormat);
            data.days.push({
                displayValue: renderedDate,
                value: nonWorkingDay.iso8601Date
            });
        });

        // render the off days
        var $existingOffDays = $('#ghx-existing-off-days');
        $existingOffDays.html(GH.tpl.workingdaysconfig.renderDays({
            days: data.days,
            canEdit: WorkingDaysConfig.model.canEdit
        }));

        // handle clicks on the different buttons
        $existingOffDays.find('.aui-icon-close').click(WorkingDaysConfig.handleRemoveDay);
    };

    WorkingDaysConfig.handleAddDay = function () {
        var dayString = $('#ghx-add-off-day').val();
        if (!dayString) {
            return;
        }
        var day = Date.parseDate(dayString, GH.ChartTimeFrames.dateFormat);

        // store the day
        WorkingDaysConfig.addDay(day);

        // update the model
        WorkingDaysConfig.addDayToModel(day);

        // then rerender
        WorkingDaysConfig.renderDays();

        $('#ghx-add-off-day').val('');
    };

    WorkingDaysConfig.addDayToModel = function (day) {
        var iso8601Date = WorkingDaysConfig.toISODate(day);
        var nonWorkingDays = WorkingDaysConfig.model.nonWorkingDays;

        var hasDate = _.find(nonWorkingDays, function (day) {
            return day.iso8601Date === iso8601Date;
        });
        if (!hasDate) {
            nonWorkingDays.push({
                iso8601Date: iso8601Date
            });
        }
    };

    WorkingDaysConfig.removeDayFromModel = function (day) {
        var iso8601Date = WorkingDaysConfig.toISODate(day);
        var nonWorkingDays = WorkingDaysConfig.model.nonWorkingDays;

        WorkingDaysConfig.model.nonWorkingDays = _.reject(nonWorkingDays, function (day) {
            return day.iso8601Date === iso8601Date;
        });
    };

    WorkingDaysConfig.handleRemoveDay = function (event) {
        var elem = $(this).closest('.js-day-off');
        var date = WorkingDaysConfig.parseIsoDateString(elem.attr('data-date'));

        WorkingDaysConfig.removeDay(date);

        // remove from model
        WorkingDaysConfig.removeDayFromModel(date);

        // rerender
        WorkingDaysConfig.renderDays();
    };

    WorkingDaysConfig.toISODate = function (day) {
        return day.print("%Y-%m-%d");
    };

    WorkingDaysConfig.parseIsoDateString = function (iso8601Date) {
        return Date.parseDate(iso8601Date, WorkingDaysConfig.ISO8601Format);
    };

    // AJAX

    WorkingDaysConfig.setTimezone = function (timeZoneId) {
        // put together the data to submit. We update all off days at once
        var request = {
            rapidViewId: WorkingDaysConfig.model.rapidViewId,
            timeZoneId: timeZoneId
        };

        Ajax.put({
            url: '/rapidviewconfig/workingdays/timezone',
            data: request
        });
    };

    WorkingDaysConfig.saveWeekDays = function () {
        // put together the data to submit. We update all off days at once
        var request = {
            rapidViewId: WorkingDaysConfig.model.rapidViewId,
            monday: $('#ghx-monday').is(':checked'),
            tuesday: $('#ghx-tuesday').is(':checked'),
            wednesday: $('#ghx-wednesday').is(':checked'),
            thursday: $('#ghx-thursday').is(':checked'),
            friday: $('#ghx-friday').is(':checked'),
            saturday: $('#ghx-saturday').is(':checked'),
            sunday: $('#ghx-sunday').is(':checked')
        };

        Ajax.put({
            url: '/rapidviewconfig/workingdays/weekdays',
            data: request
        });
    };

    WorkingDaysConfig.addDay = function (day) {
        var request = {
            rapidViewId: WorkingDaysConfig.model.rapidViewId,
            iso8601Date: WorkingDaysConfig.toISODate(day)
        };

        Ajax.post({
            url: '/rapidviewconfig/workingdays/nonworkingday/add',
            data: request
        });
    };

    WorkingDaysConfig.removeDay = function (day) {
        var request = {
            rapidViewId: WorkingDaysConfig.model.rapidViewId,
            iso8601Date: WorkingDaysConfig.toISODate(day)
        };

        Ajax.post({
            url: '/rapidviewconfig/workingdays/nonworkingday/remove',
            data: request
        });
    };

    return WorkingDaysConfig;
});

AJS.namespace("GH.WorkingDaysConfig", null, require("jire-agile/rapid/configuration/working-days-config"));