/**
 * Swimlanes configuration tab
 */
define('jira-agile/rapid/configuration/swimlane-config', [
    "jira-agile/rapid/ajax",
    "jira/util/formatter",
    "aui/restful-table",
    "underscore",
    "jquery",
    "jira/util/strings",
    "jira-agile/rapid/analytics-tracker",
    "jira-agile/rapid/configuration/delete-confirmation"
], function(Ajax, formatter, RestfulTable, _, $, Strings, AnalyticsTracker, DeleteConfirmation) {

    "use strict";

    const Swimlanes = {};

    Swimlanes.Analytics = {};

    Swimlanes.Analytics.base = new AnalyticsTracker("gh.swimlane");
    Swimlanes.Analytics.add = new AnalyticsTracker("gh.swimlane", "create");
    Swimlanes.Analytics.edit = new AnalyticsTracker("gh.swimlane", "edit");
    Swimlanes.Analytics.remove = new AnalyticsTracker("gh.swimlane", "delete");

    /**
     * All available swimlane strategies
     */
    Swimlanes.strategies = [
        {
            id: 'custom',
            primarySelectId: 'custom',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.custom'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.custom.desc')
        },
        {
            id: 'parentChild',
            primarySelectId: 'parentChild',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.parents'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.parents.desc')
        },
        {
            id: 'assignee',
            primarySelectId: 'assignee',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees.desc')
        },
        {
            id: 'assigneeUnassignedFirst',
            primarySelectId: 'assignee',
            inPrimarySelection: false,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees.desc')
        },
        {
            id: 'epic',
            primarySelectId: 'epic',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.epics'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.epics.desc')
        },
        {
            id: 'project',
            primarySelectId: 'project',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.projects'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.projects.desc')
        },
        {
            id: 'none',
            primarySelectId: 'none',
            inPrimarySelection: true,
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.none'),
            description: formatter.I18n.getText('gh.rapid.swimlane.strategy.none.desc')
        }
    ];

    Swimlanes.assigneeStrategies = [
        {
            id: 'assignee',
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees.unassigned.option.1')
        },
        {
            id: 'assigneeUnassignedFirst',
            name: formatter.I18n.getText('gh.rapid.swimlane.strategy.assignees.unassigned.option.2')
        }
    ];


    Swimlanes.TableColumns = [
        {
            id        : 'name',
            header    : formatter.I18n.getText('gh.rapid.config.swimlanes.label.name'),
            fieldName : 'name'
        },
        {
            id        : 'query',
            header    : formatter.I18n.getText('gh.rapid.config.swimlanes.label.jql'),
            fileName  : 'query',
            editView  : GH.RapidBoard.Config.Util.JQLEditView.extend({ namespace: 'ghx-swimlane' })
        },
        {
            id        : 'description',
            header    : formatter.I18n.getText('gh.rapid.config.swimlanes.label.description'),
            fileName  : 'description',
            editView  : GH.RapidBoard.Config.Util.TextAreaEditView
        }
    ];


    /**
     * Holds the model for the configuration.
     */
    Swimlanes.model = undefined;

    /**
     * Initializes the swimlanes config.
     */
    Swimlanes.init = function(data) {
        Swimlanes.model = data.swimlanesConfig;
        Swimlanes.boardType = data.isSprintSupportEnabled ? "scrum" : "kanban";
        Swimlanes.renderConfig();
    };

    Swimlanes.getStrategyById = function(id) {
        return _.find(Swimlanes.getStrategies(), function(strategy) {
            return strategy.id == id;
        });
    };

    Swimlanes.renderConfig = function() {
        var model = Swimlanes.model;
        var canEdit = model.canEdit;

        var currentStrategyId = model.swimlaneStrategy;

        // fetch the chosen strategy, fall back to none
        var strategy = Swimlanes.getStrategyById(currentStrategyId);
        if (!strategy) {
            strategy = Swimlanes.getStrategyById('none');
        }

        // render the strategies selector
        $('#ghx-config-swimlanes').html(GH.tpl.swimlaneconfig.renderSwimlaneConfigView({
            canEdit:canEdit,
            selectedStrategy: strategy,
            strategies: Swimlanes.getStrategies()
        }));

        Swimlanes.updateStrategyConfig(strategy);

        // attach listener to dropdown
        $('#ghx-swimlanestrategy-select').change(Swimlanes.saveStrategySelection);
    };

    Swimlanes.getStrategies = function(){
        return _.filter(Swimlanes.strategies, function(strategy){
            return strategy.boardType ? strategy.boardType == Swimlanes.boardType : true;
        });
    };

    Swimlanes.updateStrategyConfig = function(strategy) {
        $('#ghx-swimlanestrategy-description').text(strategy.description);
        switch(strategy.id) {
        case 'custom':
            Swimlanes.initializeCustomConfig();
            return;
        case 'assignee':
            /* falls through */
        case 'assigneeUnassignedFirst':
            Swimlanes.initializeAssigneeConfig(strategy);
            return;
        case 'parentChild':
            /* falls through */
        case 'none':
            /* falls through */
        default:
            Swimlanes.initializeNone();
        }
    };

    /**
     * Updates the strategy
     */
    Swimlanes.saveStrategySelection = function(event) {
        // fetch new strategy
        var primaryStrategyId = $('#ghx-swimlanestrategy-select').val();
        var assigneeStrategyId = $('#ghx-assignee-strategy-select').val();

        // get the strategy
        var newStrategyId = primaryStrategyId === 'assignee' && assigneeStrategyId ? assigneeStrategyId : primaryStrategyId;
        var strategy = Swimlanes.getStrategyById(newStrategyId);
        if (!strategy) {
            return;
        }

        // update the model
        var model = Swimlanes.model.swimlaneStrategy = strategy.id;

        // update the ui
        Swimlanes.updateStrategyConfig(strategy);

        // save the config change
        var rapidViewId = Swimlanes.model.rapidViewId;
        Ajax.put({
            url : '/rapidviewconfig/swimlaneStrategy',
            data : {
                id: rapidViewId,
                swimlaneStrategyId: strategy.id
            }
        }).done(function() {
            JIRA.trace("gh.swimlanes.strategy.saved");
        });
    };

    // Strategies
    Swimlanes.initializeAssigneeConfig = function(selectedStrategy) {
        var model = Swimlanes.model;
        var canEdit = model.canEdit;
        $('#ghx-swimlane-strategy-config').html(GH.tpl.swimlaneconfig.renderAssigneeStrategyConfig({
            canEdit: canEdit,
            assigneeStrategies: Swimlanes.assigneeStrategies,
            selectedStrategy: selectedStrategy
        }));
        $('#ghx-assignee-strategy-select').change(Swimlanes.saveStrategySelection);
    };

    Swimlanes.initializeNone = function() {
        $('#ghx-swimlane-strategy-config').empty();
    };

    // Custom strategy

    /**
     * Renders the edit form using the locally stored model
     */
    Swimlanes.initializeCustomConfig = function() {
        var model = Swimlanes.model;
        var canEdit = model.canEdit;
        var rapidViewId = model.rapidViewId;
        var swimlanes = model.swimlanes;

        // apply the fact that we are readonly to all swimlanes, as the row renderer only gets the data object for the row
        // only do this if we can't edit as this won't cut it for updated rows (there the response of the rest calls is directly
        // used, so we have no way of adapting the entries upon success.
        // Not setting this for canEdit = true ensures nobody misuses readonly for that matter...
        if (! canEdit) {
            for (var i=0; i < swimlanes.length; i++) {
                swimlanes[i].readonly = true;
            }
        }

        // render the table
        $('#ghx-swimlane-strategy-config').html('<table id="ghx-swimlane-table" class="aui"></table>');

        // setup the restful table
        var restfulTable = new RestfulTable({
            el: $("#ghx-swimlane-table"), // table to add entries to. Entries are appended to the tables <tbody> element
            resources: {
                all: function(next) {
                    next(Swimlanes.model.swimlanes);
                },
                self: Ajax.buildRestUrl('/swimlanes/' + rapidViewId + '/')
            },
            columns: Swimlanes.TableColumns,
            deleteConfirmationCallback: function (model) {
                return DeleteConfirmation.show(
                    GH.tpl.rapid.view.deleteForm({
                        message: formatter.I18n.getText('gh.rapid.swimlane.delete.confirm', Strings.escapeHtml(String(model.name)))
                    })
                );
            },
            views: {
                row : Swimlanes.Row,
                editRow: Swimlanes.EditRow
            },
            allowEdit: canEdit,
            allowDelete: canEdit,
            allowReorder: canEdit,
            allowCreate: canEdit
        });

        // Keep the restful table model in sync with our own, so switching tabs and back again doesn't cause confusion
        var restfulTableModels = restfulTable.getModels();

        restfulTableModels.on("add", function (swimlane, models) {
            Swimlanes.model.swimlanes = models.toJSON();
        });

        restfulTableModels.on("remove", function (swimlane, models) {
            Swimlanes.model.swimlanes = models.toJSON();
        });

        restfulTableModels.on("change", function(swimlane, models) {
            var self = this;
            // need to debounce this as at the time, this (the collection) has not been updated with the updated row
            setTimeout(function() {
                Swimlanes.model.swimlanes = self.toJSON();
            }, 0);
        });

        // there is no Backbone collections event to bind on for re-ordering, so we bind on the table's event
        $(restfulTable).bind(RestfulTable.Events.REORDER_SUCCESS, function () {
            // this.getModels() is not sorted - explicitly sort by DOM order when the event is fired
            Swimlanes.model.swimlanes = this.getModels().sort().toJSON();
        });

        //restful table sortable is initialized when user can edit, don't set options when sortable is not initialized.
        if (canEdit) {
            // Only make rows with js-sortable class sortable
            restfulTable.$tbody.sortable('option', 'items', '> .js-sortable');
        }

        if (model.canEdit) {
            // register jql auto complete
            _.delay(GH.JQLAutoComplete.initialize, 0, 'ghx-swimlane-jql-create', 'ghx-swimlane-jql-errormsg-create');

            // initializing the autocomplete when the field is hidden results in incorrect dropdown positioning
            $(document).bind("tabSelect", function(event, data) {
                GH.JQLAutoComplete.fixSuggestionPositioning('ghx-swimlane-jql-create');
            });

            $(restfulTable).bind(RestfulTable.Events.ROW_REMOVED, function(evt, row) {
                Swimlanes.Analytics.remove.trigger(row.model.attributes.query);
            });
        }
    };

    /**
     * Table view row object
     */
    Swimlanes.Row = RestfulTable.Row.extend({

        render: function () {
            RestfulTable.Row.prototype.render.apply(this, arguments);
            // set the correct class onto the row - either the row is sortable or it is the default row
            var isDefaultSwimlane = this.model.get('isDefault');
            this.$el
                .toggleClass('js-sortable', !isDefaultSwimlane)
                .toggleClass('ghx-default-swimlane', isDefaultSwimlane);

            if (isDefaultSwimlane) {
                this.$el.find('.aui-restfultable-delete').remove();
            }
            return this;
        }
    });

    /**
     * Table edit row object
     */
    Swimlanes.EditRow = RestfulTable.EditRow.extend({
        // used to handle JQL errors - so form is not submitted
        submit: function () {
            var values = this.$el.serializeObject();
            var oldJql = this.model.attributes.query;
            var newJql = values && values.query ? values.query : "";
            if(this.model.id)
            {
                // if the model existed already, then it's an edit
                Swimlanes.Analytics.edit.trigger("old: " + oldJql + ", new: " + newJql);
            } else {
                // else we have a create event
                Swimlanes.Analytics.add.trigger(newJql);
            }

            return RestfulTable.EditRow.prototype.submit.apply(this, arguments);
        }
    });

    return Swimlanes;
});

AJS.namespace('GH.RapidBoard.Config.Swimlanes', null, require('jira-agile/rapid/configuration/swimlane-config'));
