AJS.test.require('com.pyxis.greenhopper.jira:gh-manage-boards', function () {
    var ManageBoardsModel = require("jira-agile/rapid/configuration/manage-boards-model");

    var generateBoards = function generateBoards(number) {
        var boards = [];
        for (var i = 1; i <= number; i++) {
            boards.push({
                'id': i,
                'name': 'Board ' + i,
                'canEdit': true,
                'sprintSupportEnabled': i % 2 === 0,
                'filter': {
                    'id': i,
                    'name': 'Filter for board ' + i,
                    'permissionEntries': [{
                        'values': [{
                            'type': 'Project',
                            'name': 'Project ' + i
                        }]
                    }]
                },
                'boardAdmins': {
                    'userKeys': [{
                        'key': 'admin',
                        'displayName': 'admin'
                    }],
                    'groupKeys': []
                }
            });
        }
        return {
            views: boards,
            startAt: 0,
            total: boards.length
        };
    };

    module('ManageBoardsModel', {
        setup: function setup() {
            this.generateBoards = generateBoards;
            this.model = new ManageBoardsModel();
            this.handler = sinon.spy();
            this.model.on('change:boards', this.handler);
        }
    });

    test('should have proper categories set', function () {
        var categoryIds = _.pluck(this.model.getBoardsCategories(), 'id');
        deepEqual(categoryIds, ['all']);
        equal(this.model.getCurrentCategory().id, 'all');

        this.model.setRecentBoards(this.generateBoards(5).views);

        categoryIds = _.pluck(this.model.getBoardsCategories(), 'id');
        deepEqual(categoryIds, ['recent', 'all']);
        equal(this.model.getCurrentCategory().id, 'all');

        this.model.setRecentBoards([]);

        categoryIds = _.pluck(this.model.getBoardsCategories(), 'id');
        deepEqual(categoryIds, ['all']);
        equal(this.model.getCurrentCategory().id, 'all');
    });

    test('should trigger event on category change', function () {
        this.model.setRecentBoards(this.generateBoards(5).views);
        sinon.assert.calledOnce(this.handler);

        this.model.setCurrentCategory('all');
        sinon.assert.calledTwice(this.handler);
    });

    test('should properly return current boards', function () {
        this.model.setRecentBoards(this.generateBoards(2).views);
        this.model.populate(this.generateBoards(20));

        equal(this.model.getCurrentCategory().id, 'all');
        deepEqual(this.model.getCurrentBoards(), this.model.getAllBoards());

        this.model.setCurrentCategory('recent');
        equal(this.model.getCurrentCategory().id, 'recent');
        deepEqual(this.model.getCurrentBoards(), this.model.getRecentBoards());
    });

    test('should properly return if boards are filterable', function () {
        equal(this.model.isFilterable(), false);

        this.model.populate(this.generateBoards(10));
        equal(this.model.isFilterable(), true);

        this.model.setRecentBoards(this.generateBoards(2).views);
        this.model.setCurrentCategory('recent');

        equal(this.model.isFilterable(), false);

        this.model.setCurrentCategory('all');
        equal(this.model.isFilterable(), true);
    });

    test('should return proper pagination parameters', function () {
        this.model.populate(this.generateBoards(20));
        var paginationParams = this.model.getPaginationState();

        equal(paginationParams.firstPage, 1);
        equal(paginationParams.totalPages, 1);

        this.model.populate(this.generateBoards(60));
        paginationParams = this.model.getPaginationState();

        equal(paginationParams.firstPage, 1);
        equal(paginationParams.lastPage, 3);
        equal(paginationParams.totalPages, 3);

        this.model.setRecentBoards(this.generateBoards(2).views);
        this.model.setCurrentCategory('recent');
        paginationParams = this.model.getPaginationState();

        equal(paginationParams.firstPage, 0);
        equal(paginationParams.totalPages, 0);
    });

    test('should change pages', function () {
        this.model.populate(this.generateBoards(60));
        var paginationParams = this.model.getPaginationState();

        equal(paginationParams.currentPage, 1);
        equal(paginationParams.totalPages, 3);

        this.model.changePage(3);
        paginationParams = this.model.getPaginationState();
        equal(paginationParams.currentPage, 3);
        equal(paginationParams.totalPages, 3);

        this.model.changePage(-3);
        paginationParams = this.model.getPaginationState();
        equal(paginationParams.currentPage, 1);
        equal(paginationParams.totalPages, 3);

        this.model.changePage(5);
        paginationParams = this.model.getPaginationState();
        equal(paginationParams.currentPage, 3);
        equal(paginationParams.totalPages, 3);
    });

    test('should remove boards', function () {
        this.model.populate(this.generateBoards(10));
        this.model.setRecentBoards(this.generateBoards(2).views);

        equal(this.model.getAllBoards().length, 10);
        equal(this.model.getRecentBoards().length, 2);

        this.model.removeBoard(5);
        equal(this.model.getAllBoards().length, 9);
        equal(this.model.getRecentBoards().length, 2);
        deepEqual(_.pluck(this.model.getAllBoards(), 'id'), [1, 2, 3, 4, 6, 7, 8, 9, 10]);

        this.model.removeBoard(2);
        equal(this.model.getAllBoards().length, 8);
        equal(this.model.getRecentBoards().length, 1);
        deepEqual(_.pluck(this.model.getAllBoards(), 'id'), [1, 3, 4, 6, 7, 8, 9, 10]);
        deepEqual(_.pluck(this.model.getRecentBoards(), 'id'), [1]);
    });
});