/* globals Calendar */

/**
 * Inline editable element support.
 *
 * Binds to a specific element (e.g. given by selector) and turns the element into an edit view when being clicked.
 */
define('jira-agile/rapid/configuration/inline-edit-field', [
    "jira-agile/rapid/validation",
    "jira-agile/rapid/logger",
    "jira-agile/rapid/ajax",
    "jira/util/formatter",
    "jira/ajs/select/single-select",
    "jira/ajs/list/item-descriptor",
    "jira/ajs/list/group-descriptor",
    "underscore",
    "jquery",
    "jira/util/logger"
], function(
    Validation,
    Logger,
    Ajax,
    formatter,
    SingleSelect,
    ItemDescriptor,
    GroupDescriptor,
    _,
    $,
    JiraLogger
) {
    "use strict";

    const InlineEditable = {};

    //
    // Generic helper functions
    //

    /**
     * Fetches data-fieldname and data-fieldvalue element attributes.
     */
    InlineEditable.defaultFieldData = function(viewElement, options) {
        var fieldName = viewElement.attr('data-fieldname');
        var fieldValue = viewElement.attr('data-fieldvalue');
        var defaultValue = viewElement.attr('data-defaultvalue');
        var maxDisplayLength = viewElement.attr('data-maxdisplaylength');
        var ariaLabel = viewElement.attr('data-ariaLabel');

        return {
            fieldName: fieldName,
            fieldValue: fieldValue,
            defaultValue: defaultValue,
            maxDisplayLength: maxDisplayLength,
            ariaLabel,
        };
    };

    /**
     * Default activate function. Focuses the root element
     */
    InlineEditable.defaultActivate = function(editData) {
        editData.editElement.focus();
        editData.editElement.select();
    };

    /**
     * Default renderView function.
     */
    InlineEditable.defaultTextRenderView = function(editData) {
        var displayView = $(GH.tpl.inlineeditfield.renderBasicTextView({
            canEdit: true, // as only used when editing finishes
            fieldName : editData.fieldName,
            fieldValue : editData.newValue,
            displayValue : editData.newValue, //editData._.has(editData , 'displayValue')  ? editData.displayValue : editData.newValue
            defaultValue: editData.defaultValue,
            maxDisplayLength: editData.maxDisplayLength,
            extraClasses: editData.extraClasses
        }));
        return displayView;
    };

    /**
     * Default renderView function.
     */
    InlineEditable.pencilTextRenderView = function(editData) {
        var displayView = $(GH.tpl.inlineeditfield.renderPencilTextView({
            canEdit: true, // as only used when editing finishes
            fieldName : editData.fieldName,
            fieldValue : editData.newValue,
            displayValue : editData.newValue
        }));
        return displayView;
    };

    /**
     * Default renderView function.
     */
    InlineEditable.remoteTriggeredTextRenderView = function(editData) {
        var displayView = $(GH.tpl.inlineeditfield.renderRemoteTriggeredTextView({
            fieldName : editData.fieldName,
            fieldValue : editData.newValue,
            displayValue : editData.newValue
        }));
        return displayView;
    };

    /**
     * Renders the inline edit for a textfield, optionally using the provided render function
     * @renderFunction function that renders the markup, typically a soy template. The parameter passed to renderFunction is editData
     */
    InlineEditable.renderTextFieldInlineEdit = function(editData, renderFunction) {
        if (!renderFunction) {
            renderFunction = GH.tpl.inlineeditfield.renderBasicTextFieldEdit;
        }

        // render the element
        var editView = $(renderFunction(editData));

        // currently it's the input directly. Change to .find('input...') if necessary.
        editView.val(editData.fieldValue);

        // save on blur
        editView.blur(function(event) {
            InlineEditable.submitChanges(editData);
        });

        // handle escape and return
        editView.keydown(function(event) {
            if (event.keyCode === 27) { // escape key
                event.stopPropagation();
                InlineEditable.cancelEdit(editData);
            }
            else if (event.keyCode === 13) { // return
                event.preventDefault();
                InlineEditable.submitChanges(editData);
            }
        });

        return editView;
    };

    /**
     * Validates that the text field is not empty, and assigns the new value in case of success.
     */
    InlineEditable.defaultTextValidation = function(editData) {
        // new value
        var newValue = editData.editElement.val();
        editData.newValue = newValue;
        return true;
    };


    //
    // User picker
    //

    /**
     * Renders the inline edit for a textfield, optionally using the provided render function
     * @renderFunction function that renders the markup, typically a soy template. The parameter passed to renderFunction is editData
     */
    InlineEditable.renderUserPickerInlineEdit = function(editData, renderFunction) {

        var data = {
            value: editData.fieldValue
        };

        // generate edit dom
        var editDom = $(GH.tpl.inlineeditfield.renderUserPickerFieldEdit({
            name: "userpickerSelect",
            data: data
        }));

        return editDom;
    };

    /**
     * Default activate function. Focuses the root element
     */
    InlineEditable.userPickerActivate = function(editData) {

        var editDomField = editData.editElement.find('select');

        // checks whether new value was selected
        var finishHandler = function(editData) {
            // ensure we only run once through this path
            if (editData.finishSubmitted) {
                return;
            }
            editData.finishSubmitted = true;

            // fetch the value to submit
            var selectObject = editData.selectObject;
            var descriptor = selectObject.getSelectedDescriptor();
            var fieldValue = null;
            if (descriptor) {
                fieldValue = descriptor.value();
            } else {
                // we got no value currently. simply cancel
                InlineEditable.cancelEdit(editData);
                return;
            }

            // only actually submit if the values differ
            if (fieldValue !== editData.fieldValue) {
                InlineEditable.submitChanges(editData);
            } else {
                InlineEditable.cancelEdit(editData);
            }
        };

        const GHSingleSelect = SingleSelect.extend({
            submitForm: function () {
                if (!this.suggestionsVisible) {
                    this.handleFreeInput();
                    finishHandler(editData);
                }
            },
            _handleEscape: function (e) {
                if (this.suggestionsVisible) {
                    e.stopPropagation();
                    if (e.type === "keyup") {
                        this.hideSuggestions();
                    }
                }
                if (e.type === "keyup") {
                    InlineEditable.cancelEdit(editData);
                }
            },
            _deactivate: function () {
                this.acceptFocusedSuggestion();
                this.hideSuggestions();
                finishHandler(editData);
            }
        });

        // setup the ajax select
        var selectObject = new GHSingleSelect({
            element: editDomField,
            showDropdownButton: false,
            removeOnUnSelect: true,
            submitInputVal: true,
            // overlabel: AJS.I18n.getText("user.picker.ajax.short.desc"),
            errorMessage: formatter.I18n.getText("admin.errors.invalid.user"),
            ajaxOptions: {
                url: Ajax.CONTEXT_PATH + "/rest/api/1.0/users/picker",
                query: true, // keep going back to the sever for each keystroke
                minQueryLength: 0,
                data: {
                    showAvatar: true
                },
                formatResponse: function (response) {
                    var ret = [];
                    $(response).each(function(i, suggestions) {
                        var groupDescriptor = new GroupDescriptor({
                            weight: i, // order or groups in suggestions dropdown
                            id: "user-suggestions",
                            replace: true,
                            label: suggestions.footer // Heading of group
                        });

                        $(suggestions.users).each(function(){
                            groupDescriptor.addItem(new ItemDescriptor({
                                value: this.name, // value of item added to select
                                label: this.displayName, // title of lozenge
                                html: this.html,
                                allowDuplicate: false,
                                icon: this.avatarUrl
                            }));
                        });
                        ret.push(groupDescriptor);
                    });

                    return ret;
                }
            }
        });
        editData.selectObject = selectObject;
        var editTextField = editData.editElement.find("input");
        editTextField.focus();
        editDomField.select();
    };

    /**
     * Validates the user picker input
     */
    InlineEditable.userPickerValidation = function(editData) {
        // new value
        var descriptor = editData.selectObject.getSelectedDescriptor();
        editData.newValue = descriptor.value();
        editData.newDisplayValue = descriptor.label();
        if (editData.newDisplayValue.length === 0) {
            editData.newDisplayValue = editData.newValue;
        }
        return true;
    };


    //
    // Date Picker
    //

    /**
     * Renders the inline edit for a textfield, optionally using the provided render function
     * @renderFunction function that renders the markup, typically a soy template. The parameter passed to renderFunction is editData
     */
    InlineEditable.renderDatePickerInlineEdit = function(editData, renderFunction) {
        // generate edit dom
        var editDom = $(GH.tpl.inlineeditfield.renderDatePickerFieldEdit({
            fieldName: editData.fieldName,
            fieldValue: editData.fieldValue
        }));

        return editDom;
    };

    /**
     * Default activate function. Focuses the root element. Configuration can be overridden by the given config properties.
     */
    InlineEditable.datePickerActivate = function(editData, config) {

        // setup the calendar
        var fieldName = editData.fieldName;

        config = _.extend({}, {
            cache: false,
            firstDay: 0,
            inputField: 'ghx-datepicker-' + fieldName,
            button: 'ghx-datepicker-' + fieldName + '_trigger_c',
            align: 'Br',
            singleClick: true,
            showsTime: true,
            timeFormat: 12,
            positionStyle: 'fixed',
            onClose: function (cal) {
                InlineEditable.submitChanges(editData);
                cal.hide();
            },
            ifFormat: GH.TimeFormat.dateTimePickerFormat // Note: won't be initialized until global configuration is loaded
        }, config);

        Calendar.setup(config);

        $('#ghx-datepicker-' + fieldName).keydown(function(event) {
            if (event.keyCode === 27) { // escape key
                event.stopPropagation();
                InlineEditable.cancelEdit(editData);
                Validation.clearContextualErrors();
            } else if (event.keyCode === 13) { // return
                event.preventDefault();
                InlineEditable.submitChanges(editData);
            }
        });

        $('#ghx-datepicker-' + fieldName).focus();
        $('#ghx-datepicker-' + fieldName).blur(function() {
            InlineEditable.submitChanges(editData);
        });
        JiraLogger.trace("gh.inlineEdit.datePicker.activate");
    };

    /**
     * Validates the date picker field text
     */
    InlineEditable.datePickerValidation = function(editData) {
        // new value
        var newValue = editData.editElement.find('input').val();
        editData.newValue = newValue;
        return true;
    };

    //
    // Inline editable fields support
    //

    InlineEditable.activeEdits = [];

    /**
     * Cancels all active edits
     */
    InlineEditable.cancelActiveEdits = function() {
        _.each(InlineEditable.activeEdits, function(editData) {
            InlineEditable.cancelEdit(editData);
        });
    };

    /**
     * Cancels all active edits
     * @returns {Boolean} true if there was at least one active inline edit, false otherwise
     */
    InlineEditable.submitActiveEdits = function() {
        if(_.isEmpty(InlineEditable.activeEdits)) {
            return false;
        } else {
            _.each(InlineEditable.activeEdits, function(editData) {
                InlineEditable.submitChanges(editData);
            });
            return true;
        }
    };

    /**
     * Register a new inline edit with all the necessary options
     */
    InlineEditable.register = function(selector, options) {

        // defaults
        var effectiveOptions = {
            getViewElement: function(event) { return $(this); },
            getData:    InlineEditable.defaultFieldData,
            preEdit:    function(editData) {},
            renderEdit: InlineEditable.renderTextFieldInlineEdit,
            activate:   InlineEditable.defaultActivate,
            validate:   InlineEditable.defaultTextValidation,
            save:       function(editData) {
                // directly update the view
                InlineEditable.updateView(editData);
            },
            renderView: InlineEditable.defaultTextRenderView,
            postEdit:   function(editData) {},
            startEdit:  InlineEditable.startEdit/*,
                                                                     notLive: options.notLive || false*/
        };

        // override the defaults with the given options
        if (options) {
            $.extend(true, effectiveOptions, options);
        }

        var $document = $(document);
        $document.off('click', selector).on('click', selector, effectiveOptions, function(event) {
            var options = event.data;
            var viewElement =  options.getViewElement.call(this, event);
            effectiveOptions.startEdit(viewElement, options);
        });
    };

    /**
     * Called when a user clicks on the view element.
     */
    InlineEditable.startEdit = function(viewElement, options) {
        Logger.log("InlineEditable.startEdit", Logger.Contexts.ui);

        // fetch the edit data from the clicked view element
        var editData;
        if (options.getData) {
            editData = options.getData(viewElement, options);
        }
        if(!editData) {
            return;
        }
        editData.viewElement = viewElement;
        editData.options = options;

        // invoke pre edit
        if (options.preEdit) {
            var cancelEdit = options.preEdit(editData);
            if (cancelEdit === false) {
                return;
            }
        }

        // consider this an active edit
        InlineEditable.activeEdits.push(editData);

        // render the edit dom
        var editElement = options.renderEdit(editData);
        editData.editElement = editElement;

        // replace the view for the edit
        $(editData.viewElement).replaceWith(editData.editElement);

        // finally activate the edit
        if (options.activate) {
            options.activate(editData);
        }
    };

    /**
     * Cancels a currently edited inline edit.
     */
    InlineEditable.cancelEdit = function(editData) {
        editData.canceled = true;

        // replace the edit for the view
        editData.editElement.replaceWith(editData.viewElement);

        // remove all error messages, if any
        Validation.clearContextualErrors(editData.viewElement);

        // call the post edit callback
        if (editData.options.postEdit) {
            editData.options.postEdit(editData);
        }

        // consider this an active edit
        InlineEditable.removeFromActiveEdits(editData);
    };

    /**
     * Triggers submission of edit changes.
     * Validation occurs before submission
     */
    InlineEditable.submitChanges = function(editData) {
        // first validate the changes
        var success = editData.options.validate(editData);
        if (! success) {
            editData.options.activate(editData);
            return;
        }

        // save the data
        editData.options.save(editData);
    };

    /**
     * Updates the view, thus replaces the edit dom by a newly generated view dom.
     */
    InlineEditable.updateView = function(editData) {
        editData.updated = true;

        // render the view and replace the edit
        var viewElement = editData.options.renderView(editData);
        editData.viewElement = viewElement;
        editData.editElement.replaceWith(viewElement);

        // remove all error messages, if any
        Validation.clearContextualErrors(editData.viewElement);

        // we are done, call the post function
        if (editData.options.postEdit) {
            editData.options.postEdit(editData);
        }

        // consider this an active edit
        InlineEditable.removeFromActiveEdits(editData);
    };

    InlineEditable.removeFromActiveEdits = function(editData) {
        InlineEditable.activeEdits = _.without(InlineEditable.activeEdits, editData);
    };

    return InlineEditable;
});

AJS.namespace('GH.RapidBoard.Util.InlineEditable', null, require('jira-agile/rapid/configuration/inline-edit-field'));
