/**
 * Handles the "Filter" tab in the configuration page
 */
define("jira-agile/rapid/configuration/filter-config", [
    "jira-agile/rapid/util",
    "jira-agile/rapid/ajax",
    "jira-agile/rapid/analytics-tracker",
    "jira-agile/rapid/ui/notification",
    "jira/util/logger",
    "jira/util/formatter",
    "jquery"
], function (Util, Ajax, AnalyticsTracker, Notification, Logger, formatter, $) {
    "use strict";

    const FilterConfig = {};

    FilterConfig.analytics = new AnalyticsTracker('gh.config.subquery.edit');

    FilterConfig.init = function (model) {
        // set up the model
        GH.RapidBoard.Config.FilterModel.setModel(model);

        // show the filter info and bind the inline edit handlers
        FilterConfig.showFilter();

        // bind the Add Rank button
        $(document).delegate('.js-update-filter-order-by', 'click', function (e) {
            FilterConfig.confirmOrderByUpdate();
            e.preventDefault();
        });

        // initialise JQL autocomplete for sub query
        if (GH.RapidBoard.Config.FilterModel.canChangeFilter) {
            $(document).delegate('#js-sub-container-work .js-editable', 'click', FilterConfig.initSubQueryEdit);
        }

        // Show warning message when issues in unmapped status
        $(GH).bind(GH.RapidBoard.Config.UNMAPPED_STATUSES_WITH_ISSUES, function () {
            $("#unmapped-issues-warning").removeClass("hidden");
        });

        // Show warning message when issues in unmapped status
        $(GH).bind(GH.RapidBoard.Config.NO_UNMAPPED_STATUSES_WITH_ISSUES, function () {
            $("#unmapped-issues-warning").addClass("hidden");
        });
    };

    /**
     * Renders the filter details from the passed filter, and initialises the inline frother edit
     */
    FilterConfig.showFilter = function () {
        var canEdit = GH.RapidBoard.Config.FilterModel.canChangeFilter,
            savedFilter = GH.RapidBoard.Config.FilterModel.savedFilter,
            rapidViewId = GH.RapidBoard.Config.FilterModel.id,
            isSubqueriesConfigurable = !GH.RapidBoard.Config.FilterModel.isSprintSupportEnabled,
            subQueries = isSubqueriesConfigurable ? [FilterConfig.getSubQueryForDisplay()] : [],
            isOldDoneIssuesCutoffConfigurable = GH.RapidBoard.Config.FilterModel.isOldDoneIssuesCutoffConfigurable,
            oldDoneIssuesCutoffOptions = GH.RapidBoard.Config.FilterModel.oldDoneIssuesCutoffOptions,
            oldDoneIssuesCutoff = GH.RapidBoard.Config.FilterModel.oldDoneIssuesCutoff;

        $('#ghx-config-filter').html(GH.tpl.rapid.view.renderFilterDetails({
            savedFilter: savedFilter,
            subqueries: subQueries,
            isSubqueriesConfigurable: isSubqueriesConfigurable,
            canEdit: canEdit,
            rapidViewId: rapidViewId,
            isOldDoneIssuesCutoffConfigurable: isOldDoneIssuesCutoffConfigurable,
            oldDoneIssuesCutoffOptions: oldDoneIssuesCutoffOptions,
            oldDoneIssuesCutoff: oldDoneIssuesCutoff
        }));

        // Switch to columns tab
        $("#js-view-columns").bind('simpleClick', Util.wrapPreventDefault(function () {
            GH.RapidBoard.Config.setTab('columns');
            GH.State.replace({
                "rapidView": GH.RapidBoard.Config.editedRapidViewId,
                "tab": 'columns'
            });

        }));

        if (canEdit) {
            FilterConfig.initClickHandler(savedFilter);
        }

        // Init tooltip regarding view edit project page permission if any
        var projectsInBoardViewPermissionTooltips = $('#ghx-show-projects-in-board .aui-iconfont-info').tooltip({gravity: 'w'});
        if (projectsInBoardViewPermissionTooltips.length > 0) {
            projectsInBoardViewPermissionTooltips.tooltip({gravity: 'w'});
        }

        $("#ghx-limit-days").change(FilterConfig.saveOldDoneIssuesCutoff);
    };

    FilterConfig.saveOldDoneIssuesCutoff = function () {
        let value = $("#ghx-limit-days").val()
        GH.Ajax.put({
            url: '/rapidviewconfig/oldDoneIssuesCutoff',
            data: {
                rapidViewId: GH.RapidBoard.Config.FilterModel.id,
                oldDoneIssuesCutoff: value
            }
        }).done(function (data) {
            GH.RapidBoard.Config.FilterModel.setOldDoneIssuesCutoff(value);
        });

        AJS.trigger('analytics', {
            name: 'gh.workmode.fast.kanban.configure.cutoff',
            data: {
                value: value,
                boardId: GH.RapidBoard.Config.FilterModel.id
            }
        });
    };

    /**
     * Set up a single select frother to replace the filter name display on click. Once the frother is done, the tab
     * section is redrawn, which removes the frother as well. This means we've got to tidy up after ourselves so not to
     * leave dangling event listeners around
     */
    FilterConfig.initClickHandler = function (currentFilter) {
        // single-use event listener
        $('#ghx-filter-change').one('click', function (event) {

            // dispose of the view mode
            $(this).hide();

            var filterPicker = new GH.SavedFilterPicker({
                selected: FilterConfig.selectHandler,
                selector: '#ghx-filter-change-select',
                blur: FilterConfig.showFilter,
                stayActivated: true
            });
            filterPicker.show(currentFilter);

        });
    };

    /**
     * When the user selects a value, save it to the database and redraw the tab
     */
    FilterConfig.selectHandler = function (event, selected) {
        if (!selected.properties) {
            return;
        }
        // save data
        Ajax.put({
            url: '/rapidviewconfig/filter',
            data: {
                'id': GH.RapidBoard.Config.FilterModel.id,
                'savedFilterId': selected.properties.value
            }
        })
            .done(function (data) {
                // set the new data
                GH.RapidBoard.Config.FilterModel.setSavedFilter(data.success);

                // rebuild the filter display with the new data
                FilterConfig.showFilter();
                GH.RapidBoard.Config.analytics.fieldChange.trigger('savedfilter.update');
            });
    };

    FilterConfig.confirmOrderByUpdate = function () {
        // create a dialog
        var dialog = GH.Dialog.create({
            width: 400,
            height: 200,
            id: "confirm-filter-dialog",
            closeOnOutsideClick: true
        });

        dialog.addHeader(formatter.I18n.getText('gh.rapid.config.filter.rank.confirm.title'));
        dialog.addPanel("Panel 1", "<p>" + formatter.I18n.getText('gh.rapid.config.filter.rank.confirm.desc') + "</p>", "panel-body");

        dialog.addButton(formatter.I18n.getText('gh.rapid.operations.confirm'), function (dialog) {
            dialog.disableControls();
            FilterConfig.updateOrderBy(dialog);
        }, 'aui-button');
        GH.Dialog.addCancelButton(dialog);
        $('#' + dialog.id).addClass('ghx-dialog');
        dialog.show();
    };

    FilterConfig.updateOrderBy = function (dialog) {
        var rapidViewId = GH.RapidBoard.Config.FilterModel.id;
        if (rapidViewId) {
            // ask to update
            Ajax.put({
                url: '/rapidviewconfig/orderbyrank',
                data: {
                    'id': rapidViewId
                }
            })
                .done(function (data) {
                    // set the new data
                    GH.RapidBoard.Config.FilterModel.setSavedFilter(data.getSuccessObject());

                    // render the details
                    FilterConfig.showFilter();

                    // show message and hide dialog
                    Notification.showSuccess(formatter.I18n.getText('gh.rapid.config.filter.rank.updated'));

                    dialog.hide();
                })
                .fail(dialog.enableControls);
        }
    };

    FilterConfig.initSubQueryEdit = function (event) {
        var editContainer = $('#js-sub-container-work'),
            editControls = GH.tpl.rapid.view.renderSubQueryControls({
                section: 'board.kanban.work',
                queryValue: GH.RapidBoard.Config.FilterModel.getSubQuery().query
            });

        // draw the edit controls
        editContainer.empty().append(editControls);

        // register Update click
        $('#js-sub-container-work .js-subquery-submit').click(FilterConfig.updateSubQuery);

        // register Cancel click
        $('#js-sub-container-work .js-subquery-cancel').click(FilterConfig.cancelSubQueryEdit);

        // register JQL Autocomplete
        GH.JQLAutoComplete.initialize('ghx-sub-jql', 'ghx-sub-jql-errormsg');

        // focus on input
        $('#ghx-sub-jql').focus();
    };

    FilterConfig.cancelSubQueryEdit = function (event) {
        // replace edit controls with view
        var viewContainer = $('#js-sub-container-work'),
            viewControls = GH.tpl.rapid.view.renderSubQueryView({
                section: 'board.kanban.work',
                queryValue: FilterConfig.getSubQueryForDisplay(),
                canEdit: GH.RapidBoard.Config.FilterModel.canChangeFilter
            });

        // draw the view controls
        viewContainer.empty().append(viewControls);
    };

    /**
     * Updates a subquery
     */
    FilterConfig.updateSubQuery = function (event) {
        // ensure we don't submit the form
        event.preventDefault();

        Logger.log("update subquery");

        // fetch all data for the call
        var rapidViewId = GH.RapidBoard.Config.FilterModel.id;
        var textarea = $(this).parent().children('textarea');
        var section = textarea.attr('data-section');
        var query = textarea.val();

        FilterConfig.analytics.trigger(section); // SAFE

        // save data
        Ajax.put({
            url: '/subqueries/' + rapidViewId,
            data: {
                //'section': section,
                'query': query
            },
            errorContextMap: {
                'query': FilterConfig.displaySubQueryError
            }
        })
            .done(function (data) {
                // update model with new subQuery
                GH.RapidBoard.Config.FilterModel.updateSubQuery(data.getSuccessObject());
                GH.RapidBoard.Config.analytics.fieldChange.trigger('subquery.update');

                // redraw tab
                FilterConfig.showFilter();
            });
    };

    FilterConfig.getSubQueryForDisplay = function () {
        var subQuery = GH.RapidBoard.Config.FilterModel.getSubQuery();
        return subQuery.query ? subQuery.query : false;
    };

    /**
     * Rendering error for sub query is non-standard: our error needs to be placed after all the edit controls.
     */
    FilterConfig.displaySubQueryError = function (ctxError, res) {
        var ctx = '#ghx-sub-jql';
        ctx = $(ctx);

        // remove previous errors
        ctx.siblings(".ghx-error").remove();

        // and attach the new ones.
        var renderedError = GH.tpl.rapid.notification.renderContextualErrors({
            errors: ctxError.errors
        });

        // add as child of context element's parent
        ctx.parent().append(renderedError);
    };

    return FilterConfig;
});

AJS.namespace("GH.RapidBoard.Config.Filter", null, require("jira-agile/rapid/configuration/filter-config"));