/**
 * Detail View configuration tab
 */
define("jira-agile/rapid/configuration/detail-view-config", [
    "jira-agile/rapid/ajax",
    "jira/util/formatter",
    "aui/restful-table",
    "underscore",
    "jquery"
], function (Ajax, formatter, RestfulTable, _, $) {

    const DetailViewConfig = {};

    DetailViewConfig.UNSUPPORTED_CATEGORY = "unsupported";

    /**
     * Holds the model for the configuration.
     */
    DetailViewConfig.model = undefined;

    DetailViewConfig.restfulTableHeadings = {
        details: formatter.I18n.getText("gh.rapid.config.detailview.fields.details"),
        dates: formatter.I18n.getText("gh.rapid.config.detailview.fields.dates"),
        people: formatter.I18n.getText("gh.rapid.config.detailview.fields.people"),
        links: formatter.I18n.getText("gh.rapid.config.detailview.fields.links"),
        unsupported: formatter.I18n.getText("gh.rapid.config.detailview.fields.unsupported")
    };

    /**
     * Object to hold references to the Restful Tables created later
     */
    DetailViewConfig.restfulTable = {};

    DetailViewConfig.TableColumns = [
        {
            id: 'fieldId',
            header: formatter.I18n.getText('gh.rapid.config.detailview.table.heading.field'),
            fieldName: 'fieldId',
            readView: RestfulTable.CustomReadView.extend({
                render: function (renderData) {
                    return GH.tpl.detailviewconfig.renderDetailViewFieldCell(this.model.toJSON());
                }
            })
        }
    ];

    DetailViewConfig.createView = function (category) {
        return RestfulTable.CustomCreateView.extend({
            render: function (renderData) {
                var self = this;
                var $select = $('<select class="select"></select>')
                    .attr('name', renderData.name);

                // this part won't be used, the actually displayed values are set in the promise callback bellow
                var values = DetailViewConfig.getAvailableDetailViewFields(false);
                values = _.filter(values, function (val) {
                    return val.category === category;
                });

                // render the initial options or hide the select if empty
                if (values.length) {
                    $select.html(this.renderOptions(values));
                } else {
                    $select.hide();
                }

                // load the new values
                DetailViewConfig.getAvailableDetailViewFields(true).done(function (result) {
                    values = result.fields;
                    values = _.filter(values, function (val) {
                        return val.category === category;
                    });
                    values.sort(function (field1, field2) {
                        return field1.name.localeCompare(field2.name);
                    });
                    if (values.length) {
                        DetailViewConfig.restfulTable[category].getCreateRow().enable();
                        $select.html(self.renderOptions(values)).show();
                    } else {
                        $select.hide();
                        DetailViewConfig.restfulTable[category].getCreateRow().disable();
                    }
                });
                return $select;
            },
            renderOptions: function (values) {
                return values.map(function (optionData) {
                    return `<option value="${optionData.fieldId}">${_.escape(optionData.name)}</option>`
                }).join();
            }
        });
    };

    /**
     * Initializes the Detail View config.
     */
    DetailViewConfig.init = function (model) {
        DetailViewConfig.model = model.detailViewFieldConfig;

        DetailViewConfig.renderConfig();
    };

    DetailViewConfig.renderConfig = function () {
        var model = DetailViewConfig.model;

        // render the form
        $('#ghx-config-detailView').html(GH.tpl.detailviewconfig.renderDetailViewConfig({}));

        DetailViewConfig.renderConfigurationTable(DetailViewConfig.UNSUPPORTED_CATEGORY);

        DetailViewConfig.renderConfigurationTable("details");
        DetailViewConfig.renderConfigurationTable("dates");
        DetailViewConfig.renderConfigurationTable("people");
        DetailViewConfig.renderConfigurationTable("links");
    };

    /**
     * Renders the edit form using the locally stored model
     */
    DetailViewConfig.renderConfigurationTable = function (category) {
        var model = DetailViewConfig.model;
        var canEdit = model.canEdit;
        var rapidViewId = model.rapidViewId;
        var form = $("#ghx-detail-view-field-table-form-" + category);
        var heading = DetailViewConfig.restfulTableHeadings[category];

        if (!canEdit) {
            form.html(GH.tpl.detailviewconfig.renderEmptyDetailViewTable({heading: heading}));
            return;
        }

        var fields = _.filter(DetailViewConfig.model.currentFields, function (field) {
            return field.category === category;
        });

        var availableFields = _.filter(DetailViewConfig.getAvailableDetailViewFields(false), function (field) {
            return field.category === category;
        });

        if (fields.length === 0 && availableFields.length === 0) {
            return;
        }

        // get rid of old table
        var tableId = 'ghx-detail-view-field-table-' + category;
        form.html(GH.tpl.detailviewconfig.renderDetailViewTable({
            heading: heading,
            tableId: tableId,
            description: category === DetailViewConfig.UNSUPPORTED_CATEGORY ? formatter.I18n.getText("gh.rapid.config.detailview.fields.unsupported.message") : ""
        }));

        var columns = DetailViewConfig.TableColumns;
        columns[0].createView = DetailViewConfig.createView(category);

        var params = {
            el: $("#" + tableId), // table to add entries to. Entries are appended to the tables <tbody> element
            resources: {
                all: function (next) {
                    return next(fields);
                },
                self: Ajax.buildRestUrl('/detailviewfield/' + rapidViewId + '/field/')
            },
            columns: columns,
            allowEdit: false,
            allowDelete: canEdit,
            allowReorder: canEdit,
            allowCreate: canEdit
        };
        if (category === DetailViewConfig.UNSUPPORTED_CATEGORY) {
            params.allowReorder = false;
            params.allowCreate = false;
        }

        // setup the restful table
        // todo: change this to "addRestfulTable" - then we will have to replace all occurances ... aargh
        var restfulTable = DetailViewConfig.restfulTable[category] = new RestfulTable(params);

        $(restfulTable).bind(RestfulTable.Events.ROW_REMOVED, () => {
            if (category === DetailViewConfig.UNSUPPORTED_CATEGORY) {
                if (restfulTable.isEmpty()) {
                    form.remove();
                }
            } else {
                restfulTable.getCreateRow().render({values: {}, errors: {}});
            }
        });
    };

    /**
     * Return the available fields that can be added (i.e. the list which will populate the dropdown)
     * @param defer use true to return a deferred result; false to return the list as it is in the model (will not be up to date)
     * @return {*}
     */
    DetailViewConfig.getAvailableDetailViewFields = function (defer) {
        var model = DetailViewConfig.model;
        if (defer) {
            return Ajax.get({
                url: '/detailviewfield/' + model.rapidViewId + '/available'
            }, "detailViewConfig")
                .done(function (result) {
                    model.availableFields = result.fields;
                });
        } else {
            return model.availableFields;
        }
    };

    return DetailViewConfig;
});

AJS.namespace("GH.DetailViewConfig", null, require("jira-agile/rapid/configuration/detail-view-config"));
