/* globals GH */

/**
 * Columns configuration tab.
 *
 * column/status mapping and column constraint functionality
 */
define('jira-agile/rapid/configuration/column-config', [
    "jira/util/formatter",
    "underscore",
    "jquery",
    'jira-agile/rapid/events',
    'jira-agile/rapid/configuration/column-config-analytics',
    'jira-agile/rapid/ui/kanplan/kan-plan-feature-service',
    "jira/util/logger",
    'jira/featureflags/feature-manager'
], function (formatter,
             _,
             $,
             Events,
             ColumnConfigAnalytics,
             KanPlanFeatureService,
             logger,
             FeatureFlagManager) {

    const ConfigColumn = new Events();
    const flexibleBoardsDisabled = !FeatureFlagManager.isFeatureEnabled("com.atlassian.jira.agile.darkfeature.flexible.boards");

    ConfigColumn.BACKLOG_COLUMN_STATUS_UPDATE = 'backlog-column-status-change';

    /**
     * Model backing the rapid list configuration
     */
    ConfigColumn.model = undefined;
    ConfigColumn.kanPlanConfig = undefined;

    /**
     * Initializes the rapid list configuration tab
     */
    ConfigColumn.init = function (data) {
        ConfigColumn.data = data;
        // store the provided model
        ConfigColumn.model = data.rapidListConfig;
        ConfigColumn.kanPlanConfig = {
            isKanPlanLabEnabled: data.globalConfig.kanPlanLabEnabled,
            isKanbanBoard: data.isSprintSupportEnabled === false,
            isKanPlanEnabled: data.isKanPlanEnabled
        };

        // render the column mapping
        ConfigColumn.renderColumnMapping();
        $(".ghx-status-not-present-warning").tooltip();

        // bind to the tab link, we need to align the column heights. This can only be done once the dom is visible
        if (flexibleBoardsDisabled) {
            $('#ghx-columns-config-tab-link').bind('tabSelect', ConfigColumn.alignColumnHeights);
        }

        ConfigColumn.EditField.registerInlineEdits();

        $(GH).bind(GH.Dialogs.AddColumn.EVENT_COLUMN_ADDED, ConfigColumn.updateNewColumn);
        $(GH).bind(GH.Dialogs.CreateStatus.EVENT_STATUS_CREATED, ConfigColumn.updateNewStatus);
        $(GH).bind(GH.Dialogs.RemoveStatus.EVENT_STATUS_REMOVED, ConfigColumn.updateRemovedStatus);
        $(GH).bind(GH.Dialogs.ConvertWorkflow.EVENT_WORKFLOW_CONVERTED, ConfigColumn.reloadColumnsModel);
    };


    ConfigColumn.tabLoaded = _.once(function () {
        ColumnConfigAnalytics.triggerTabOpenedAnalytics(ConfigColumn.model,
            ConfigColumn.model.canEdit);
    });


    ConfigColumn.renderColumnMapping = function () {
        var model = ConfigColumn.model;

        /* This is set here instead of init because some of the column related endpoints doesn't return this value,
         * so after performing operation like add column that resets the model with server response,
         * the column model wouldn't have it
         */
        model.showEpicAsPanel = ConfigColumn.data.showEpicAsPanel;

        // split unmapped statuses into those that have issues and those that don't
        model.unmappedStatusesWithIssues = _.filter(model.unmappedStatuses, function (status) {
            return status.issueCount > 0;
        });

        if (model.unmappedStatusesWithIssues.length > 0) {
            $(GH).trigger(GH.RapidBoard.Config.UNMAPPED_STATUSES_WITH_ISSUES, model.unmappedStatusesWithIssues);
        } else {
            $(GH).trigger(GH.RapidBoard.Config.NO_UNMAPPED_STATUSES_WITH_ISSUES);
        }

        model.unmappedStatusesWithoutIssues = _.filter(model.unmappedStatuses, function (status) {
            return status.issueCount === 0;
        });

        var mappedColumns = model.mappedColumns.filter(function (column) {
            return column.mappedStatuses.length > 0 && !column.isKanPlanColumn;
        });

        if (mappedColumns.length > 0) {
            mappedColumns[0].isFirstColumnWithStatuses = true;
            mappedColumns[mappedColumns.length - 1].isLastColumnWithStatuses = true;
        }

        // escape some things
        var simplifiedWorkflowProjectName;
        var adminWorkflowUrl = '';
        var simplifiedWorkflowLearnMoreUrl = GH.HelpPaths.getHelpPath('simplified.workflow').url;
        if (!_.isUndefined(model.workflow.simplifiedWorkflowProjectName)) {
            simplifiedWorkflowProjectName = AJS.escapeHTML(String(model.workflow.simplifiedWorkflowProjectName));
            adminWorkflowUrl = GH.Ajax.CONTEXT_PATH + '/plugins/servlet/project-config/' + model.workflow.simplifiedWorkflowProjectKey + '/workflows';
        }
        var contactAdminUrl = GH.Ajax.CONTEXT_PATH + '/secure/ContactAdministrators.jspa';

        // render the column config
        var $tabColumns = $('#columns');

        var scrollableContainer = document.getElementById('ghx-mapping-columns-wrapper');
        var scrollPosition;
        if (scrollableContainer) {
            scrollPosition = document.getElementById('ghx-mapping-columns-wrapper').scrollLeft;
        }

        $tabColumns.html(GH.tpl.columnconfig.renderColumnMappingView({
            model: model,
            lastUpdated: new Date().getTime(),
            simplifiedWorkflowProjectName: simplifiedWorkflowProjectName,
            adminWorkflowUrl: adminWorkflowUrl,
            contactAdminUrl: contactAdminUrl,
            kanbanBacklogUrl: GH.HelpPaths.getHelpPath('board.kanplanbacklog').url,
            simplifiedWorkflowLearnMoreUrl: simplifiedWorkflowLearnMoreUrl,
            shouldRenderKanbanBacklogColumn: ConfigColumn.shouldRenderKanbanBacklogColumn(),
            showEpicSwitcher: KanPlanFeatureService.isEpicsAndVersionsEnabled()
            && ConfigColumn.kanPlanConfig.isKanbanBoard
            && ConfigColumn.kanPlanConfig.isKanPlanEnabled,
            flexibleBoardsDisabled: flexibleBoardsDisabled
        }));

        if (scrollPosition) {
            document.getElementById('ghx-mapping-columns-wrapper').scrollLeft = scrollPosition;
        }

        $tabColumns.find('.aui-iconfont-help').tooltip({
            html: true,
            gravity: 's'
        });

        // align the heights.
        if (flexibleBoardsDisabled) {
            ConfigColumn.alignColumnHeights();
        }

        // initialise everyone who cares about the DOM
        if (ConfigColumn.model.canEdit) {
            ConfigColumn.DnD.init();
            ConfigColumn.attachListeners(model);
            ConfigColumn.EditField.init();

            $('#ghx-mapping .ghx-action[data-deleteindex] .js-column-delete').click(ConfigColumn.deleteColumn);
        }
    };

    /**
     * Sets the same column height for all columns. Otherwise shorter columns won't be droppable on the whole height
     */
    ConfigColumn.alignColumnHeights = function () {
        // remove the set heights
        var viewTable = $('#ghx-mapping');
        var columns = viewTable.find('.ghx-config-status:not(.ghx-kanplan-backlog)');
        columns.css({'height': ''});

        // make the columns the same height = height of parent + twice a status height
        var viewTableHeight = viewTable.height();
        var viewStatusHeight = viewTable.find('.ghx-status:first').outerHeight();
        columns.css({'height': viewTableHeight + (viewStatusHeight * 2)});
    };

    ConfigColumn.attachListeners = function (model) {
        if (model.workflow.isSimplifiedWorkflowFeatureEnabled
            && !model.workflow.usingSimplifiedWorkflow
            && model.workflow.canSimplifyWorkflow
            && model.workflow.userCanSimplifyWorkflow) {
            $('#js-convert-workflow').click(ConfigColumn.convertWorkflowDialog);
        }

        if (model.workflow.usingSimplifiedWorkflow
            && model.workflow.isProjectAdminOfSimplifiedWorkflow) {
            $('#ghx-config-addcolumn').click(ConfigColumn.addColumnDialog);
            $('#ghx-config-addstatus').click(ConfigColumn.addStatusDialog);
            $('#ghx-mapping .ghx-unmapped .js-remove-status').click(ConfigColumn.removeStatusDialog);

            $('#ghx-mapping .js-resolution-update').click(ConfigColumn.resolutionUpdate);
            $('#ghx-mapping .js-resolution-update').each(function () {
                var statusElement = $(this);
                var columnElement = statusElement.parents("ul.ghx-config-status");
                var tipsyDirection = columnElement.hasClass("ghx-last") ? 'e' : 'n';

                GH.Tooltip.tipsify({
                    selector: statusElement,
                    gravity: tipsyDirection
                });
            });
        } else {
            $('#ghx-config-addcolumn').click(ConfigColumn.addColumn);
        }

        $('#ghx-mapping').sortable({
            handle: '.ghx-header',
            items: '.ghx-column-wrapper:not(.ghx-fixed-column)',
            forceHelperSize: true,
            forcePlaceholderSize: true,
            opacity: '0.9',
            placeholder: 'ghx-sortable-placeholder',
            tolerance: 'pointer',
            start: function () {
                // cancel ongoing edits
                GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();
            },
            update: function () {
                ConfigColumn.save();
            }
        });

        $('#ghx-statisticsfield-select').change(ConfigColumn.save);
        $('.js-config-load-days-toggle').change(ConfigColumn.saveShowDaysInColumn);

        var $epicCheckbox = $('#ghx-checkbox-epic-appearance');

        $epicCheckbox.on('change', function () {
            var state = $epicCheckbox.prop('checked');
            ConfigColumn.updateShowEpicAsPanel(state);
        });
    };


    /**
     *
     * @param {Boolean} state
     * @description whether to display epics as card or in epics panel
     */
    ConfigColumn.updateShowEpicAsPanel = function (state) {
        var checkbox = $('#ghx-checkbox-epic-appearance');

        var config = ConfigColumn.model;
        var model = {
            rapidViewId: config.rapidViewId,
            showEpicAsPanel: state
        };

        return GH.Ajax.put({
            url: '/rapidviewconfig/showEpicAsPanel',
            data: model
        }).done(function () {
            ConfigColumn.data.showEpicAsPanel = model.showEpicAsPanel;
            ConfigColumn.model.showEpicAsPanel = model.showEpicAsPanel;
            logger.trace('gh.kanplan.columnconfig.showEpicAsPanel.toggle.success');
        }).fail(function () {
            // reset the checkbox to its previous value
            checkbox.prop("checked", !state);
            logger.trace('gh.kanplan.columnconfig.showEpicAsPanel.toggle.fail');
        });
    };

    ConfigColumn.convertWorkflowDialog = function (event) {
        event.preventDefault();
        GH.Dialogs.ConvertWorkflow.showDialog(ConfigColumn.model.rapidViewId);
    };

    ConfigColumn.addStatusDialog = function () {
        GH.Dialogs.CreateStatus.showDialog(ConfigColumn.model.rapidViewId);
    };

    ConfigColumn.addColumnDialog = function () {
        GH.Dialogs.AddColumn.showDialog(ConfigColumn.model.rapidViewId, ConfigColumn.model.workflow.usingSimplifiedWorkflow);
    };

    ConfigColumn.removeStatusDialog = function (event) {
        var el = $(this);
        var statusId = el.closest('.ghx-status').attr('data-statusid');
        GH.Dialogs.RemoveStatus.showDialog(ConfigColumn.model.rapidViewId, statusId);
    };

    ConfigColumn.addColumn = function () {
        // add a new column to the model
        var newColumn = {
            name: formatter.I18n.getText('gh.rapid.view.columnmapping.newname'),
            mappedStatuses: [],
            min: formatter.I18n.getText('gh.rapid.config.capacity.setmin'),
            max: formatter.I18n.getText('gh.rapid.config.capacity.setmax'),
            isKanPlanColumn: false
        };
        var mappedCols = ConfigColumn.model.mappedColumns;
        var newPosition = mappedCols.length - 1;

        // insert it second to the last (so one before "done")
        mappedCols.splice(newPosition, 0, newColumn);

        // now render, since we're using the DOM to save the current status,
        // this also readjusts the column widths
        ConfigColumn.renderColumnMapping();

        // save the new view
        ConfigColumn.save(false, function () {
            // focus on the name field for the new column
            $('#ghx-mapping .ghx-mapped:eq(' + newPosition + ') .ghx-header-name').click();
        });
        ColumnConfigAnalytics.triggerColumnAdded(mappedCols.length);
    };

    ConfigColumn.deleteColumn = function (event) {
        var index = $(this).closest('.ghx-action').data('deleteindex');

        // remap any statuses from deleted column to the unmapped column
        var statusesInColumn = ConfigColumn.model.mappedColumns[index].mappedStatuses;
        var existingUnmappedStatuses = ConfigColumn.model.unmappedStatuses;
        ConfigColumn.model.unmappedStatuses = existingUnmappedStatuses.concat(statusesInColumn);

        // remove the selected column from the model
        ConfigColumn.model.mappedColumns.splice(index, 1);

        // now render, since we're using the DOM to save the current status
        // this also readjusts the column widths
        ConfigColumn.renderColumnMapping();

        // save the new view
        ConfigColumn.save(false);
    };

    /**
     * Take the current DOM and turn it back into the model. This saves us from having to synch model and DOM
     * during drag and drop, we just take what the user sees and persist it.
     *
     * @param reload : should the view be reloaded after saving, defaults to 'true'
     */
    ConfigColumn.save = function (reload, onReload) {
        if (typeof reload === 'undefined') {
            reload = true;
        }

        var model = {};

        // statistics field
        model.currentStatisticsField = {
            id: $('#ghx-statisticsfield-select').val()
        };

        // columns
        var columnsData = [];
        model.rapidViewId = ConfigColumn.model.rapidViewId;
        model.mappedColumns = columnsData;
        $('#ghx-mapping .ghx-mapped').each(function (index, column) {
            column = $(column);
            var columnData = {};
            var mappedStatuses = [];

            // status mapping for this column
            column.find('.ghx-status').each(function (index, status) {
                mappedStatuses[index] = {
                    id: $(status).attr('data-statusid')
                };
            });
            columnData.mappedStatuses = mappedStatuses;

            // name, min and max constraint
            // ok, nearly what the user sees. Minus html encoding. WYSIWYM.
            var columnId = column.attr('data-column-id');
            if (columnId) {
                columnData.id = parseInt(columnId, 10);
            }
            var name = column.find('.ghx-header-name[data-fieldname="name"]').attr('data-fieldvalue');
            if (name) {
                columnData.name = name;
            }
            columnData.min = column.find('span[data-fieldname="min"]').attr('data-fieldvalue');
            columnData.max = column.find('span[data-fieldname="max"]').attr('data-fieldvalue');
            columnData.isKanPlanColumn = column.hasClass("ghx-kanplan-backlog");

            // add column
            columnsData.push(columnData);
        });

        GH.Ajax.put({
            url: '/rapidviewconfig/columns',
            data: model
        })
            .done(function (result) {
                // assign the new model
                ConfigColumn.model = result.success;

                if (reload) {
                    ConfigColumn.renderColumnMapping();
                }
                if (onReload) {
                    onReload();
                }
            });
    };

    ConfigColumn.saveShowDaysInColumn = function () {

        var toggle = $(".js-config-load-days-toggle");

        // create the model
        var model = {
            rapidViewId: ConfigColumn.model.rapidViewId,
            showDaysInColumn: toggle.is(":checked")
        };

        GH.Ajax.put({
            url: '/rapidviewconfig/showDaysInColumn',
            data: model
        })
            .done(function (result) {
                ConfigColumn.model.showDaysInColumn = model.showDaysInColumn;
                // doesn't need re-render
            })
            .fail(function () {
                // reset the checkbox to its previous value
                toggle.prop("checked", !model.showDaysInColumn);
            });
    };

    ConfigColumn.updateNewColumn = function (event, newModel) {
        ColumnConfigAnalytics.triggerColumnAdded(newModel.mappedColumns.length);
        ConfigColumn.model = newModel;

        // re render the columns
        ConfigColumn.renderColumnMapping();
    };

    ConfigColumn.updateNewStatus = function (event, newStatus) {
        // add new status into model (if it is not already in it)
        if (!ConfigColumn.containsStatus(newStatus)) {
            ConfigColumn.model.unmappedStatuses.push(newStatus);
        }

        // re render the columns
        ConfigColumn.renderColumnMapping();

        // highlight and scroll to the new status
        GH.RapidBoard.Animation.fadeElement('.ghx-status[data-statusid="' + newStatus.id + '"]');

        var statusOffset = $('.ghx-status[data-statusid="' + newStatus.id + '"]').offset().top;
        $(window).scrollTop(statusOffset);
    };

    ConfigColumn.updateRemovedStatus = function (event, oldStatusId) {
        // add new status into model
        var unmappedStatuses = ConfigColumn.model.unmappedStatuses;
        ConfigColumn.model.unmappedStatuses = _.reject(unmappedStatuses, function (status) {
            return status.id === oldStatusId;
        });

        // re render the columns
        ConfigColumn.renderColumnMapping();
    };

    ConfigColumn.reloadColumnsModel = function () {
        var promise = GH.Ajax.get({
            url: '/rapidviewconfig/editmodel.json',
            data: {
                rapidViewId: ConfigColumn.model.rapidViewId
            }
        }, "columnConfig");

        promise.done(ConfigColumn.processColumnMappingData);
        return promise;
    };

    ConfigColumn.processColumnMappingData = function (data) {
        ConfigColumn.model = data.rapidListConfig;

        // re render the columns
        ConfigColumn.renderColumnMapping();
    };

    /**
     * Is this status already in our model?
     * @param status
     * @return {Boolean}
     */
    ConfigColumn.containsStatus = function (status) {
        // is it unmapped?
        var found = _.any(ConfigColumn.model.unmappedStatuses, function (s) {
            return s.id === status.id;
        });
        if (found) {
            return true;
        }

        // is it mapped?
        _.any(ConfigColumn.model.mappedColumns, function (column) {
            found = _.any(column.mappedStatuses, function (s) {
                return s.id === status.id;
            });

            if (found) {
                return true;
            }
        });

        return found;
    };


// ====== drag drop ======

    ConfigColumn.DnD = {};

    ConfigColumn.DnD.init = function () {
        ConfigColumn.DnD.registerStatusDraggables();
    };

    ConfigColumn.DnD.registerStatusDraggables = function () {
        var $status = $(".ghx-status");
        var statusWidth = $status.outerWidth();
        var statusHeight = $status.outerHeight();
        $('.ghx-config-status').sortable({
            items: '.ghx-status',
            connectWith: '.ghx-config-status',
            placeholder: 'ghx-sortable-status-placeholder',
            start: function (event, ui) {
                // this is to fix sortable problem with padding attribute on item
                ui.item.width(statusWidth);
                ui.item.height(statusHeight);
                // cancel ongoing edits
                GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();
            },
            update: function (event, ui) {
                var item = ui.item;
                if (!item.parent().is("li")) {
                    item.wrap('<li></li>');
                }
            },
            stop: function (event, ui) {
                // this is to fix tipsy problem when dragging status
                $('.tipsy').hide();

                var $toColumn = ui.item.closest('.ghx-config-status');
                var $srcColumn = $(event.target);
                ConfigColumn.save(true, function () {
                    if ($toColumn.hasClass('ghx-kanplan-backlog') || $srcColumn.hasClass('ghx-kanplan-backlog')) {
                        ConfigColumn.trigger(ConfigColumn.BACKLOG_COLUMN_STATUS_UPDATE, $srcColumn, $toColumn);
                    }
                    ColumnConfigAnalytics.triggerStatusDnDAnalytics(ConfigColumn.model, $srcColumn, $toColumn);
                });
            }
        });
    };


//====== inline edit ======

    ConfigColumn.EditField = {};

    ConfigColumn.EditField.MAX_ISSUE_COUNT_LIMIT = 1000;

    ConfigColumn.EditField.init = function () {
    };

    /**
     * Data gathering function for the constraints fields.
     */
    ConfigColumn.EditField.constraintsData = function (viewElement, options) {
        var data = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement, options);
        data.columnIndex = viewElement.attr('data-columnindex');
        return data;
    };

    /**
     * Renders the edit for the constraint fields
     */
    ConfigColumn.EditField.renderConstraintInlineEdit = function (editData) {
        var elem = GH.RapidBoard.Util.InlineEditable.renderTextFieldInlineEdit(editData, GH.tpl.columnconfig.renderCapacityFieldEdit);
        return elem;
    };

    /** Called prior to starting the inline edit on the column configuration page. */
    ConfigColumn.EditField.preInlineEditCallback = function (editData) {
        // blur existing textfields - somehow focusing the new one doesn't trigger the blur on the old one
        $('#ghx-mapping .ghx-header input[type="text"]').blur();
    };

    /**
     * Update the column name
     */
    ConfigColumn.EditField.columnNameValidation = function (editData) {
        // new value
        var newValue = editData.editElement.val();

        // perform validation
        if (!GH.Validation.notBlank(editData.editElement, formatter.I18n.getText('gh.rapid.view.error.name.required'))) {
            return false;
        }

        editData.newValue = newValue;
        return true;
    };

    /**
     * Save the changes without rerendering the page
     */
    ConfigColumn.EditField.constraintValidation = function (editData) {
        // new value
        var newValue = editData.editElement.val();

        // first trim the value
        newValue = $.trim(newValue);

        var editView = editData.editElement.parent().find(":not(.ghx-error):last");
        var fieldName = editData.fieldName;
        var columnIndex = editData.columnIndex;

        // if the new value is empty, we can skip all validation
        if (newValue !== '') {

            // transform the value into a number
            /* TODO re-add this when we enable this feature GHS-3094
            newValue = parseFloat(newValue);
            */
            newValue = parseInt(newValue, 10);

            // make sure it actually is a number
            if (isNaN(newValue)) {
                GH.Validation.showContextualErrors(editView, formatter.I18n.getText('gh.rapid.view.error.capacity.notnumber'));
                return false;
            }

            // TODO remove this when we enable this feature GHS-3094
            // round the number down
            newValue = Math.floor(newValue);

            // make sure the number is not less than 0
            if (newValue < 0) {
                GH.Validation.showContextualErrors(editView, formatter.I18n.getText('gh.rapid.view.error.capacity.toosmall', 0));
                return false;
            }

            // make sure the number is smaller/equal to our limit
            if (newValue >= ConfigColumn.EditField.MAX_ISSUE_COUNT_LIMIT) {
                GH.Validation.showContextualErrors(editView, formatter.I18n.getText(
                    'gh.rapid.view.error.capacity.toobig',
                    ConfigColumn.EditField.MAX_ISSUE_COUNT_LIMIT
                ));
                return false;
            }

            // now ensure that min value is smaller equal to max value
            var oppositeField = fieldName === 'min' ? 'max' : 'min';
            var oppositeValue = $(
                '#ghx-mapping .ghx-mapped .js-edit-' + oppositeField + '[data-columnindex=' + columnIndex + ']'
            ).attr('data-fieldvalue');

            // only check for min max numeric value constriants if
            // - newValue is not empty
            // - oppositeValue is not empty
            if (oppositeValue !== '') {
                /* TODO re-add this when we enable this feature GHS-3094
                oppositeValue = parseFloat(oppositeValue);
                */
                oppositeValue = parseInt(oppositeValue, 10);
                if (fieldName === 'min') {
                    // ensure we actually compare numbers
                    if (newValue > oppositeValue) {
                        GH.Validation.showContextualErrors(editView, formatter.I18n.getText('gh.rapid.view.error.mintoobig'));
                        return false;
                    }
                } else {
                    if (newValue < oppositeValue) {
                        GH.Validation.showContextualErrors(editView, formatter.I18n.getText('gh.rapid.view.error.maxtoosmall'));
                        return false;
                    }
                }
            }
        }

        editData.newValue = newValue;
        return true;
    };

    /**
     * Save the column configuration upon successfuly inline edits
     */
    ConfigColumn.EditField.saveEditChanges = function (editData) {
        // switch to display mode for fast user feedback
        // also, and more importantly, the column model is saved as a whole, so the view needs to be available before actually saving
        GH.RapidBoard.Util.InlineEditable.updateView(editData);

        // save, but don't reload, since that would overwrite consecutive edit actions,
        // like when clicking the next field directly
        ConfigColumn.save(false);
    };

    /**
     * Renders the column name view
     */
    ConfigColumn.EditField.renderColumnNameView = function (editData) {
        var displayView = $(GH.tpl.columnconfig.renderColumnNameField({
            canEdit: true, // as only used when editing finishes
            fieldName: editData.fieldName,
            fieldValue: editData.newValue,
            displayValue: editData.newValue
        }));
        return displayView;
    };

    /**
     * Renders the min/max constraints
     */
    ConfigColumn.EditField.renderColumnConstraintView = function (editData) {
        var newValue = editData.newValue;
        var hasValue = _.isNumber(newValue);
        var displayView;

        displayView = $(GH.tpl.columnconfig.renderCapacityField({
            canEdit: true, // as only used when editing finishes
            fieldName: editData.fieldName,
            fieldValue: newValue,
            index: editData.columnIndex,
            hasValue: hasValue
        }));
        return displayView;
    };

    /**
     * Register the inline edit handlers for the column inline edit fields.
     */
    ConfigColumn.EditField.registerInlineEdits = function () {
        // shorten the namespaces...
        var inlineEdit = GH.RapidBoard.Util.InlineEditable;
        var editField = ConfigColumn.EditField;

        // title edit
        inlineEdit.register('#ghx-mapping .ghx-mapped .js-edit-name-trigger', {
            preEdit: editField.preInlineEditCallback,
            validate: editField.columnNameValidation,
            save: editField.saveEditChanges,
            renderView: editField.renderColumnNameView
        });

        // min/max edit
        inlineEdit.register('#ghx-mapping .ghx-mapped .js-edit-min,.js-edit-max', {
            getData: editField.constraintsData,
            preEdit: editField.preInlineEditCallback,
            renderEdit: editField.renderConstraintInlineEdit,
            validate: editField.constraintValidation,
            save: editField.saveEditChanges,
            renderView: editField.renderColumnConstraintView
        });
    };

    ConfigColumn.resolutionUpdate = function () {
        var el = $(this);

        var done = el.is(':checked');
        var statusId = el.closest('.ghx-status').attr('data-statusid');

        // Get the column in which the status exists
        var columnId = parseInt(el.closest('.ghx-config-status').attr('data-column-id'), 10);
        var column = _.findWhere(ConfigColumn.model.mappedColumns, {
            id: columnId
        });
        // Get the status from the model that corresponds to the status ID
        var status = _.findWhere(column.mappedStatuses, {
            id: statusId
        });

        // Update the model
        status.isResolutionDone = done;

        // Notify the server
        GH.Ajax.post({
            url: '/workflow/status/resolution',
            data: {
                rapidViewId: ConfigColumn.model.rapidViewId,
                statusId: statusId,
                done: done
            }
        });
    };

    ConfigColumn.shouldRenderKanbanBacklogColumn = function () {
        return ConfigColumn.kanPlanConfig.isKanbanBoard
            && ConfigColumn.kanPlanConfig.isKanPlanLabEnabled;
    };

    return ConfigColumn;
});

AJS.namespace('GH.RapidBoard.Config.Columns', null, require('jira-agile/rapid/configuration/column-config'));
