define('jira-agile/rapid/versions-model', ["underscore"], function(_) {
    "use strict";
    /**
     * The model for versions on the board.
     *
     * The supplied projects collection is used to build a view of versions containing the project details. Since a version
     * only contains the project id, we need this collection to get the plain project object. This collection will be an invariant
     * of this model, only the versions part can be modified thanks to {@link #updateRawData}.
     *
     * That fits the lifecycle of our board where the projects remains the same during the whole life whereas the versions data can change.
     *
     * @param {Array.<{id: Number, key: String, name: String}>} projects
     * @constructor
     */
    const VersionsModel = function (projects) {
        this.projects = projects;
        /**
         * @type {Object.<Number, {id: Number, name: String, description: String, sequence: Number, released: Boolean, startDate: Number, startDateFormatted: String, releaseDate: Number, releaseDateFormatted: String,
     * versionStats: {notDoneEstimate: Number, doneEstimate: Number, totalEstimate: Number, percentageCompleted: Number, estimated: Number, percentageEstimated: Number, notEstimated: Number, percentageUnestimated: Number, notDone: Number, done: Number, totalIssueCount: Number},
     * project: {id: Number, key: String, name: String}}>}
         */
        this.fixVersions = undefined;
        this.canCreateVersion = false;
        this.linkToDevStatusVersionAvailable = false;
    };

    /**
     * Sets the projects required by the model to work
     */
    VersionsModel.prototype.setProjects = function(projects) {
        this.projects = projects;
    };

    /**
     * Sets the raw data coming from server and processes it to build the fix version map.
     *
     * @param {{versionsPerProject: Object.<Number, Array.<{id: Number, name: String, description: String, sequence: Number, released: Boolean, startDate: Number, startDateFormatted: String, releaseDate: Number, releaseDateFormatted: String,
 * versionStats: {notDoneEstimate: Number, doneEstimate: Number, totalEstimate: Number, percentageCompleted: Number, estimated: Number, percentageEstimated: Number, notEstimated: Number, percentageUnestimated: Number, notDone: Number, done: Number, totalIssueCount: Number}}>>, canCreateVersion: Boolean}} data
     */
    VersionsModel.prototype.updateRawData = function(data) {
        this.canCreateVersion = data.canCreateVersion;
        this.linkToDevStatusVersionAvailable = data.isLinkToDevStatusVersionAvailable;
        this.fixVersions = VersionsModel.buildFixVersions(data, this.projects);
        this.versionsPerProject = VersionsModel.buildVersionsPerProject(data, this.projects, this.fixVersions);
    };

    /**
     * Is this model properly initialized?
     */
    VersionsModel.prototype.isValid = function() {
        return !!this.fixVersions;
    };

    /**
     * Get the versions stored in this model
     */
    VersionsModel.prototype.getVersions = function() {
        return this.fixVersions;
    };

    /**
     * Get the projects stored in this model
     * @return {Array.<{id: number, name: string}>}
     */
    VersionsModel.prototype.getProjects = function() {
        return this.projects;
    };

    /**
     * Get the projects stored in this model
     */
    VersionsModel.prototype.canUserCreateVersion = function() {
        return this.canCreateVersion;
    };

    /**
     * Get the projects stored in this model
     */
    VersionsModel.prototype.isLinkToDevStatusVersionAvailable = function() {
        return this.linkToDevStatusVersionAvailable;
    };
    /**
     * Returns a descriptor of versions that contains all the versions sorted by ascending sequence number and a formatted
     * versions list to display.
     *
     * @param {Array.<number>} versions versions ids
     * @return {{versions: Array, formattedVersions: string}}
     */
    VersionsModel.prototype.getVersionsForIds = function(versions) {
        // filter out invalid versions ids and build an array of versions objects form the ids
        var versionsObjects =_.chain(versions)
            .filter(function(versionId) {
                return _.has(this.fixVersions, versionId);
            }, this)
            .map(function(versionId) {
                return this.fixVersions[versionId];
            }, this)
            .value();
        return this.getFixVersionFromArray(versionsObjects);
    };

    /**
     * Returns the version for the given version id.
     *
     * @param {number} versionId
     * @return {{id: Number, name: String, description: String, sequence: Number, released: Boolean, startDate: Number, startDateFormatted: String, releaseDate: Number, releaseDateFormatted: String, versionStats: {notDoneEstimate: Number, doneEstimate: Number, totalEstimate: Number, percentageCompleted: Number, estimated: Number, percentageEstimated: Number, notEstimated: Number, percentageUnestimated: Number, notDone: Number, done: Number, totalIssueCount: Number, estimateDone: String, estimateTotal: String, isIssueCountStatistic: Boolean, estimateStatisticName: String}, projectId: Number}} the version or null if there is no version associated with this id
     */
    VersionsModel.prototype.getVersion = function(versionId) {
        return _.has(this.fixVersions, versionId) ? this.fixVersions[versionId] : null;
    };

    VersionsModel.prototype.getVersionsIdsForProject = function(projectId) {
        if(_.has(this.versionsPerProject, projectId)) {
            return this.versionsPerProject[projectId];
        }
        return null;
    };

    VersionsModel.prototype.getVersionsForProject = function(projectId) {
        let versions = this.getVersionsIdsForProject(projectId);
        if(versions) {
            return this.getVersionsForIds(versions)
        }
        return null;
    };

    VersionsModel.prototype.getFixVersionFromArray = function(versions) {
        const allVersions = _.sortBy(versions, function(version) {return version.sequence; });

        const versionData = {};
        versionData.versions = allVersions;
        versionData.formattedVersions = _.map(allVersions, this.getToolTip).join(", ");
        return versionData;
    };

    VersionsModel.prototype.getToolTip = function(version) {
        if (version.released) {
            return version.name + " (released)";
        }
        return version.name;
    };

    VersionsModel.prototype.sort = function(v1, v2) {
        if (v1.project && v2.project) {
            if (v1.project.key === v2.project.key) {
                return v1.sequence - v2.sequence;
            } else if (v1.project.key < v2.project.key) {
                return -1;
            } else {
                return 1;
            }
        } else {
            return v1.sequence - v2.sequence;
        }
    };

    VersionsModel.prototype.getUnreleasedVersionList = function() {
        return _.chain(_.toArray(this.fixVersions))
            .where({ released: false })
            .value()
            .sort(this.sort);

    };

    VersionsModel.prototype.getAllVersions = function() {
        return _.chain(_.toArray(this.fixVersions))
            .value()
            .sort(this.sort);
    };

    VersionsModel.prototype.containsId = function(versionId) {
        return _.has(this.fixVersions, versionId);
    };

    VersionsModel.prototype.isEmpty = function() {
        return _.isEmpty(this.fixVersions);
    };

    VersionsModel.prototype.hasMultiProjects = function() {
        return this.projects.length > 1;
    };

    VersionsModel.prototype.updateVersion = function(version) {
        if (_.isUndefined(this.fixVersions[version.id])) {
            return false;
        }
        this.fixVersions[version.id] = version;
        return true;
    };

    /**
     * A "static" methods that turns a versions data raw structure to one usable by the model.
     *
     * @param rawData
     * @param {Array.<{id: number}>} projects
     * @return {Object.<number, {id: number, name: string, project: {id: number}}>}
     */
    VersionsModel.buildFixVersions = function (rawData, projects) {
        const fixVersions = {};
        _.each(rawData.versionsPerProject, function(versions, projectId) {

            projectId = parseInt(projectId, 10);

            _.each(versions, function(version) {
                fixVersions[version.id] = _.clone(version);
                fixVersions[version.id].project = _.findWhere(projects, {id: projectId});
            });
        });

        return fixVersions;
    };

    VersionsModel.buildVersionsPerProject = function (rawData) {
        const versionsPerProject = {};
        _.each(rawData.versionsPerProject, function(rawVersions, projectId) {
            versionsPerProject[parseInt(projectId, 10)] = _.pluck(rawVersions, "id")
        });

        return versionsPerProject;
    };

    return VersionsModel;
});

AJS.namespace('GH.VersionsModel', null, require('jira-agile/rapid/versions-model'));