/**
 * Handles loading data for plan mode and work mode
 */
define('jira-agile/rapid/rapid-board-data-loader', ['jira-agile/rapid/const/page-mode', 'underscore'], function (PageMode, _) {
    'use strict';

    var _this = this;

    var RapidBoardDataLoader = {};

    /**
     * @returns {Promise}
     */
    RapidBoardDataLoader.loadData = function (mode, requestFn) {
        var deferred = AJS.$.Deferred();
        var claimDataFallback = function claimDataFallback() {
            requestFn().done(function (data) {
                var issues = RapidBoardDataLoader._jsonToIssues(mode, data);
                RapidBoardDataLoader.resolveEntities(issues, data.entityData);
                deferred.resolveWith(_this, [data]);
            }).fail(deferred.reject.bind(deferred));
        };
        var dataKey = 'boardFetch' + mode;
        if (window[dataKey] && _.isFunction(window[dataKey].then) && _.isFunction(window[dataKey].catch)) {
            window[dataKey].then(function (result) {
                return result.json.then(function (json) {
                    if (result.ok) {
                        var issues = RapidBoardDataLoader._jsonToIssues(mode, json);
                        RapidBoardDataLoader.resolveEntities(issues, json.entityData);
                        deferred.resolve(json);
                    } else {
                        // fallback to legacy REST handling, maybe there is something wrong with script pipe...
                        claimDataFallback();
                    }
                });
            });
            delete window[dataKey];
        } else {
            claimDataFallback();
        }
        return deferred.promise();
    };

    RapidBoardDataLoader._jsonToIssues = function (mode, json) {
        var issues = mode === PageMode.PLAN ? json.issues : mode === PageMode.WORK ? json.issuesData && json.issuesData.issues || [] : [];
        return issues;
    };

    // Resolve entities like status, priority, type, and epic using their key/id to apply values from entityData
    RapidBoardDataLoader.resolveEntities = function (issues, entityData) {
        if (entityData) {
            issues.forEach(function (issue) {
                return RapidBoardDataLoader.resolveEntity(issue, entityData);
            });
        }
    };

    RapidBoardDataLoader.resolveEntity = function (issue, entityData) {
        return (Object.assign || _.extend)(issue, entityData.statuses[issue.statusId], entityData.types[issue.typeId], entityData.priorities[issue.priorityId], issue.epicId && entityData.epics[issue.epicId]);
    };

    return RapidBoardDataLoader;
});