/**
 * Handles loading data for plan mode and work mode
 */
define('jira-agile/rapid/rapid-board-data-loader', [
    'jira-agile/rapid/const/page-mode',
    'underscore'
], function (PageMode, _) {
    'use strict';

    const RapidBoardDataLoader = {};

    /**
     * @returns {Promise}
     */
    RapidBoardDataLoader.loadData = (mode, requestFn) => {
        const deferred = AJS.$.Deferred();
        const claimDataFallback = () => {
            requestFn().done((data) => {
                const issues = RapidBoardDataLoader._jsonToIssues(mode, data);
                RapidBoardDataLoader.resolveEntities(issues, data.entityData);
                deferred.resolveWith(this, [data]);
            }).fail(deferred.reject.bind(deferred));
        };
        let dataKey = `boardFetch${mode}`;
        if (window[dataKey] && _.isFunction(window[dataKey].then)  && _.isFunction(window[dataKey].catch)) {
            window[dataKey].then(result => result.json.then(json => {
                if (result.ok) {
                    const issues = RapidBoardDataLoader._jsonToIssues(mode, json);
                    RapidBoardDataLoader.resolveEntities(issues, json.entityData);
                    deferred.resolve(json);
                } else {
                    // fallback to legacy REST handling, maybe there is something wrong with script pipe...
                    claimDataFallback();
                }
            }));
            delete window[dataKey];
        } else {
            claimDataFallback();
        }
        return deferred.promise();
    };

    RapidBoardDataLoader._jsonToIssues = (mode, json) => {
        const issues = mode === PageMode.PLAN ? json.issues : mode === PageMode.WORK ? (json.issuesData && json.issuesData.issues || []) : [];
        return issues;
    }

    // Resolve entities like status, priority, type, and epic using their key/id to apply values from entityData
    RapidBoardDataLoader.resolveEntities = (issues, entityData) => {
        if (entityData) {
            issues.forEach(issue => RapidBoardDataLoader.resolveEntity(issue, entityData));
        }
    };

    RapidBoardDataLoader.resolveEntity = (issue, entityData) => {
        return (Object.assign || _.extend)(issue,
            entityData.statuses[issue.statusId],
            entityData.types[issue.typeId],
            entityData.priorities[issue.priorityId],
            issue.epicId && entityData.epics[issue.epicId]);
    };

    return RapidBoardDataLoader;
});