define('jira-agile/rapid/ajax', [
    "jira/util/formatter",
    "jira/ajs/ajax/smart-ajax",
    "jira/util/data/meta",
    "underscore",
    "wrm/context-path",
    "jquery",
    "jira/jquery/deferred"
], function(Formatter, SmartAjax, Meta, _, wrmContextPath, $, Deferred) {
    /**
     * This library provides AJAX methods with defaults and automatic error handling.
     */
    var Ajax = {};

    Ajax.CONTEXT_PATH = wrmContextPath();
    Ajax.REST_URL_BASE = '/rest/greenhopper/1.0';
    Ajax.PUBLIC_REST_URL_BASE = '/rest/agile/1.0';

    /**
     * Holds xhr requests information, such as any ongoing xhr
     * and timestamp of last execution
     */
    Ajax.xhrRegistery = {};
    Ajax.stallTimeout = 300; // stall time duration for subsequent requests

    /**
     * Make a GET request. Setting a different HTTP method will be overridden.
     *
     * @param settings : object jQuery settings object (see jQuery.ajax), with GH extensions:
     * @param resourceId : string the id of the get resource, used to abort ongoing requests if a subsequent request is executed
     */
    Ajax.get = function(settings, resourceId) {

        // enrich the ajax settings object with GET specific values
        $.extend(settings, {
            type : 'GET',
            headers : {
                // signal whether this request is for a gadget by looking for GH.gadget which should only be included for gadget resources
                // see com.atlassian.greenhopper.web.AbstractResource#HTTP_HEADER_GREENHOPPER_GADGET
                'X-Atlassian-GreenHopper-Gadget': !_.isUndefined(GH.gadget)
            }
        });

        // execute new request
        var xhr = Ajax._executeGet(settings, resourceId);

        // transform the promise result to return directly the success object in case of success
        return xhr.pipe(function(result, textStatus, xhr) {
            // pipe the result. pass through the result.success instead of just the result
            return Deferred().resolveWith(this, [result.success, textStatus, xhr]);
        });
    };

    /**
     * Make a POST request. Setting a different HTTP method will be overridden. The datatype returned by the server is expected to be JSON.
     *
     * @param settings : object the jQuery settings object (see jQuery.ajax)
     *        GH extensions are:
     *      - errorContextMap : map between contextIds as returned from the server and a CSS selector defining where the field errors should be displayed
     *                          Alternatively, specify a function callback as value that receives the error and does custom stuff with it.
     *      - contextualErrorRenderer : optional. Alternative renderer for contextual errors. Takes the error as parameter and has to return the rendered view.
     *      - onGlobalError : optional. Global error callback
     */
    Ajax.post = function(settings) {
        settings = Ajax._getSettings(settings, true);
        // POST is vulnerable to XSRF, so we use JSON as transfer encoding - this isn't possible using plain HTML forms.
        $.extend(settings, {
            type : 'POST',
            contentType : 'application/json'
        });
        return Ajax._executeStore(settings);
    };

    /**
     * Make a PUT request. Setting a different HTTP method will be overridden. The datatype returned by the server is expected to be JSON.
     *
     * @param settings : object the jQuery settings object (see jQuery.ajax)
     *        GH extensions are:
     *      - errorContextMap : map between contextIds as returned from the server and a CSS selector defining where the field errors should be displayed.
     *                          Alternatively, specify a function callback as value that receives the error and does custom stuff with it.
     *      - contextualErrorRenderer : optional. Alternative renderer for contextual errors. Takes the error as parameter and has to return the rendered view.
     *      - onGlobalError : optional. Global error callback
     */
    Ajax.put = function(settings) {
        settings = Ajax._getSettings(settings, true);
        settings.type = 'PUT';
        return Ajax._executeStore(settings);
    };

    /**
     * Make a DELETE request. Setting a different HTTP method will be overridden. The datatype returned by the server is expected to be JSON.
     *
     * @param settings : object the jQuery settings object (see jQuery.ajax)
     *        GH extensions are::
     *      - errorContextMap : map between contextIds as returned from the server and a CSS selector defining where the field errors should be displayed.
     *                          Alternatively, specify a function callback as value that receives the error and does custom stuff with it.
     *      - contextualErrorRenderer : optional. Alternative renderer for contextual errors. Takes the error as parameter and has to return the rendered view.
     *      - onGlobalError : optional. Global error callback
     */
    Ajax.del = function(settings) {
        settings = Ajax._getSettings(settings, true);
        settings.type = 'DELETE';
        return Ajax._executeStore(settings);
    };

    /**
     * Set the defaults for the Ajax settings
     * @param settings
     * @return {*}
     */
    Ajax._getSettings = function(settings, stringify) {
        // bareUrl is already the full url, settings.url is only the greenhopper part
        if (!settings.bareUrl && settings.url) {
            settings.bareUrl = Ajax.buildRestUrl(settings.url);
        }
        // set the url to what the bare url is
        if (settings.bareUrl) {
            settings.url = settings.bareUrl;
        }
        if(!settings.dataType) {
            settings.dataType = 'json';
        }

        if (settings.data && stringify) {
            settings.data = JSON.stringify(settings.data);
        }
        settings.contentType = 'application/json';

        return settings;
    };

    /**
     * Get the xhr data for a given resource id. An empty object is returned if no data exists yet.
     */
    Ajax._getXhrData = function(resourceId) {
        var reqData = { timestamp: 0 /*, xhr: null*/ };
        if (resourceId) {
            if (Ajax.xhrRegistery[resourceId]) {
                reqData = Ajax.xhrRegistery[resourceId];
            } else {
                // store the new reqData in the registry
                reqData = Ajax.xhrRegistery[resourceId] = reqData;
            }
        }
        return reqData;
    };

    /** Executes the passed get request - or stalls in case this is a subsequent request of the same type. */
    Ajax._executeGet = function(settings, resourceId) {
        settings = Ajax._getSettings(settings);
        var reqData = Ajax._getXhrData(resourceId);

        // abort previous request if still ongoing
        if (reqData.xhr) {
            // stop ongoing request
            GH.log('aborting request for ' + resourceId, GH.Logger.Contexts.ajax);
            reqData.xhr.abort();
            delete reqData.xhr;
            delete reqData.xhrAbort;
            delete reqData.timeoutId;
        }

        // update the request timestamp
        var lastRequestStamp = reqData.timestamp;
        reqData.timestamp = new Date().getTime();

        // check whether we should execute now or stall
        if (reqData.timestamp - lastRequestStamp < Ajax.stallTimeout) {
            GH.log('stalling request for ' + resourceId + ', time since last request: ' + (reqData.timestamp - lastRequestStamp), GH.Logger.Contexts.ajax);
            var deferred = Deferred();
            var actualXhr;
            var callback = function() {
                actualXhr = Ajax._executeGet(settings, resourceId);
                actualXhr.done(deferred.resolve).fail(deferred.reject);
            };
            reqData.timeoutId = window.setTimeout(callback, Ajax.stallTimeout);

            // We want the promise to look like an xhr even though it isn't
            // create an abort method which will abort an existing xhr
            // or clear the timeout if it is still delayed
            reqData.xhr = deferred.promise();
            reqData.xhr.abort = function() {
                if (actualXhr) {
                    actualXhr.abort.apply(actualXhr, arguments);
                } else {
                    clearTimeout(reqData.timeoutId);
                }
            };
        } else {
            // execute request
            GH.log('executing request for ' + resourceId, GH.Logger.Contexts.ajax);
            reqData.xhr = Ajax.makeRequest(settings);
        }

        reqData.xhrAbort = reqData.xhr.abort;
        // abort the latest xhr
        reqData.xhr.abort = function() {
            var xhr = reqData.xhr;
            var xhrAbort = reqData.xhrAbort;
            if (xhr && xhrAbort) {
                return xhrAbort.apply(xhr, arguments);
            }
        };
        reqData.xhr.done(function() {
            // request successfully completed, remove xhr object from registry
            if (reqData.xhr) {
                // clear out request, no need to cancel anymore
                delete reqData.xhr;
            }
        }).fail(function() {
            // request successfully completed, remove xhr object from registry
            if (reqData.xhr) {
                // clear out request, no need to cancel anymore
                delete reqData.xhr;
            }
        });
        return reqData.xhr;
    };



    /**
     * Called to make an ajax request that stores data. Overrides datatype and handles a result of type ResultEntry (see Java class for details).
     */
    Ajax._executeStore = function(settings) {
        settings = settings || {};

        // GHS-4983: due to a bug in jQuery 1.5.something, POST/PUT requests which contain stringified JSON which happens to
        // contain "??" will get wrongly interpreted as a JSON-P request and will cause things to blow up. So let's hack
        // around this for now
        if (settings.data && typeof(settings.data) === 'string') {
            if (/\?\?/.test(settings.data)) {
                var qq = "\\u003F\\u003F";
                settings.data = settings.data.replace(/\?\?/g, qq);
            }
        }

        // override some settings
        settings.dataType = 'json';

        return Ajax.makeRequest(settings);
    };

    /**
     *
     * @param settings to be uses for creating the XHR
     * @return a promise based on the XHR
     */
    Ajax.makeRequest = function(settings) {
        // Create an XHR using JIRA's SmartAjax request factory, wrap it in a promise because AG.MockXhr (overrides ajax
        // in gadgets) is the devil
        var deferred = Deferred();
        settings = _.extend(settings, {
            success : function() {
                deferred.resolveWith(this, arguments);
            },
            error : function() {
                deferred.rejectWith(this, arguments);
            }
        });
        var jqXhr = deferred.promise(SmartAjax.makeRequest(settings));
        var jqXhrAbort = jqXhr.abort;
        jqXhr.abort = function(reason) {
            jqXhr.abortMarker = reason != "timeout";
            return jqXhrAbort.apply(jqXhr, arguments);
        };

        // pipe the XHR using the default interceptors. This allows us
        // to change the resolution/rejection status of the deferred.
        var pipedXhr = jqXhr.pipe(
            Ajax._generateDoneInterceptor(),
            Ajax._generateFailInterceptor(settings.errorsNoAutoEscape)
        );

        // generate a generic error handler unless they requestee wants error handling deferred
        if (!settings.deferErrorHandling) {
            pipedXhr.fail(Ajax._generateErrorHandler(
                settings.onGlobalError,
                settings.errorContextMap,
                settings.contextualErrorRenderer,
                settings.globalErrorDismissable
            ));
        }

        // Return a promise that the XHR will complete one way or another
        return pipedXhr;
    };

    /**
     * @return {Function} used to intercept the deferred
     */
    Ajax._generateDoneInterceptor = function() {
        return function(result, textStatus, jqXhr) {
            // extend the result with some convenience methods
            result = result || {};
            result = Ajax._createExtendedResult(result);
            if (!Ajax.matchesCurrentUser(jqXhr)) {
                return Ajax._handleCurrentUserChanged(jqXhr);
            }
            if (!result.isSuccess()) {
                return Deferred().rejectWith(this, [result, jqXhr, textStatus, null]);
            }
            return Deferred().resolveWith(this, [result, textStatus, jqXhr]);
        };
    };

    /**
     * @return {Function} used to intercept the deferred
     */
    Ajax._generateFailInterceptor = function(errorsNoAutoEscape) {
        return function(jqXhr, textStatus, errorThrown) {
            // handle case where we aborted the request ourselves
            if (jqXhr.abortMarker) {
                return Deferred(); // Don't resolve or reject this. We don't want to invoke any of the callbacks
            }
            if (!Ajax.matchesCurrentUser(jqXhr)) {
                return Ajax._handleCurrentUserChanged(jqXhr);
            }

            // the response text might contain JSON
            var responseJson = {},
                result = null;

            // try to parse the response json (which will fail in case it is not json)
            if (jqXhr.responseText) {
                try {
                    responseJson = JSON.parse(jqXhr.responseText);
                } catch(e) {
                    // silently ignore
                }
            }

            // convert error response into an extended Result
            if (responseJson.errors || responseJson.errorMessages) {
                // error response is a JIRA REST ErrorCollection
                result = Ajax._convertJiraErrors(responseJson, errorsNoAutoEscape);
            } else if (responseJson.message) {
                // error response was thrown by REST framework (e.g. failed to authenticate)
                result = Ajax._convertFrameworkErrors(responseJson);
            }

            if (!result) {
                // no known errors in JSON response
                result = Ajax._convertGenericErrors(jqXhr, errorThrown);
            }
            return Deferred().rejectWith(this, [result, jqXhr, textStatus, errorThrown]);
        };
    };

    /**
     * Pick up the field errors returned by the server, resolve the respective element in the page where they should be
     * displayed, and render there errors there.
     *
     * @param res : object the result, extended through GH.Ajax._extendResult
     * @param errorContextMap : map between the internal field name (as added to ErrorCollection) and the CSS selector for where
     *          it should be displayed (or a function that does the rendering).
     * @param renderer : function optional. Used to render the individual error messages.
     *
     * @returns true if there were field errors to be handled, false otherwise.
     */
    Ajax.handleContextualErrors = function(res, errorContextMap, renderer) {
        errorContextMap = errorContextMap || {};

        // check if we have anything useful to work on
        if (!res.hasContextualErrors()) {
            return false;
        }

        // render the contextual errors
        $(res.getContextualErrors()).each(function(index, ctxError) {

            // look up the context. This can be a jQuery selector, a DOM element or a function.
            var ctx = errorContextMap[ctxError.contextId];

            if (!ctx) {
                // the server specified a context, but the client didn't map it, so we render the error as global.
                // the global error handling comes after this, and since we don't want to mess with the AUI message
                // directly (there might be actual global errors as well), we'll just add it to the data object.
                $(ctxError.errors).each(function(index, err) {
                    res.addGlobalError(err);
                });

            } else if ($.isFunction(ctx)) {
                // the mapped context is a function, so pass in the error and let it do whatever it wants to do with it.
                ctx(ctxError, res);

            } else {
                // we have a context, and it's not a function, so it's either an element or a CSS selector. Regardless, we JQ it
                // and show the errors next to it.
                ctx = $(ctx);

                // remove previous errors
                ctx.siblings(".ghx-error").remove();

                // and attach the new ones.
                var renderedError;
                if (renderer) {
                    // caller specified an own renderer
                    renderedError = renderer(ctxError);

                } else {
                    // default renderer
                    renderedError = GH.tpl.rapid.notification.renderContextualErrors({
                        errors : ctxError.errors
                    });
                }

                // add as siblings to the context element
                ctx.after(renderedError);
            }
        });

        return true;
    };

    /**
     * Pick up the global errors returned by the server and display them through the notification system.
     * Global errors can be either produced by the server or added by the contextual error handler for unmapped
     * contextual errors.
     *
     * @param res : object the result object (see ResultEntry), extended by GH.Ajax._extendResult
     * @param onGlobalError : function optional. will be called after errors have been displayed, and have the wrapped result passed in.
     * @param globalErrorDismissable : boolean optional. sets the display of error message
     */
    Ajax.handleGlobalErrors = function(res, onGlobalError, globalErrorDismissable) {

        // see if we have something to work with.
        if (!res.hasGlobalErrors()) {
            return false;
        }

        // display errors with an AUI message
        GH.Notification.showErrors({ errors : res.getGlobalErrors()}, globalErrorDismissable);

        if(onGlobalError && (onGlobalError instanceof Function))
        {
            onGlobalError(res);
        }

        return true;
    };

    /**
     * Turn the given uri into a default private REST url.
     * Example: passing in '/rapidView' will result in '/rest/greenhopper/1.0/rapidView'.
     *
     * @param uri string: the path snippet to be turned into a rest URL. Leading slash is added if necessary.
     */
    Ajax.buildRestUrl = function(uri) {
        if (uri.indexOf('/') !== 0) {
            uri = '/' + uri;
        }

        return Ajax.buildBareRestUrl(Ajax.REST_URL_BASE + uri);
    };

    /**
     * Turn the given uri into a default public REST url.
     * Example: passing in '/sprint' will result in '/rest/agile/1.0/sprint'.
     *
     * @param uri string: the path snippet to be turned into a rest URL. Leading slash is added if necessary.
     */
    Ajax.buildPublicRestUrl = function(uri) {
        if (uri.indexOf('/') !== 0) {
            uri = '/' + uri;
        }

        return Ajax.buildBareRestUrl(Ajax.PUBLIC_REST_URL_BASE + uri);
    };

    /**
     * Just prepend the JIRA context path. For use when a REST call outside GH is made.
     *
     * @param uri string: the path snippet to be turned into a rest URL. Leading slash is added if necessary.
     */
    Ajax.buildBareRestUrl = function(uri) {
        if (uri.indexOf('/') !== 0) {
            uri = '/' + uri;
        }

        return Ajax.CONTEXT_PATH + uri;
    };

    /**
     * Create an Extended Result
     * @param result
     */
    Ajax._createExtendedResult = function(result) {
        if (!result) {
            // handle the no-content response case
            return Ajax._extendResult({success: true});
        } else {
            // something was returned - box it
            return Ajax._extendResult({success: result});
        }
    };

    /**
     * Extend the result coming from the server with some convenience methods. Basically, we want to
     * avoid checking all over the place if something is there or not and rather have some convenience
     * methods that are more readable.
     * @param result the Result of the response, in GH-expected format.
     */
    Ajax._extendResult = function(result) {
        return $.extend(result, {

            getGlobalErrors : function() {
                return result.error ? result.error.errors : null;
            },

            getContextualErrors : function() {
                return result.error ? result.error.contextualErrors : null;
            },

            hasGlobalErrors : function() {
                return result.error && result.error.errors && result.error.errors.length > 0;
            },

            hasContextualErrors : function() {
                return result.error && result.error.contextualErrors;
            },

            addGlobalError : function(error) {
                if (!result.error) {
                    result.error = {};
                }
                if (!result.error.errors) {
                    result.error.errors = [];
                }
                result.error.errors[result.error.errors.length] = error;
            },

            getSuccessObject : function() {
                return result.success;
            },

            isSuccess : function() {
                // in this magical, messed up world free of status codes success is defined as the absence of errors
                // rather than, I don't know, a 2XX status code
                return !result.error;
            }
        });
    };

    /**
     * Generates the appropriate error handler for non-200 replies.
     * @param onGlobalError
     * @param errorContextMap
     * @param contextualErrorRenderer
     * @param globalErrorDismissable
     */
    Ajax._generateErrorHandler = function(onGlobalError, errorContextMap, contextualErrorRenderer, globalErrorDismissable) {
        return function(result) {
            // handle the errors
            Ajax.handleContextualErrors(result, errorContextMap, contextualErrorRenderer);
            Ajax.handleGlobalErrors(result, onGlobalError, globalErrorDismissable);
        };
    };

    /**
     * Convert the JSON representation of a JIRA REST ErrorCollection into the GH format.
     * @param responseJson
     * @param errorsNoAutoEscape set this to true if the error messages will contain safe HTML that should not be escaped
     */
    Ajax._convertJiraErrors = function(responseJson, errorsNoAutoEscape) {
        var globalErrors = responseJson.errorMessages ? responseJson.errorMessages : [];
        globalErrors = _.map(globalErrors, function(str) {
            return {message: str, noAutoescape: errorsNoAutoEscape};
        });

        var contextualErrors = responseJson.errors ? responseJson.errors : {};
        contextualErrors = _.map(contextualErrors, function(message, key) {
            return {
                contextId: key,
                errors: [{message: message, noAutoescape: errorsNoAutoEscape}]
            };
        });

        var ghError = {
            error : {
                errors : globalErrors,
                contextualErrors : contextualErrors
            }
        };
        return Ajax._extendResult(ghError);
    };

    /**
     * Converts the JSON representation of REST framework errors into the GH format.
     * @param responseJson
     */
    Ajax._convertFrameworkErrors = function(responseJson) {
        var ghError = {
            error : {
                errors : [{message: responseJson.message}],
                contextualErrors : []
            }
        };
        return Ajax._extendResult(ghError);
    };

    /**
     * Converts generic errors returned by the XHR request into the GH format.
     * @param jqXHR
     * @param errorThrown
     */
    Ajax._convertGenericErrors = function(jqXHR, errorThrown) {
        var result = Ajax._extendResult({});
        if (jqXHR.responseText) {
            GH.log("Got an error response with text that we don't know how to parse: " + jqXHR.responseText);
        }

        // if we've got a timeout we want something better than "abort"
        if (jqXHR.status === 504 || jqXHR.status === 0 && $.contains(["abort", "timeout"],errorThrown)) {
            result.addGlobalError({status: 504, message: Formatter.I18n.getText("gh.ajax.error.timeout")});
        } else {
            // only add global error if errorThrown isn't empty, which seems to be the case when an ajax request is aborted
            // as as result of navigating to another page while the ajax request is in progress (Showing "Error" without
            // any message is of not much use anyways)
            if (errorThrown) {
                result.addGlobalError({status: jqXHR.status, message: errorThrown});
            } else {
                GH.log("Got an Ajax request error response with empty 'errorThrown' value. Silently logging");
            }
        }
        return result;
    };

    /**
     * Is there a username response header
     * @param xhr {XMLHttpRequest}
     * @return {Boolean} true if the A-AUSERNAME is not empty
     */
    Ajax._hasUsernameResponseHeader = function(xhr) {
        return !!xhr.getResponseHeader('X-AUSERNAME');
    };

    /**
     * return the current user from the request. 'anonymous' if the user is anonymous
     * @param xhr {XMLHttpRequest}
     * @return {String} the name of the user
     */
    Ajax._getUserForResponse = function(xhr) {
        return xhr.getResponseHeader('X-AUSERNAME') ? decodeURIComponent(xhr.getResponseHeader('X-AUSERNAME')) : 'anonymous';
    };

    /**
     * translate from empty string to "anonymous" to match what is returned from the server
     *
     * @return {String} "anonymous" if no logged in user, else username
     */
    Ajax.getUserFromAJS = function() {
        var remoteUser = Meta.get('remote-user');
        return remoteUser ? remoteUser : 'anonymous';
    };
    /**
     *
     * @param xhr the response
     * @return {Boolean} whether or not the user matches the current request
     */
    Ajax.matchesCurrentUser = function(xhr) {
        // XHR could be a mock XHR due to gadgets
        return !xhr.getResponseHeader || !xhr.getAllResponseHeaders ||
            // regard names as matching if we got no or an empty response header, which could be the result of an utf8 username
            !Ajax._hasUsernameResponseHeader(xhr) ||
            // or if the two names actually match
            Ajax.getUserFromAJS() === Ajax._getUserForResponse(xhr) ||
            // IE bug. Headers aren't available on the XHR for 204 requests so ignore this scenario.
            // See http://stackoverflow.com/questions/4268931/xmlhttprequest-response-has-no-headers-in-internet-explorer
            (xhr.status === 204 && !xhr.getAllResponseHeaders() && $.browser.msie);
    };

    /**
     * Checks whether the the user changes between requests in order to force a board reload or redirect to the login page
     * (handles cases where the user got logged out)
     *
     * @param xhr the response
     * @return A jQuery Deferred
     */
    Ajax._handleCurrentUserChanged = function(xhr) {
        var originalUsername = Ajax.getUserFromAJS();
        var newUsername = Ajax._getUserForResponse(xhr);

        // If there a situation where this isn't what the user wants and would
        // prefer to be prompted to either reload or log back in?
        if (originalUsername !== 'anonymous' && newUsername === 'anonymous') {
            // If the user has been logged out, redirect to the login page for them
            GH.log('User changed. original: ' + originalUsername + ' new: ' + newUsername + '. Redirecting to the login page');
            location.href = GH.Navigation.getLoginUrl();
        } else {
            // else the user has changed and therefore the pages state is inconsistent
            // Reload it to ensure we have a consistent state
            GH.log('User changed. original: ' + originalUsername + ' new: ' + newUsername + '. Reloading the page');
            location.reload();
        }

        return Deferred(); // Don't resolve or reject this. We don't want to invoke any of the callbacks
    };

    return Ajax;
});

AJS.namespace('GH.Ajax', null, require('jira-agile/rapid/ajax'));
