/**
 * Holds global GreenHopper keyboard shortcut implementation code
 *
 * @class gh.app.globalkeyboardshortcuts
 * @requires AJS
 */
define('jira-agile/gh-globalkeyboardshortcuts', ["jira-agile/rapid/ui/notification", "jira/ajs/ajax/smart-ajax", "jira/data/session-storage", "jira/issuenavigator/issue-navigator/shortcuts", "jira/issue", "jira/issuenavigator/issue-navigator", "jira/dialog/dialog", "jquery", "jira/util/browser"], function (Notification, SmartAjax, sessionStorage, IssueNavShortcutsApi, IssueApi, IssueNavigatorApi, Dialog, $, Browser) {

    var shortcuts = {};

    var wrmContextPath = require('wrm/context-path');

    /**
     * Get the selected issue id.
     *
     * This function checks for functions available on the issuenavigator and issue pages to fetch the id.
     *
     * GreenHopper provides overwrites for all these functions, so this code will work on GreenHopper pages too.
     */
    shortcuts.findSelectedIssueId = function () {
        // fetch the current issue id, either from the navigator or the issue page
        var issueId;
        if (IssueNavigatorApi && IssueNavigatorApi.getSelectedIssueId) {
            issueId = IssueNavigatorApi.getSelectedIssueId();
        }
        if (!issueId && IssueApi && IssueApi.getIssueId) {
            issueId = IssueApi.getIssueId();
        }
        return issueId;
    };

    /**
     * Implementation of the "go to agile" shortcut.
     */
    shortcuts.goToAgile = function () {
        var url = '/secure/RapidBoard.jspa';
        var contextPath = wrmContextPath();
        if (contextPath) {
            url = contextPath + url;
        }
        window.location.href = url;
    };

    /**
     * Follows a link given a link selector.
     */
    shortcuts.followLink = function (selector) {
        // Copied from whenitype.js
        var elem = $(selector);
        if (elem.length > 0 && (elem[0].nodeName.toLowerCase() === "a" || elem[0].nodeName.toLowerCase() === "link")) {
            elem.click();
            window.location.href = elem.attr("href");
        }
    };

    shortcuts.sendToTop = function (source, issueId) {

        if (window.GH && GH.RapidBoard && GH.Shortcut && GH.Shortcut.sendToTop) {
            GH.Shortcut.sendToTop(source);
            shortcuts.closeDialogAndPopMessage();
            return;
        }

        // support st/sb keyboard shortcuts only on rapid board
        if (source != 'Action') {
            return;
        }

        var selectedIssueId = issueId || shortcuts.findSelectedIssueId();
        var contextPath = wrmContextPath() || '';

        SmartAjax.makeRequest({
            type: 'post',
            contentType: 'application/json',
            url: contextPath + '/rest/greenhopper/1.0/rank/global/first',
            data: JSON.stringify({ issueId: selectedIssueId }),
            success: shortcuts.handleRankSuccess
        });

        $(document).trigger("gh.global.rankissues", {
            action: "rankToTop" + (source ? source : ""),
            count: 1
        });
    };

    shortcuts.sendToBottom = function (source, issueId) {
        if (window.GH && GH.RapidBoard && GH.Shortcut && GH.Shortcut.sendToBottom) {
            GH.Shortcut.sendToBottom(source);
            shortcuts.closeDialogAndPopMessage();
            return;
        }

        // support st/sb keyboard shortcuts only on rapid board
        if (source != 'Action') {
            return;
        }

        var selectedIssueId = issueId || shortcuts.findSelectedIssueId();
        var contextPath = wrmContextPath() || '';

        SmartAjax.makeRequest({
            type: 'post',
            contentType: 'application/json',
            url: contextPath + '/rest/greenhopper/1.0/rank/global/last',
            data: JSON.stringify({ issueId: selectedIssueId }),
            success: shortcuts.handleRankSuccess
        });

        $(document).trigger("gh.global.rankissues", {
            action: "rankToBottom" + (source ? source : ""),
            count: 1
        });
    };

    shortcuts.closeDialogAndPopMessage = function () {
        if (IssueNavigatorApi.isNavigator()) {
            Browser.reloadViaWindowLocation();
        } else {
            if (Dialog && Dialog.current) {
                Dialog.current.hide();
            }
            // check for greenhopper
            if (window.GH) {
                if (GH.RapidBoard && Notification && Notification.showIssueMessage) {
                    Notification.showIssueMessage();
                }

                // reload planning board if we happen to be on it
                if (window.Boards) {
                    window.Boards.refreshAll();
                }
            } else if (IssueNavigatorApi && IssueNavShortcutsApi && IssueNavShortcutsApi.flashIssueRow) {
                IssueNavShortcutsApi.flashIssueRow();
            }
        }
    };

    shortcuts.handleRankSuccess = function (result, textStatus, xhr) {
        // Display a message
        shortcuts.storeSuccessMessage(result.issueId, result.issueKey, "thanks_issue_updated");
        shortcuts.closeDialogAndPopMessage();
    };

    shortcuts.storeSuccessMessage = function (issueId, issueKey, msg) {
        var sessionStorge = sessionStorage;
        try {
            sessionStorge.setItem('selectedIssueId', issueId);
            sessionStorge.setItem('selectedIssueKey', issueKey);
            sessionStorge.setItem('selectedIssueMsg', msg);
        } catch (e) {}
    };

    return shortcuts;
});

require(['jquery', 'jira-agile/gh-globalkeyboardshortcuts'], function ($, Shortcuts) {
    $(document).delegate('.issueaction-greenhopper-rank-top-operation', 'click', function (event) {
        event.preventDefault();
        var issueId = $(this).attr("data-issueid");
        Shortcuts.sendToTop('Action', issueId);
    });
    $(document).delegate('.issueaction-greenhopper-rank-bottom-operation', 'click', function (event) {
        event.preventDefault();
        var issueId = $(this).attr("data-issueid");
        Shortcuts.sendToBottom('Action', issueId);
    });
});

AJS.namespace('gh.app.globalkeyboardshortcuts', null, require('jira-agile/gh-globalkeyboardshortcuts'));