define("jira/editor/undomanager", [
    "jira/editor/undo-redo-analytics",
    "jira/editor/undomanager-adapter/text-adapter",
    "jira/editor/undomanager-adapter/visual-adapter",
    "jira/editor/undomanager-adapter/noop-adapter",
    "jira/editor/undo-history-store",
    "jira/editor/constants",
    "jira/util/logger",
    "jira/editor/marionette",
    "jira/dialog/dialog",
    "jquery"
], function (
    UndoRedoAnalytics,
    TextAdapter,
    VisualAdapter,
    NoOpAdapter,
    undoHistoryStore,
    EditorConstants,
    logger,
    Marionette,
    Dialog,
    $
) {
    "use strict";

    var Modes = EditorConstants.Modes;

    var UndoManager = Marionette.Object.extend({
        initialize: function initialize(options) {
            this.context = this.determineContext(options.textarea);
            this.restoreHistory = false;

            try {
                this.textUndoManager = new TextAdapter({textarea: options.textarea});
            } catch (e) {
                this.textUndoManager = new NoOpAdapter();
            }
            this.visualUndoManager = new NoOpAdapter();
            this.activeUndoManager = new NoOpAdapter();

            // gather analytics
            this.analytics = new UndoRedoAnalytics(this);

            this.setActiveMode(options.mode);

            this.listenTo(this.textUndoManager, "action", this.notifyHistoryChanged);
        },
        undo: function undo() {
            this.activeUndoManager.undo();
            this.notifyHistoryChanged(EditorConstants.EventTypes.UNDO, EditorConstants.EventSources.BUTTON);
        },
        redo: function redo() {
            this.activeUndoManager.redo();
            this.notifyHistoryChanged(EditorConstants.EventTypes.REDO, EditorConstants.EventSources.BUTTON);
        },
        hasUndo: function hasUndo() {
            return this.activeUndoManager.hasUndo();
        },
        hasRedo: function hasRedo() {
            return this.activeUndoManager.hasRedo();
        },
        setActiveMode: function setActiveMode(mode) {
            this.activeMode = mode;
            this.activeUndoManager.clear();

            if (Modes.TEXT === mode) {
                this.activeUndoManager = this.textUndoManager;
            } else if (Modes.VISUAL === mode) {
                this.activeUndoManager = this.visualUndoManager;
            } else {
                this.activeUndoManager = new NoOpAdapter();
                logger.warn("UndoManager: incorrect mode", mode);
            }

            this.activeUndoManager.clear();

            this.analytics.clearCounters();
        },
        attachToVisualEditor: function attachToVisualEditor(richEditorEl) {
            this.visualUndoManager = new VisualAdapter({richEditorEl: richEditorEl});
            this.listenTo(this.visualUndoManager, "action", this.notifyHistoryChanged);
        },
        restoreUndoHistory: function restoreUndoHistory() {
            if (!this.restoreHistory) {
                return;
            }

            if (Modes.VISUAL === this.activeMode && EditorConstants.Context.UNKNOWN === this.context) {
                var undoEditorData = undoHistoryStore.get(this.visualUndoManager.getId());
                if (undoEditorData) {
                    this.visualUndoManager.clear();
                    undoEditorData.forEach(function (level) {
                        this.visualUndoManager.add(level);
                    }.bind(this));

                    this.restoreHistory = false;
                }
            }
            this.analytics.clearCounters();
        },
        shouldRestoreHistory: function shouldRestoreHistory(restoreHistory) {
            this.restoreHistory = !!restoreHistory;
        },

        /**
         * @private
         */
        onBeforeDestroy: function onBeforeDestroy() {
            this.saveUndoHistory();
        },
        /**
         * @private
         */
        saveUndoHistory: function saveUndoHistory() {
            if (Modes.VISUAL === this.activeMode && EditorConstants.Context.UNKNOWN === this.context) {
                undoHistoryStore.put(this.visualUndoManager.getId(), this.visualUndoManager.getData().concat([]));
            }
        },
        /**
         * @private
         */
        notifyHistoryChanged: function notifyHistoryChanged(eventName, source) {
            source = source || this.EventSources.UNKNOWN;
            this.trigger("editor:historychanged", eventName, source, this.activeMode);
        },
        /**
         * @private
         */
        determineContext: function determineContext(textarea) {
            if (Dialog.current) {
                return EditorConstants.Context.DIALOG;
            } else if ($(textarea).parents("div.jira-wikifield[field-id=comment]").length) {
                return EditorConstants.Context.COMMENT;
            } else {
                return EditorConstants.Context.UNKNOWN;
            }
        }
    });

    return UndoManager;
});