define("jira/editor/ui/undo-redo-buttons", [
    "jira/editor/constants",
    "jira/editor/marionette",
    "jira/util/logger",
    "jquery"
], function(
    EditorConstants,
    Marionette,
    logger,
    $
) {
    var UndoRedoButtons = Marionette.ItemView.extend({
        tagName: "div",
        template: function (serializedData) {
            return JIRA.Editor.Tags.Templates.undoRedoButtons();
        },

        ui: {
            undo: '.wiki-operation-undo',
            redo: '.wiki-operation-redo'
        },

        events: {
            'click @ui.undo': 'onUndo',
            'click @ui.redo': 'onRedo'
        },

        initialize: function initialize(options) {
            this.$container = this.determineButtonsContainer(options.textarea);
            if (null === this.$container) {
                logger.warn("Could not attach undo/redo buttons to the editor");
                return;
            }

            this.undoManager = options.undoManager;
        },
        /**
         * @public
         */
        enable: function enable($button) {
            if (!this.undoManager) {
                return;
            }
            $button = $button || $([this.ui.undo, this.ui.redo]);
            $button.removeAttr('disabled');
            $button.attr('aria-disabled', 'false');
        },
        /**
         * @public
         */
        disable: function disable($button) {
            if (!this.undoManager) {
                return;
            }
            $button = $button || $([this.ui.undo, this.ui.redo]);
            $button.attr('disabled', '');
            $button.attr('aria-disabled', 'true');
        },
        /**
         * @public
         */
        highlight: function highlight($button) {
            if (!this.undoManager) {
                return;
            }
            $button.addClass('active');
            setTimeout(function() {
                $button.removeClass('active');
            }, 100);
        },
        /**
         * @public
         */
        updateUI: function updateUI(eventName, source) {
            if (!this.undoManager) {
                return;
            }

            this.updateButton(this.ui.undo, this.undoManager.hasUndo());
            this.updateButton(this.ui.redo, this.undoManager.hasRedo());

            if (EditorConstants.EventSources.SHORTCUT === source) {
                if (EditorConstants.EventTypes.UNDO === eventName) {
                    this.highlight(this.ui.undo);
                } else if (EditorConstants.EventTypes.REDO === eventName) {
                    this.highlight(this.ui.redo);
                }
            }
        },

        updateButton: function updateButton($button, enable) {
            if (enable) {
                this.enable($button);
            } else {
                this.disable($button);
            }
        },

        onRender: function onRender() {
            if (!this.undoManager) {
                return;
            }

            this.$el.addClass("wiki-edit-undo-redo-buttons aui-buttons");
            if (this.shouldUseDivider()) {
                this.$el.addClass("wiki-edit-separate-section");
            }

            // IE11 issue: tooltips are sometimes not removed
            this.ui.undo.tooltip('destroy');
            this.ui.redo.tooltip('destroy');

            this.updateUI();

            this.$el.prependTo(this.$container);

            this.attachHandlers();
        },

        attachHandlers: function attachHandlers() {
            this.undoManager.on("editor:historychanged", function handleHistoryChange(eventName, source) {
                this.updateUI(eventName, source);
            }, this);
        },

        determineButtonsContainer: function determineButtonsContainer(textarea) {
            var container;

            // inline edit (agile, detail) - it overrides the default one so it should be checked first
            container = $(textarea).closest("div.inline-edit-fields").nextAll(".save-options").find(".wiki-button-bar-content");
            if (!container.length) {
                // create issue, add/edit comment (default one)
                container = $(textarea).closest("div.jira-wikifield").next(".save-options").find(".wiki-button-bar-content");
            }
            // SD fields
            if (!container.length) {
                container = $(textarea).siblings(".sd-rte-bar").find(".wiki-button-bar-content");
            }

            // SD attach files dialog
            if (!container.length) {
                container = $(textarea).parent(".wiki-edit-content").siblings(".sd-rte-bar").find(".wiki-button-bar-content");
            }

            return (container.length) ? container : null;
        },

        shouldUseDivider: function shouldUseDivider() {
            var toolbarEl = this.$container.children(":visible");
            return toolbarEl.length > 0 && !toolbarEl.eq(0).is("button, input");
        },

        onUndo: function onUndo() {
            this.undoManager.undo();
            this._focusTextarea();
        },

        onRedo: function onRedo() {
            this.undoManager.redo();
            this._focusTextarea();
        },

        _focusTextarea: function _focusTextarea() {
            this.options.textarea.focus();
        }
    });

    return UndoRedoButtons;
});