define("jira/editor/instance", [
    "jira/dialog/dialog",
    'jira/editor/analytics',
    'jira/editor/analytics-shortcuts',
    'jira/util/navigator',
    'jira/editor/tinymce',
    'jira/editor/context-manager',
    'jira/editor/context-detector',
    "jira/editor/schema",
    "jira/editor/selection",
    'wiki-edit/SpeechRecognition',
    'jira/util/logger',
    'jquery',
    'backbone',
    'underscore',
    "wrm/context-path"
], function (Dialog,
             Analytics,
             AnalyticsShortcuts,
             Navigator,
             tinymce,
             ContextManager,
             ContextDetector,
             EditorSchema,
             EditorSelection,
             SpeechRecognition,
             logger,
             $,
             Backbone,
             _,
             contextPath) {
    var EditorInstance = function (element, options) {
        this.element = element;
        this.options = _.extend({}, options);
    };

    _.extend(EditorInstance.prototype, Backbone.Events);

    var ESC_KEY = 27;

    EditorInstance.prototype.init = function (editor) {
        this.editor = editor;

        this.operationOverride = {};

        this.analyticsShortcuts = new AnalyticsShortcuts(this.editor);
        this.selection = new EditorSelection(this.editor);

        this.editor.on('NodeChange', function (e) {
            var nodeChangePerf = Analytics.startMeasure();

            if (!e) {
                return;
            }

            if (e.element.nodeName.toLowerCase() === 'img') {
                this.editor.fire('content');
            }

            var node = $(e.element);
            this.trigger("selection:update", {
                insidePreformatted: ContextDetector.detectPre(node),
                preformattedSelected: ContextDetector.detectPreWithinSelection(this.editor.selection.getContent()),
                insideTable: ContextDetector.detectTable(node),
                insideA: ContextDetector.detectA(node)
            });
            nodeChangePerf.measure('nodechange');
        }.bind(this));

        this.editor.on("change SetContent blur", this._onChange.bind(this));
        this.editor.on("keyup", _.debounce(this._onChange.bind(this), 1000));

        this.editor.on('init', function (e) {
            var editor = e.target;
            ['tt', 'del', 'sup', 'sub', 'cite'].forEach(function (tag) {
                editor.formatter.register(tag, {block: tag, remove: 'all'});
            });
            // extend RemoveFormat (Clear formatting) feature
            var removeFormat = editor.formatter.get('removeformat');
            if (removeFormat.length > 0 && typeof removeFormat[0] === 'object') {
                removeFormat[0].selector += ",tt";
            }
        });

        editor.on('keydown', function (e) {
            if (e.isDefaultPrevented()) {
                return;
            }

            // we need this to preserve correct table markup, which does not support multiple
            // paragraphs inside table cells
            const $selectionStart = $(editor.selection.getStart());

            if (shouldInsertLinebreak(e, $selectionStart)) {
                e.preventDefault();

                editor.execCommand("InsertLineBreak", false, e);
            }

            if (Dialog.current && e.keyCode === ESC_KEY) {
                Dialog.current.hide(true, {
                    reason: Dialog.HIDE_REASON.cancel,
                })
            }

            if (e.keyCode === tinymce.util.VK.ENTER && ($selectionStart.is('panel-title, panel-title *') || $selectionStart.parent().hasClass('panelHeader'))) {
                e.preventDefault();
            }
        });

        var getBrowserAnalyticsName = function () {
            if ((/(Edge)\/(\d+)\.(\d+)/).test(Navigator._getUserAgent())) {
                return 'edge';
            } else if (Navigator.isIE()) {
                return 'ie';
            } else if (Navigator.isChrome()) {
                return 'chrome';
            } else if (Navigator.isMozilla()) {
                return 'firefox';
            } else if (Navigator.isSafari()) {
                return 'safari';
            }
            return '';
        };

        Analytics.sendEvent("editor.instance.init");
        Analytics.sendEvent("bundled.editor.instance.init");
        var browserName = getBrowserAnalyticsName();
        if (browserName) {
            Analytics.sendEvent('editor.instance.init.' + browserName);
        }

        this.contextManager = new ContextManager(this);
        editor.contextManager = this.contextManager;
    };

    EditorInstance.prototype.getId = function () {
        return this.editor.id;
    };

    /**
     * Relay event handler to TinyMCE instance
     * @param name
     * @param fn
     * @param prepend
     */
    EditorInstance.prototype.relayEvent = function (name, fn, prepend) {
        this.editor.on(name, function (e) {
            fn(e);
        }, prepend);
    };

    //TODO define contract on generated HTML content
    EditorInstance.prototype.getAllowedOperations = function () {
        return [
            "paragraph", "h1", "h2", "h3", "h4", "h5", "h6", "monospace", "paragraph-quote", "block-quote", "delete", "superscript", "subscript",
            "cite", "monospace-inline",
            ":)", ":(", ":P", ":D", ";)", "(y)", "(n)", "(i)", "(/)", "(x)", "(!)", "(+)", "(-)", "(?)", "(on)", "(off)", "(*)", "(*r)", "(*g)", "(*b)", "(*y)",
            "bold", "italic", "underline", "color",
            "bullet-list", "numbered-list", "mention", "table", "code", "noformat", "panel",
            "hr", "speech", "link", "link-mail", "link-anchor", "link-attachment", "image", "image-attachment", "attachment",
            "editorInsertContent", "editorInsertContentInNewLine", "editorReplaceContent", "editorReplaceContentInNewLine"
        ];
    };

    EditorInstance.prototype._isOperationSupported = function (name) {
        var allowed = this.getAllowedOperations().filter(function (el) {
            if (el instanceof Object) {
                return el.name === name;
            } else {
                return el === name;
            }
        });
        return allowed.length > 0;
    };

    EditorInstance.prototype._assertOperationIsSupported = function (operationName) {
        if (!this._isOperationSupported(operationName)) {
            logger.error("Operation not supported:", operationName);
        }
    };

    EditorInstance.prototype._selectedTextSanitized = function () {
        return EditorSchema.sanitizeHtml(this.editor.selection.getContent(), this.editor, this.pasteInsidePreSchemaSpec);
    };

    EditorInstance.prototype.executeOperation = function (operationName, args) { // eslint-disable-line complexity
        this._assertOperationIsSupported(operationName);
        var tinymceMapped = this._mapOperationNameToTinymce(operationName);

        var funOverride = this.operationOverride[operationName];
        if (funOverride) {
            Analytics.sendEvent("editor.instance.operation." + operationName);
            funOverride(args);
            return true;
        }

        if ('editorReplaceContentInNewLine' === operationName) {
            this.editor.execCommand('mceReplaceContent', false, '<br />' + args.content);
        } else if ('editorReplaceContent' === operationName) {
            this.editor.execCommand('mceReplaceContent', false, args.content);
        } else if ('editorInsertContentInNewLine' === operationName) {
            this.editor.insertContent('<br />' + args.content);
        } else if ('editorInsertContent' === operationName) {
            this.editor.insertContent(args.content);
        } else if ('hr' === operationName) {
            this.editor.insertContent('<hr />');
        } else if ('color' === operationName) {
            this.editor.execCommand('ForeColor', true, args);
        } else if (['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'paragraph', 'paragraph-quote', 'block-quote', 'monospace', 'monospace-inline', 'cite'].indexOf(operationName) > -1) {
            this.editor.execCommand('mceToggleFormat', true, tinymceMapped);
        } else if (['bold', 'italic', 'underline', 'delete', 'superscript', 'subscript'].indexOf(operationName) > -1) {
            if (this.selection.trimSelection()) {
                if (!this.selection.hasSelection()) {
                    Analytics.sendEvent("editor.instance.selection.collapsed", {op: operationName});
                    return false;
                } else {
                    Analytics.sendEvent("editor.instance.selection.trimmed", {op: operationName});
                }
            }

            this.editor.execCommand(tinymceMapped, true);
        } else if (['bullet-list', 'numbered-list'].indexOf(operationName) > -1) {
            this.editor.execCommand(tinymceMapped, true);
        } else if (this._emoticonSourceMap(operationName)) {
            var imageContextPath;
            if (AJS && AJS.contextPath) {
                imageContextPath = AJS.contextPath();
            } else {
                imageContextPath = '';
            }
            var sourceImageUrl = imageContextPath + '/images/icons/emoticons/' + this._emoticonSourceMap(operationName);
            this.editor.insertContent('<img class="emoticon" src="' + sourceImageUrl + '" height="16" width="16" align="absmiddle" alt="" border="0">');
        } else if ("link" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.wiki.editor.operation.link.external.placeholder'));
            this.editor.selection.setContent('[' + content + '|http://example.com]');
        } else if ("link-anchor" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.wiki.editor.operation.link.anchor.placeholder'));
            this.editor.selection.setContent('[#' + content + ']');
        } else if ("link-mail" === operationName) {
            var content = (this._selectedTextSanitized() || 'mail@example.com');
            this.editor.selection.setContent('[mailto:' + content + ']');
        } else if ("mention" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.wiki.editor.operation.userMention.placeholder'));
            this.editor.selection.setContent('@');
            var start = this.getSelectionStart();
            this.editor.selection.setContent(content);
            this.setSelectionStart(start);
        } else if ("attachment" === operationName && args.attachment) {
            if (args.attachment.type === 'image') {
                var source = args.attachment.href || contextPath() + '/images/icons/attach/image.gif';
                if (args.attachment.thumbnailable) {
                    source = source.replace('/attachment/', '/thumbnail/');
                }

                var attachment = JIRA.Editor.Tags.Templates.attachedImage({
                    source: source,
                    filename: args.attachment.name
                });
            } else {
                var attachment = JIRA.Editor.Tags.Templates.attachedFile({
                    title: args.attachment.name,
                    href: args.attachment.href,
                    filename: args.attachment.name
                });
            }

            this.editor.insertContent(attachment);
        } else if ("code" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.editor.macro.code.placeholder') + '\n');
            this.editor.selection.setContent('<pre class="code panel" data-language="code-java">' + content + '</pre>');
        } else if ("panel" === operationName) {
            var content = (this.editor.selection.getContent() || AJS.I18n.getText('jira.wiki.editor.operation.panel.placeholder'));
            this.editor.selection.setContent('<div class="plain panel" style="border-width: 1px;"><panel-title>' + AJS.I18n.getText('jira.wiki.editor.operation.panel.placeholder.title') + '</panel-title>' +
                '<p>' + content + '</p></div>');
        } else if ("noformat" === operationName) {
            var content = (this._selectedTextSanitized() || AJS.I18n.getText('jira.wiki.editor.operation.noFormat.placeholder'));
            this.editor.selection.setContent('<pre class="noformat panel">' + content + '</pre>');
        } else if ("table" === operationName) {
            var content = (this.editor.selection.getContent() || (AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.column') + ' A1'));
            this.editor.selection.setContent('<div class="table-wrap"><table class="confluenceTable mce-item-table" data-mce-selected="1"><tbody><tr><th class="confluenceTh">' +
                AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.heading') + ' 1</th><th class="confluenceTh">' +
                AJS.I18n.getText('jira.wiki.editor.operation.table.placeholder.heading') + ' 2</th></tr><tr><td class="confluenceTd">' +
                content + '</td><td class="confluenceTd">Col A2</td></tr></tbody></table></div>');
            Analytics.sendEvent("editor.instance.table.toolbar");
        } else if ("speech" === operationName) {
            SpeechRecognition.start(null, this);
        } else if (["link-attachment", "image", "image-attachment"].indexOf(operationName) > -1) {
            logger.warn('Not supported yet ' + operationName);
            return false;
        } else {
            logger.warn('Unsupported operation ' + operationName);
            return false;
        }

        Analytics.sendEvent("editor.instance.operation." + operationName);
        this.trigger("content");
        logger.trace("jira.editor.operation.executed");
        return true;
    };

    EditorInstance.prototype._mapOperationNameToTinymce = function (operationName) {
        var map = {
            'bold': 'Bold',
            'italic': 'Italic',
            'underline': 'Underline',
            'bullet-list': 'InsertUnorderedList',
            'numbered-list': 'InsertOrderedList',
            'blockquote': 'mceBlockQuote',
            'paragraph': 'p',
            'paragraph-quote': 'blockquote',
            'block-quote': 'blockquote',
            'monospace': 'samp',
            'monospace-inline': 'samp-inline',
            'delete': 'strikethrough',
            'superscript': 'superscript',
            'subscript': 'subscript'
        };
        if (operationName in map) {
            return map[operationName];
        }
        return operationName;
    };

    EditorInstance.prototype._emoticonSourceMap = function (emoticonName) {
        var emoticonSourceMap = {
            ':)': 'smile.png', // We can't distinguish between ':-)' and ':)' , they render the same
            ':(': 'sad.png',
            ':P': 'tongue.png',
            ':D': 'biggrin.png',
            ';)': 'wink.png',
            '(y)': 'thumbs_up.png',
            '(n)': 'thumbs_down.png',
            '(i)': 'information.png',
            '(/)': 'check.png',
            '(x)': 'error.png',
            '(!)': 'warning.png',

            '(+)': 'add.png',
            '(-)': 'forbidden.png',
            '(?)': 'help_16.png',
            '(on)': 'lightbulb_on.png',
            '(off)': 'lightbulb.png',
            '(*)': 'star_yellow.png', // We can't distinguish between '(*y)' and '(*)', they render the same
            '(*r)': 'star_red.png',
            '(*g)': 'star_green.png',
            '(*b)': 'star_blue.png',
            '(*y)': 'star_yellow.png',

            '(flag)': 'flag.png',
            '(flagoff)': 'flag_grey.png'
        };
        if (emoticonName in emoticonSourceMap) {
            return emoticonSourceMap[emoticonName];
        }
    };

    /**
     * focuses the editor (places dom focus in the editor).
     */
    EditorInstance.prototype.focus = function () {
        if (!this.editor.initialized || this.editor.removed || this.editor.destroyed) {
            logger.warn('bypassing `editor-instance.focus` because the editor instance hasn\'t initialized yet or already removed');
            return this;
        }
        this.editor.focus();
        this.editor.nodeChanged();  // needed for toolbar UI update only
        return this;
    };

    /**
     * destroys instance and removes used resources
     */
    EditorInstance.prototype.destroy = function () {
        this.editor.contextManager = null;
        this.editor.remove();

        Analytics.sendEvent("editor.instance.destroy");
        Analytics.sendEvent("bundled.editor.instance.destroy");
    };

    /**
     * Sets the content into editor
     * @param content Content to set
     * @param raw
     * @param silent Do not trigger on change listeners
     */
    EditorInstance.prototype.setContent = function (content, raw, silent) {
        var options = {};
        if (raw) {
            options = {format: 'raw'};
        }
        options.no_events = silent;

        content = content || '';
        this.editor.setContent(content, options);

        this.lastContent = this.getContent();

        if (!silent) {
            this.trigger("content");
        }
    };

    EditorInstance.prototype.replaceSelection = function (content) {
        this.editor.selection.setContent(content);

        this.trigger("content");
    };

    EditorInstance.prototype.selectAll = function () {
        this.editor.selection.select(this.editor.getBody(), true);
    };

    EditorInstance.prototype.getSelectionStart = function () {
        return this.editor.selection.getRng(true);
    };

    EditorInstance.prototype.setSelectionStart = function (selectionStart) {
        var rng = this.editor.selection.getRng(true);
        rng.setStart(selectionStart.startContainer, selectionStart.startOffset);
        this.editor.selection.setRng(rng);
    };

    /**
     * Gets the content from editor
     * @param raw
     */
    EditorInstance.prototype.getContent = function (raw) {
        var options = {};
        if (raw) {
            options = {format: 'raw'};
        }
        return this.editor.getContent(options);
    };

    EditorInstance.prototype._onChange = function () {
        if (this.editor.destroyed) {
            return;
        }

        var newContent = this.getContent();
        var changed = newContent !== this.lastContent;
        this.lastContent = newContent;

        if (!this.hidden && changed) {
            this.trigger("content", this.getContent());
        }
    };

    EditorInstance.prototype.getSelectedContent = function (plainText) {
        if (plainText) {
            return this.editor.selection.getContent({format: 'text'});
        } else {
            return this.editor.selection.getContent();
        }
    };

    EditorInstance.prototype.addShortcut = function (shortcut, callback) {
        this.editor.addShortcut(shortcut, '', callback);
    };

    EditorInstance.prototype.removeShortcut = function (shortcut) {
        this.editor.shortcuts.remove(shortcut);
    };

    EditorInstance.prototype.addOperationOverride = function (operationName, fun) {
        this._assertOperationIsSupported(operationName);
        this.operationOverride[operationName] = fun;
    };

    EditorInstance.prototype.removeOperationOverride = function (operationName) {
        this._assertOperationIsSupported(operationName);
        delete this.operationOverrideoperationOverride[operationName];
    };

    EditorInstance.prototype.hide = function () {
        this.hidden = true;
        this.editor.hide();
    };

    EditorInstance.prototype.show = function () {
        this.editor.show();
        delete this.hidden;
    };

    EditorInstance.prototype.isVisible = function () {
        return !this.editor.isHidden();
    };

    EditorInstance.prototype.switchMode = function (showSource) {
        this.trigger("switchMode", showSource);
    };

    EditorInstance.prototype.enable = function () {
        this.editor.setProgressState(false);
    };

    EditorInstance.prototype.disable = function () {
        this.editor.setProgressState(true);
    };

    function shouldInsertLinebreak(e, $selectionStart) {
        const tableElements = 'th, td, th *, td *';
        const excludedElements = 'li';

        return e.keyCode === tinymce.util.VK.ENTER && !e.shiftKey && $selectionStart.is(tableElements) && !$selectionStart.is(excludedElements);
    };

    return EditorInstance;
});
