define("jira/editor/converter", [
    "jquery",
    "jira/editor/converter/util/rte-strings",
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jira/editor/converter/new-lines/new-lines",
    "jira/editor/converter/new-lines/new-lines-merger",
    "jira/editor/converter/handler-mapper",
    "jira/editor/analytics",
    "exports"
], function (
    $,
    RteStrings,
    NodeHandler,
    EmptyHandler,
    NewLines,
    NewLinesMerger,
    HandlerMapper,
    Analytics,
    exports
) {
    "use strict";

    HandlerMapper.registerHandlers();

    var handleDom = function (dom) {
        // IE11 fix DOMParser.parseFromString('', 'text/html') returns document with document.body==null
        if (dom == null) {
            return "";
        }

        var handlerStack = [NodeHandler.createHandler(dom)];

        var textStack = [];

        // By processing the document elements are converted to the handler objects
        // the handler objects are stored in the handler stack each handler object has children,
        // which should be add to handler stack too.

        // Text stack contain both NewLine markers and strings, sibling NewLine markers are merged to
        // strings after processing the whole stack
        while (handlerStack.length !== 0) {
            var handler = handlerStack.pop();
            if (handler instanceof EmptyHandler) {
                continue;
            }

            if (handler instanceof NewLines) {
                textStack.push(handler);
                continue;
            }

            handlerStack.push(handler.newLinesBefore());
            handlerStack.push(handler.before());

            var children = handler.children();
            handlerStack = handlerStack.concat(children);

            handlerStack.push(handler.after());

            textStack.push(handler.processText());
            textStack.push(handler.newLinesAfter());
        }

        // text stack contains both new lines handlers and text handlers
        textStack = NewLinesMerger.mergeNewLines(textStack);

        // reversing the text to get the natural order
        var result = RteStrings.trim(textStack.reverse().join(''));
        return result;
    };

    /**
     * Convert given html markup to wiki markup. This method is synchronous for internal use only.
     * @internal
     * @param {String} html
     * @return {String}
     */
    var handleMarkup = function (html) {
        return handleDom(RteStrings.parseHtml(html))
    };

    /**
     * Convert given HTML to Wiki text markup.
     *
     * @param {String} html
     * @returns {Promise} which will be resolved to wiki markup or reject if conversion was not possible.
     */
    var convert = function (html) {
        // deferring conversion result, because at some point we might need to do some asynchronous operation(i.e. REST call)
        var deferred = new $.Deferred();

        Analytics.mark("rte-convert");
        try {
            var dom = RteStrings.parseHtml(html);
            var nodesCount = (dom ? dom.getElementsByTagName('*').length : 0);
            var markup = handleDom(dom);

            deferred.resolve(markup);
            Analytics.oneShotMeasure("rte-convert", { nodesCount: nodesCount });
        } catch (e) {
            deferred.reject(e);
            Analytics.oneShotMeasure("rte-convert", { nodesCount: -1 });
        }

        return deferred.promise();
    };

    exports.handleMarkup = handleMarkup;
    exports.convert = convert;
});