/**
 * @class JIRA.DevStatus.SummaryTransitionView
 * @extends Backbone.View
 *
 * A generic view used for applying transition to any summary data
 * Currently supports:
 * 1. Rolling transition from the top for update
 * 2. Sliding down for new data
 * 3. Sliding up for deleted data
 */
Backbone.define('JIRA.DevStatus.SummaryTransitionView', Backbone.View.extend({
    slideTransitionTime: 500,
    initialize: function() {},

    renderVisible: function(content) {
        /**
         * Check whether the container is in the DOM
         */
        var lateStartTransition = !AJS.$.contains(document, this.$el[0]);

        if (this._hasPreviousData()) {
            // Previous data exist, perform a roll transition if data has changed
            if (this._hasDataChanged()) {
                this._prepareContentsForTransition(content, this.$el);
                this.$el.addClass("roll-transition");
            }
        } else {
            // Previous data doesn't exist, perform a slide down transition
            this._preStartSlideTransition();
            this.$el.addClass("slide-down-transition");
            lateStartTransition = true;
        }

        this.$el.html(content);
        /**
         * Adding tooltip needs to happen after element has been rendered into $el (DOM), otherwise nothing will happen
         */
        JIRA.DevStatus.Date.addTooltip(content);

        if (lateStartTransition) {
            /**
             * Elements have to be in the DOM and have their dimensions calculated by the browser before starting the animation.
             * Otherwise, animation may start too early and it doesn't look nice
             */
            _.defer(_.bind(this._initiateTransition, this));
        } else {
            this._initiateTransition();
        }
    },

    renderHidden: function(callback) {
        var instance = this;
        this.$el.slideUp(this.slideTransitionTime, function() {
            /**
             * slideUp set display:none on element. Removing the none value to display because it overrides .show()
             * (in other words .show() won't show element that has been display:none)
             * Passing an empty string will remove the css attribute
             */
            instance.$el.css("display", "");
            callback();
        });
    },

    _initiateTransition: function() {
        if (this.$el.hasClass("slide-down-transition")) {
            this._postStartSlideTransition();
            this.$el.removeClass("slide-down-transition");
            this._startSlideDownTransition(this.$el, 0);
        } else {
            if (this.$el.hasClass("roll-transition")) {
                this.$el.removeClass("roll-transition");
                this._startRollTransition();
            }

            /**
             * Find any sliding containers and transition those as well
             */
            var slidingContainers = this.$el.find(".sliding-container");
            if (slidingContainers.length > 0) {
                var instance = this;
                slidingContainers.each(function(index, container) {
                    var $container = AJS.$(container);
                    var prevHeight = $container.data("prevHeight");
                    instance._startSlideDownTransition($container, prevHeight);
                });
            }
        }
    },

    _startSlideDownTransition: function(element, previousHeight) {
        var currentHeight = element.height();
        /**
         * Animate the height if the content height is different between 2 different (previous and current) renderings
         */
        if (previousHeight !== currentHeight) {
            element.height(previousHeight).animate({height: currentHeight}, this.slideTransitionTime, function() {
                //Setting height back to auto to prevent any dangling old values
                element.css("height", "auto");
                element.find(".rolling-content").removeClass("transit");
            });
        }
    },

    _startRollTransition: function() {
        /**
         * Find each rolling-container and animate them independently.
         */
        this.$el.find(".rolling-container").each(_.bind(function(index, container) {
            var $container = AJS.$(container);
            var rollingContent = $container.find(".rolling-content");
            var summaryContent = rollingContent.find(":not(.old-content)");
            var contentHeight = summaryContent.height();

            $container.height(contentHeight);
            rollingContent.addClass("transit");

            summaryContent.css("margin-top", "-" + contentHeight + "px").animate({"margin-top": "0"}, this.slideTransitionTime, function() {
                $container.css("height", "auto");
                $container.find(".old-content").remove();

                rollingContent.removeClass("transit");
                summaryContent.css("margin-top", ""); //Removing the CSS on element
            });
        }, this));
    },

    _prepareContentsForTransition: function(newContent, oldContainer) {
        /**
         * Append content into a container, so we don't need to make the distinction between .find and .filter
         */
        newContent = AJS.$("<div></div>").append(newContent);

        /**
         * Calculate the height of the old content so it can be animated to the new height
         */
        var newSlideContainers = newContent.find(".sliding-container");
        var oldSlideContainers = oldContainer.find(".sliding-container");
        newSlideContainers.each(function(index, content) {
            var oldSlideContainer = oldSlideContainers.get(index);
            if (oldSlideContainer) {
                var oldContainer = AJS.$(oldSlideContainer);
                AJS.$(content).data("prevHeight", oldContainer.height());
            }
        });

        /**
         * Animation works by basically appending the old content as sibling of the new content in the DOM tree.
         * It's done by matching/pairing elements at the same index.
         *
         * These makes the assumption that between 2 different renderings, elements are in the same order.
         * This should be OK because we can control it in soy.
         *
         * The number of elements between 2 different renderings can be different, which is fine. The following logic
         * will simply match up what is available and ignore the rest.
         */
        var newRollContainers = newContent.find(".rolling-container");
        var oldRollContainers = oldContainer.find(".rolling-container");
        newRollContainers.each(function(index, content) {
            var newContainer = AJS.$(content).find(".rolling-content");
            var oldRollContainer = oldRollContainers.get(index);
            if (oldRollContainer) {
                var oldContents = AJS.$(oldRollContainer).find(".rolling-content").children();
                oldContents.addClass("old-content");
                newContainer.append(oldContents);
            }
        });

    },

    _hasPreviousData: function() {
        var prevData = this.model.getPreviousOverall();
        return prevData && prevData.count > 0;
    },

    _hasDataChanged: function() {
        return !_.isEqual(this.model.getOverall(), this.model.getPreviousOverall());
    },

    /**
     * The following 2 methods are a slight hack. Reasons:
     * 1. Slide down animation is basically done by setting height to 0 then set it back to the actual height.
     * 2. In order to get the actual height, elements have to be already rendered in the DOM and shown
     * 3. This produces a quick 'flash' when elements have the full height before being set to 0
     * 4. To counteract the effect, height and opacity is set to 0 here before being shown
     *    then immediately remove them before starting a slide down animation
     */
    _preStartSlideTransition: function() {
        this.$el.css("height", "0");
        this.$el.css("opacity", "0");
    },

    _postStartSlideTransition: function() {
        this.$el.css("height", "");
        this.$el.css("opacity", "");
    }
}));