AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:test-mocks");

(function() {

    var HINT_TEXT = "Learn about this issue or go to application links to fix it.";

    module("JIRA.DevStatus.SummaryErrorView", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.fixture = jQuery("#qunit-fixture");
            this.createPanel();
        },
        teardown : function () {
            this.sandbox.restore();
        },
        createPanel: function() {
            this.panel = jQuery('<div class="message-panel hidden">').appendTo(this.fixture);
        },

        /**
         * Creates a new MockSummaryModel.
         * 
         * @param {object} [state]
         * @param {boolean} [state.hasErrors] whether the model has errors 
         * @param {Array} [state.errorInstances] the instances with transient errors
         * @param {Array} [state.configInstances] the instances with configuration problems
         * @returns {JIRA.DevStatus.MockSummaryModel}
         */
        createMockModel: function(state) {
            state = state || {};
            var errorInstances = state.errorInstances || [];
            var configInstances = state.configInstances || [];
            var hasErrors = _.isUndefined(state.hasErrors) ? !(_.isEmpty(errorInstances) && _.isEmpty(configInstances)) : state.hasErrors;
            
            var model = new JIRA.DevStatus.MockSummaryModel();
            model.get.withArgs("hasErrors").returns(hasErrors);
            model.get.withArgs("errorInstances").returns(errorInstances);
            model.get.withArgs("configInstances").returns(configInstances);
            return model;
        },

        /**
         * Creates a number of instances numbered [1..n].
         *
         * @param {number} count number of instances to return
         * @return {Array}
         */
        createInstances: function(count) {
            return _.map(_.range(1, count+1), function(i) {
                return {
                    name: "inst_" + i,
                    baseUrl: "#" + i
                }
            });
        },

        assertGenericMsg: function(exists) {
            var index = this.panel.html().indexOf('JIRA is having difficulty contacting the applications that supply development status information');
            ok(exists ? index >= 0 : index < 0, "generic error msg is " + (exists ? "" : "not ") + "shown");
        },
        assertAndGetGenericMsgIndex: function() {
            var errorMsgPrefix = 'JIRA is having difficulty contacting ';
            var beginIndexMsg = this.panel.html().indexOf(errorMsgPrefix);
            ok(beginIndexMsg >= 0, "instance error msg is shown");
            return beginIndexMsg + errorMsgPrefix.length + 1;
        },
        /**
         * assert that a pattern text occurs after startIndex.
         */
        assertOccur: function(html, text, startIndex, msg) {
            ok(html.indexOf(text, startIndex) >=0, msg);
        },
        /**
         * assert that a pattern text occurs after startIndex, but before beforeText.
         */
        assertOccurBefore: function(html, text, beforeText, startIndex, msg) {
            var index = html.indexOf(text, startIndex);
            var beforeIndex = html.indexOf(beforeText, startIndex);
            ok(index >= 0 && index < beforeIndex, msg);
        },
        /**
         * assert that a pattern text occurs after startIndex, and after afterText.
         */
        assertOccurAfter: function(html, text, afterText, startIndex, msg) {
            var afterIndex = html.indexOf(afterText, startIndex);
            var index = html.indexOf(text, afterIndex);
            ok(index >= 0 && afterIndex >=0, msg);
        },
        /**
         * assert that a pattern text occurs after startIndex, and between beginText and endText
         * (both of which also occur after startIndex).
         */
        assertOccurBetween: function(html, text, beginText, endText, startIndex, msg) {
            var beginIndex = html.indexOf(beginText, startIndex);
            var index = html.indexOf(text, beginIndex + beginText.length);
            var endIndex = html.indexOf(endText, beginIndex + beginText.length);
            ok(beginIndex >=0 && index >= 0 && endIndex >=0 && index > beginIndex && index < endIndex, msg);
        },
        /**
         * assert that a pattern text does not occur after startIndex, and between beginText and endText
         */
        assertNotOccurBetween: function(html, text, beginText, endText, startIndex, msg) {
            var beginIndex = html.indexOf(beginText, startIndex);
            var index = html.indexOf(text, beginIndex + beginText.length);
            var endIndex = html.indexOf(endText, beginIndex + beginText.length);
            ok(index < 0 || (index >= 0 && beginIndex >=0 && endIndex >=0 && (index < beginIndex || index > endIndex)), msg);
        },

        /**
         * Returns the config problem text if any. This only returns <b>visible</b> elements.
         *
         * @returns {{description: jQuery, hints: jQuery, link: Function}}
         */
        getConfigErrors: function() {
            var configErrors = this.panel.find('.config-errors:visible');
            return {
                description: configErrors.find('.description'),

                hints: configErrors.find('.hints'),

                findLink: function(spanName) {
                    return configErrors.find(AJS.format('span[data-name={0}] a', spanName));
                }
            };
        }
    });

    test("don't show panel when no errors", function() {
        var model = this.createMockModel();
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(this.panel.hasClass("hidden"), "panel is hidden");
        ok(!this.panel.html(), "content is not rendered");
    });

    test("show panel when errors without instances", function() {
        var model = this.createMockModel({ hasErrors: true });
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(true);
    });

    test("show panel when errors with 1 instance", function() {
        var model = this.createMockModel({ errorInstances: [
                {name: "instance 1", baseUrl: "#"}
        ]});
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(false);
        var index = this.assertAndGetGenericMsgIndex();
        var panelHtml = this.panel.html();
        this.assertOccur(panelHtml, 'instance 1', index, "instance 1 is shown");
    });

    test("show panel when errors with 2 instances", function() {
        var model = this.createMockModel({ errorInstances: [
            {name: "instance 1", baseUrl: "#"},
            {name: "instance 2", baseUrl: "#"}
        ]});
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(false);
        var index = this.assertAndGetGenericMsgIndex();
        var panelHtml = this.panel.html();
        this.assertOccurBefore(panelHtml, 'instance 1', "instance 2", index, "instance 1 is shown before instance 2");
        this.assertOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'1 and 2' is shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'1, and 2' is not shown");
        this.assertOccurAfter(panelHtml, 'instance 2', 'instance 1', index, "instance 2 is shown");
    });

    test("show panel when errors with 3 instances", function() {
        var model = this.createMockModel({ errorInstances: [
            {name: "instance 1", baseUrl: "#"},
            {name: "instance 2", baseUrl: "#"},
            {name: "instance 3", baseUrl: "#"}
        ]});
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(false);
        var index = this.assertAndGetGenericMsgIndex();
        var panelHtml = this.panel.html();
        this.assertOccurBefore(panelHtml, 'instance 1', "instance 2", index, "instance 1 is shown before instance 2");
        this.assertOccurBetween(panelHtml, '>, ', 'instance 1', 'instance 2', index, "'1, 2' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'1 and 2' is not shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'1, and 2' is not shown");
        this.assertOccurBetween(panelHtml, 'instance 2', 'instance 1', 'instance 3', index, "instance 2 is shown");
        this.assertOccurBetween(panelHtml, '>, and ', 'instance 2', 'instance 3', index, "'2, and 3' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'2 and 3' is not shown");
        this.assertOccurAfter(panelHtml, 'instance 3', 'instance 2', index, "instance 3 is shown");
    });

    test("show panel when errors with 4 instances", function() {
        var model = this.createMockModel({ errorInstances: [
            {name: "instance 1", baseUrl: "#"},
            {name: "instance 2", baseUrl: "#"},
            {name: "instance 3", baseUrl: "#"},
            {name: "instance 4", baseUrl: "#"}
        ]});
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(false);
        var index = this.assertAndGetGenericMsgIndex();
        var panelHtml = this.panel.html();
        this.assertOccurBefore(panelHtml, 'instance 1', "instance 2", index, "instance 1 is shown before instance 2");
        this.assertOccurBetween(panelHtml, '>, ', 'instance 1', 'instance 2', index, "'1, 2' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'1 and 2' is not shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'1, and 2' is not shown");
        this.assertOccurBetween(panelHtml, 'instance 2', 'instance 1', 'instance 3', index, "instance 2 is shown");
        this.assertOccurBetween(panelHtml, '>, ', 'instance 2', 'instance 3', index, "'2, 3' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'2 and 3' is not shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'2, and 3' is not shown");
        this.assertOccurBetween(panelHtml, 'instance 3', 'instance 2', 'another application', index, "instance 3 is shown");
        this.assertOccurAfter(panelHtml, 'and another application', 'instance 3', index, "'and another instance' is shown");
    });

    test("show panel when errors with 5 instances", function() {
        var model = this.createMockModel({ errorInstances: [
            {name: "instance 1", baseUrl: "#"},
            {name: "instance 2", baseUrl: "#"},
            {name: "instance 3", baseUrl: "#"},
            {name: "instance 4", baseUrl: "#"},
            {name: "instance 5", baseUrl: "#"}
        ]});
        var view = new JIRA.DevStatus.SummaryErrorView({
            model: model,
            el: this.panel
        });
        view.render();
        ok(!this.panel.hasClass("hidden"), "panel is not hidden");
        ok(this.panel.html(), "content is rendered");
        this.assertGenericMsg(false);
        var index = this.assertAndGetGenericMsgIndex();
        var panelHtml = this.panel.html();
        this.assertOccurBefore(panelHtml, 'instance 1', "instance 2", index, "instance 1 is shown before instance 2");
        this.assertOccurBetween(panelHtml, '>, ', 'instance 1', 'instance 2', index, "'1, 2' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'1 and 2' is not shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'1, and 2' is not shown");
        this.assertOccurBetween(panelHtml, 'instance 2', 'instance 1', 'instance 3', index, "instance 2 is shown");
        this.assertOccurBetween(panelHtml, '>, ', 'instance 2', 'instance 3', index, "'2, 3' is shown");
        this.assertNotOccurBetween(panelHtml, '> and ', 'instance 1', 'instance 2', index, "'2 and 3' is not shown");
        this.assertNotOccurBetween(panelHtml, '>, and ', 'instance 1', 'instance 2', index, "'2, and 3' is not shown");
        this.assertOccurBetween(panelHtml, 'instance 3', 'instance 2', ' and 2 more applications', index, "instance 3 is shown");
        this.assertOccurAfter(panelHtml, 'and 2 more applications', 'instance 3', index, "'and 2 more applications' is shown");
    });

    test('show configuration error for 1 instance', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: this.createInstances(1)})
        }).render();

        var configErrors = this.getConfigErrors();
        equal(configErrors.description.text(), "Couldn't read data from inst_1.", "config error shown for inst_1");
        equal(configErrors.hints.text(), HINT_TEXT, "config hints shown");
    });

    test('show configuration error for 2 instances', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: this.createInstances(2)})
        }).render();

        var configErrors = this.getConfigErrors();
        equal(configErrors.description.text(), "Couldn't read data from inst_1 and inst_2.", "config error shown for inst_1+inst_2");
        equal(configErrors.hints.text(), HINT_TEXT, "config hints shown");
    });

    test('show configuration error for 3 instances', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: this.createInstances(3)})
        }).render();

        var configErrors = this.getConfigErrors();
        equal(configErrors.description.text(), "Couldn't read data from inst_1, inst_2, and inst_3.", "config error shown for inst_1+inst_2+inst_3");
        equal(configErrors.hints.text(), HINT_TEXT, "config hints shown");
    });

    test('show configuration error for 4 instances', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: this.createInstances(4)})
        }).render();

        var configErrors = this.getConfigErrors();
        equal(configErrors.description.text(), "Couldn't read data from inst_1, inst_2, inst_3, and 1 other application.", "config error shown for inst_1+inst_2+inst_3+1 more");
        equal(configErrors.hints.text(), HINT_TEXT, "config hints shown");
    });

    test('show configuration error for 5 instances', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: this.createInstances(5)})
        }).render();

        var configErrors = this.getConfigErrors();
        equal(configErrors.description.text(), "Couldn't read data from inst_1, inst_2, inst_3, and 2 other applications.", "config error shown for inst_1+inst_2+inst_3+2 more");
        equal(configErrors.hints.text(), HINT_TEXT, "config hints shown");
    });

    test('configuration error links are correct', function() {
        var instances = this.createInstances(4);
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({ configInstances: instances })
        }).render();

        var configErrors = this.getConfigErrors();

        var firstThreeServers = instances.slice(0, 3);
        _.each(firstThreeServers, function(server) {
            var name = server.name;
            var baseUrl = server.baseUrl;
            var link = configErrors.findLink(name);

            deepEqual({ href: link.attr('href'), text: link.text() }, { href: baseUrl, text: name }, AJS.format('{0} links to {1}', server.name, baseUrl));
        });
    });

    test('show both connection and configuration errors', function() {
        var view = new JIRA.DevStatus.SummaryErrorView({
            el: this.panel,
            model: this.createMockModel({
                errorInstances: [ {name: "inst_1", baseUrl: "#1"} ],
                configInstances: [ {name: "inst_2", baseUrl: "#2"} ]
            })
        }).render();

        equal(this.getConfigErrors().description.text(), "Couldn't read data from inst_2.", "config error is shown");
        this.assertOccur(this.panel.text(), "JIRA is having difficulty contacting inst_1", "connection error is also shown");
    });

})();
