/**
 * @class JIRA.DevStatus.SummaryErrorModel
 * @extends Backbone.Model
 */
Backbone.define('JIRA.DevStatus.SummaryErrorModel', Backbone.Model.extend({
    properties: [
        "hasErrors", // whether there are error(s) from the REST api call or not
        "errorInstances", // transient errors,
        "configInstances" // non-transient aka configuration errors,
    ],

    defaults: function() {
        return {
            "hasErrors": false,
            "errorInstances": [],
            "configInstances": []
        };
    },

    /**
     * Extract error information, if any, from the aggregatedData.
     * @param {object} aggregatedData the data to extract errors from, or undefined if
     *                       no data was returned.
     * @param {object} [aggregatedData.errors] the data to extract errors from, or undefined
     * @param {object} [aggregatedData.configErrors] the data to extract configuration errors from, or undefined
     */
    updateData: function(aggregatedData) {
        var errorInstances = this._extractErrors(aggregatedData && aggregatedData.errors);
        var configInstances = this._extractErrors(aggregatedData && aggregatedData.configErrors);

        // no data indicates an error in the ajax call
        this.set("hasErrors", !aggregatedData || !_.isEmpty(errorInstances) || !_.isEmpty(configInstances));
        this.set("errorInstances", errorInstances);
        this.set("configInstances", configInstances);
    },

    /**
     * Extracts the instance error data from an errors array.
     *
     * @param {Array} instances the array of instances to extract
     * @returns {Array} an array of instances
     */
    _extractErrors: function (instances) {
        if (!instances) {
            return [];
        }

        return _.map(
                _.filter(instances, function(dataWithInstance) {
                    return dataWithInstance.error; // keep it if it has an "error" field
                }),
                function(errorWithInstance) {
                    // ignore the actual error message
                    return errorWithInstance.instance;
                });
    }
}));
