Backbone.define('JIRA.DevStatus.CreateBranchView', Backbone.View.extend({
    /**
     * @param options
     * @param options.showTooltip
     */
    initialize: function (options) {
        this.tooltipShown = !options.showTooltip;

        this.linkTextSelector = '.devstatus-cta-link-text';
        this.issueContainer = ".issue-container";

        this._processLinkForNavigableURL();
        this._initTooltip();
        // Open the create branch dialog if instructed to do so from the URL
        _.defer(_.bind(this._openCreateBranchDialog, this));
    },

    getCreateBranchLink: function() {
        return this.$el;
    },

    /**
     * Swap the 'href' of this link into a navigable link, that is, a link that can be opened via meta/ctr + click or middle click
     */
    _processLinkForNavigableURL: function() {
        /**
         * Some history and why I wrote the code below:
         *
         * In Fusion phase 1, we store the call-to-action key of a link in the url via a hash (ie #devstatus.cta.createbranch) in atlassian-plugin.xml
         * Then we subsequently use regex to extract the key out of the link to display the correct detail dialog.
         * We did this because the create-branch link needs to appear in multiple places
         *    (cog menu from dashboard/issue navigator, more dropdown in view issue and dot dialog).
         * We had 2 options at the time, to use the 'class' attribute or the above.
         * Using the URL via hash method was decided to be the best at the time
         *
         * In order to automatically open the detail dialog when opening the link in a new context,
         *    I had to append a 'devStatusDetailDialog=create-branch' param to the url.
         *
         * In the process, I have to extract the cta key first and destroy the hash data in the URL. Why? because
         * 1. Including the hash in the URL breaks the link in IE for issue navigator/view issue
         *       (because KA uses the i# prefix for IE8/9. Multiple hash in the url will break everything)
         * 2. JIRA Agile for some reason pulls forward anything in the hash into part of the URL, essentially breaking the URL
         *       (ie /RapidBoard.jspa?id=foo#devstatus.cta.createbranch becomes /devstatus.cta.createbranch?id=foo)
         *
         * So I'm doing a little switcharoo here and store cta key in the element data
         */

        // Store the cta key into element data
        this.getCreateBranchLink().data("cta", this._getCta(this.$el));

        // Swap the URL of the link
        this.getCreateBranchLink().attr("href", JIRA.DevStatus.URL.getUrlWithDetailDialogParam("create-branch"));
    },

    _openCreateBranchDialog: function() {
        var dialogToOpen = AJS.Meta.get("fusion-open-detail-dialog");
        if (dialogToOpen === "create-branch") {
            AJS.Meta.set("fusion-open-detail-dialog", undefined);
            AJS.Meta.set("fusion-analytics-new-context-link", true);
            this.getCreateBranchLink().click();
        }
    },

    _initTooltip: function() {
        if (!this.tooltipShown && this.$el.length > 0) {
            /**
             * Wrapping this in a defer because timing might be tricky when interacting with KA components and as a result
             * we might be rendering the tooltip when there's nothing on the screen.
             *
             * This allow KA components to complete its process before running ours.
             */
            _.defer(_.bind(function() {
                var cta = this.$el.data("cta");
                if (cta) {
                    new AJS.HelpTip({
                        id: cta + '.tooltip',
                        title: AJS.I18n.getText('devstatus.cta.create-branch.tooltip.title'),
                        bodyHtml: JIRA.Templates.DevStatus.Tooltip.createBranch(),
                        anchor: this.$el.find(this.linkTextSelector),
                        isSequence: true,
                        inlineDialogOpts: {
                            //Manually specify the container so the helptip can be positioned correctly
                            container: this.issueContainer
                        }
                    });
                    this.tooltipShown = true;
                    AJS.HelpTip.Manager.showSequences();
                }
            }, this));
        }
    },

    _getCta: function(link) {
        var url = link.attr('href'),
            m = url && /\#([a-zA-Z\.]+)/.exec(url);
        return m && m[1];
    }

}));

