/**
 * @class JIRA.DevStatus.BaseSummaryView
 * @extends Backbone.View
 */
Backbone.define('JIRA.DevStatus.BaseSummaryView', Backbone.View.extend({

    // Child views must implement template
    template: undefined,

    events : {
        "simpleClick .summary": "_onSummaryLinkClick"
    },

    initialize: function (options) {
        this.options = _.clone(options);
        this.model.on("change:hasData", this._addOrRemoveHasDataClass, this);
        this.model.on("change:dataLoading", this._addOrRemoveDataLoadingClass, this);
        this.transitionView = new JIRA.DevStatus.SummaryTransitionView({
            model: this.model,
            el: this.$el
        });
    },

    render: function() {
        var visible = this.isVisible();
        if (visible) {
            var content = AJS.$(this.template(_.extend({
                issueKey: JIRA.Issue.getIssueKey(),
                livestampRelativizeThreshold: this.options.livestampRelativizeThreshold
            }, this.model.getOverall())));

            if (this._shouldAnimateDataTransition()) {
                this.transitionView.renderVisible(content);
            } else {
                this.$el.html(content);
                JIRA.DevStatus.Date.addTooltip(content);
            }

            this._renderNavigableSummaryLink();
            this.show();
        } else if (!this._isHidden()) {
            // if no summary information hide this view completely
            if (this._shouldAnimateDataTransition()) {
                this.transitionView.renderHidden(_.bind(this.hide, this));
            } else {
                this.hide();
            }
        }

        this._addOrRemoveDataLoadingClass();
        this._addOrRemoveHasDataClass();
    },

    /**
     * Generate default parameters that would be useful to all views.
     */
    getDefaultDetailDialogParameters: function() {
        return {
            issueKey: JIRA.Issue.getIssueKey(),
            issueId: JIRA.Issue.getIssueId(),
            tabs: this.model.get("byInstanceType"),
            dataType: this.model.getType(),
            livestampRelativizeThreshold: this.options.livestampRelativizeThreshold,
            showContactAdminForm: this.options.dataAttrs && this.options.dataAttrs["showContactAdminForm"]
        };
    },

    /**
     * Returns the summary link for this sub-panel
     */
    getSummaryLink: function() {
        return this.$el.find("a.summary");
    },

    /**
     * Indicate whether the view should be visible.
     */
    isVisible: function() {
        return this.model.getOverall().count > 0;
    },

    /**
     * Sets this view's <code>js-has-data</code> and <code>data-loading</code> classes depending on the model's state.
     *
     * @return {*}
     * @private
     */
    _addOrRemoveHasDataClass: function() {
        var hasData = this.model.get("hasData");

        this.$el.toggleClass("js-has-data", hasData);
    },

    _renderNavigableSummaryLink: function() {
        this.getSummaryLink().attr("href", JIRA.DevStatus.URL.getUrlWithDetailDialogParam(this.model.getType()));
    },

    /**
     * Sets this view's <code>js-has-data</code> and <code>data-loading</code> classes depending on the model's state.
     * These are used in WebDriver tests to determine if there is any data and if there are further data loads in
     * progress.
     *
     * @return {*}
     * @private
     */
    _addOrRemoveDataLoadingClass: function() {
        var dataLoading = this.model.get("dataLoading");

        this.$el.toggleClass("data-loading", dataLoading);
    },

    _onSummaryLinkClick: function(e) {
        e.preventDefault();
        console.log("Summary view not supported: ", this);
    },

    _shouldAnimateDataTransition: function() {
        //Whether there are previous data
        return !!this.model.getPreviousOverall();
    },

    _isHidden: function() {
        return this.$el.hasClass("hidden");
    },

    show: function() {
        this.$el.removeClass("hidden");
    },

    hide: function() {
        this.$el.addClass("hidden");
    }
}));
