/**
 * View for the Labs on/off links.
 *
 * @type JIRA.DevStatus.Labs.LabsOptInView
 */
Backbone.define("JIRA.DevStatus.Labs.LabsOptInView", Backbone.View.extend({
    events: {
        "click .toggle-labs": "_onClickToggle",
        "click .labs-close": "_onClickDismiss",
        "click .feedback-link": "_onClickFeedback"
    },

    /**
     * Creates a new LabsOptInView.
     *
     * @param options
     * @param options.labsOptIn {JIRA.DevStatus.Labs.LabsOptInModel}
     */
    initialize: function(options) {
        this.labsOptIn = options.labsOptIn;
        this.labsOptIn.on("change", this.render, this);
        this.labsOptIn.on("change:optedIn", this._onChangeOptedIn, this);
    },

    render: function() {
        var labs = this.labsOptIn;
        if (labs.isAllowed() && (labs.isOptedInByAdmin() || !labs.isDismissed())) {
            this._show();
        } else {
            this._hide();
        }

        return this;
    },

    /**
     * Toggles labs on and off.
     *
     * @returns {*}
     * @private
     */
    _onClickToggle: function(e) {
        e.preventDefault();
        this.labsOptIn.toggleOptedIn();
    },

    /**
     * Marks the info box as "dismissed".
     *
     * @returns {*}
     * @private
     */
    _onClickDismiss: function(e) {
        e.preventDefault();
        this.labsOptIn.setDismissed(true);
    },

    /**
     * Pops up the feedback dialog using the JIRA issue collector.
     *
     * @param e
     * @returns {*}
     * @private
     */
    _onClickFeedback: function(e) {
        e.preventDefault();
        this._showFeedbackDialog();
    },

    /**
     * Brings up the feedback dialog when the user opts out.
     *
     * @param model {JIRA.DevStatus.Labs.LabsOptInModel}
     * @param optedIn {boolean} whether the user has opted in or out
     *
     * @private
     */
    _onChangeOptedIn: function(model, optedIn) {
        if (optedIn === false) {
            this._showFeedbackDialog({ summary: AJS.I18n.getText('devstatus.labs.feedback.summary.opt.out') });
        }
    },

    /**
     * Loads the issue collector Javascript from JAC and then pops up the feedback dialog.
     *
     * @param options {object}
     * @param options.summary {string} the pre-filled feedback summary
     * @private
     */
    _showFeedbackDialog: function(options) {
        options = _.defaults({}, options, {
            summary: AJS.I18n.getText('devstatus.labs.feedback.summary')
        });

        var collectorId = JIRA.DevStatus.Labs.LabsOptInView.COLLECTOR_ID;

        // override the page properties for our issue collector only
        window.ATL_JQ_PAGE_PROPS = window.ATL_JQ_PAGE_PROPS || {};
        window.ATL_JQ_PAGE_PROPS[collectorId] = {
            fieldValues: {
                summary: options.summary
            },

            triggerFunction: function(showCollectorDialog) {
                // this should only run after the script has been loaded
                _.defer(function() {
                    showCollectorDialog();
                });
            }
        };

        // load the issue collector JS
        AJS.$.getScript(this._collectorUrlFor(collectorId));
    },

    /**
     * Returns the URL for the issue collector JS file.
     *
     * @returns {string} the URL for the issue collector JS file
     * @private
     */
    _collectorUrlFor: function(collectorId) {
        // JIC should really consider providing short URLs...
        return "https://jira.atlassian.com/s/d41d8cd98f00b204e9800998ecf8427e/en_UK-b435xk-1988229788/6144/9/1.4.0-m6/_/download/batch/com.atlassian.jira.collector.plugin.jira-issue-collector-plugin:issuecollector-embededjs/com.atlassian.jira.collector.plugin.jira-issue-collector-plugin:issuecollector-embededjs.js?collectorId=" + collectorId;
    },

    _show: function() {
        this.$el.html(JIRA.Templates.DevStatus.labsOnOff({
            isOptedIn: this.labsOptIn.isOptedIn(),
            isOptedInByAdmin: this.labsOptIn.isOptedInByAdmin()
        }));
        this.$el.removeClass("hidden");
    },

    _hide: function() {
        this.$el.empty();
        this.$el.addClass("hidden");
    }
}, {
    /**
     * The unique id of the issue collector used for JIRA Labs feedback.
     */
    COLLECTOR_ID: "effe8b72"
}));
