/**
 * Model for the labs opt-in.
 *
 * @type JIRA.DevStatus.Labs.LabsOptInModel
 */
Backbone.define("JIRA.DevStatus.Labs.LabsOptInModel", Backbone.Model.extend({
    defaults: {
        'allowed': false,
        'optedIn': false,
        'optedInByAdmin': false,
        'dismissed': false,
        'id': 1 // no need to use ids as this is a singleton
    },

    /**
     * Whether Labs opt in is allowed or not.
     *
     * @returns {boolean}
     */
    isAllowed: function() {
        return this.get('allowed');
    },

    /**
     * Gets the Labs opted in status.
     *
     * @returns {boolean}
     */
    isOptedIn: function() {
        return this.get('optedIn');
    },

    /**
     * Gets the admin opt-in status for labs.
     *
     * @returns {boolean} whether labs has been globally opted-in by an admin
     */
    isOptedInByAdmin: function() {
        return this.get('optedInByAdmin');
    },

    /**
     * Sets the Labs opted in status.
     *
     * @param optedIn {boolean} whether to opt in or out
     * @returns {jqXHR} a jQuery XHR Object
     */
    setOptedIn: function(optedIn) {
        return this.save({ optedIn: optedIn }, { wait: true }).done(function() {
            // fire analytics with on or off
            JIRA.DevStatus.LabsAnalytics.fireLabsEvent(optedIn);
        });
    },

    /**
     * Toggles the Labs opted in status.
     *
     * @returns {jqXHR} a jQuery XHR Object
     */
    toggleOptedIn: function(optedIn) {
        var isOptedIn = this.isOptedIn();

        return this.setOptedIn(!isOptedIn);
    },

    /**
     * Marks the "labs info" as dismissed or not.
     */
    setDismissed: function(dismissed) {
        return this.save({ dismissed: dismissed }, { wait: true });
    },

    /**
     * Whether the Labs info box has been dismissed.
     *
     * @returns {boolean}
     */
    isDismissed: function() {
        return this.get('dismissed');
    },

    /**
     * Returns the URL for the LabsOptIn REST resource.
     *
     * @returns {string}
     */
    url: function() {
        return AJS.contextPath() + "/rest/dev-status/1.0/labs-opt-in/";
    }

}, {
    /**
     * Creates a new LabsOptInModel, populating it with data from the given DOM element.
     *
     * @param el {Element} an element with a data-labs-json attr containing JSON
     */
    createFromDOM: function(el) {
        var $el = AJS.$(el).find('#devstatus-container');

        return new JIRA.DevStatus.Labs.LabsOptInModel($el.data('labs-json'));
    }
}));
