Backbone.define("JIRA.DevStatus.DetailDialogBuildView", JIRA.DevStatus.BaseDetailDialogView.extend({
    template: JIRA.Templates.DevStatus.DetailDialog.build,

    events: {
        "click a.project-link": "_onClickProjectLink",
        "click a.plan-link": "_onClickPlanLink",
        "click a.build-link": "_onClickBuildLink"
    },

    initialize: function(options) {
        this.analytics = JIRA.DevStatus.BuildsAnalytics;
        JIRA.DevStatus.BaseDetailDialogView.prototype.initialize.call(this, _.extend({
            type: "build",
            width: 1000,
            height: 400
        }, options));
    },

    renderSuccess: function(applicationType, result) {
        var projects = _.reduce(result, function(memo, data) {
            return memo.concat(data.projects || []);
        }, []);
        var container = this.getContentContainer(applicationType);
        container.html(this.template({
            applicationType: applicationType,
            projects: this.sortProjectsAndPlans(projects),
            hasArtifacts: this.hasArtifacts(projects),
            livestampRelativizeThreshold: this.options.livestampRelativizeThreshold
        }));
        this.renderNoPermissionToViewAllWarningAtBottom(applicationType, this.getBuildCount(projects));
        JIRA.DevStatus.Date.addTooltip(container);
        return this;
    },

    getTitle: function(count, issue) {
        return AJS.I18n.getText("devstatus.detail.panel.title.build", count, issue);
    },

    getOAuthMessageInFooter: function(instances) {
        return AJS.I18n.getText('devstatus.authentication.message.build', instances.length);
    },

    getOAuthMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.authenticate.build.title');
    },

    getConnectionMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.build.title');
    },

    getNoPermissionToViewAllMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.build.title');
    },

    hasData: function(detail) {
        return _.reduce(detail, function(memo, data) {
            return memo.concat(data.projects || []);
        }, []).length > 0;
    },

    /**
     * Sort projects by most recent build first and plans from newest to oldest build.
     */
    sortProjectsAndPlans: function(projects) {
        projects = _.sortBy(projects, function(project) {
            return _.last(_.sortBy(project.plans, function(plan) {
                return plan.build.finishedDate;
            })).build.finishedDate;
        }).reverse();
        _.each(projects, function(project) {
            // Dates should _always_ be in ISO format - so ASCII sorting will work
            project.plans = _.sortBy(project.plans, function(plan) {
                return plan.build.finishedDate;
            }).reverse();
        });
        return projects;
    },

    hasArtifacts: function(projects) {
        return _.any(projects, function(project) {
            return _.any(project.plans, function(plan) {
                return plan.build && plan.build.artifacts && plan.build.artifacts.length;
            });
        });
    },

    /**
     * Calculate the total number of builds in all projects.
     */
    getBuildCount: function (projects) {
        return _.union.apply(null, _.map(projects, function (project) {
            // project's plan
            return _.pluck(project.plans, 'key');
        })).length;
    },

    _onClickProjectLink: function() {
        JIRA.DevStatus.BuildsAnalytics.fireDetailProjectClicked();
    },

    _onClickPlanLink: function() {
        JIRA.DevStatus.BuildsAnalytics.fireDetailPlanClicked();
    },

    _onClickBuildLink: function() {
        JIRA.DevStatus.BuildsAnalytics.fireDetailBuildClicked();
    }
}));
