AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-build-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-contract-test-resource");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");

module("JIRA.DevStatus.DetailDialogBuildView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "DEV-1";
        this.issueId = 10000;
        this.server = sinon.fakeServer.create();

        JIRA.DevStatus.QunitTestUtils.spyAnalyticMethods("Builds", this.sandbox);
    },
    teardown : function () {
        this.sandbox.restore();
        this.server.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
    },
    getDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            heading: dialog.find(".devstatus-dialog-heading"),
            content: dialog.find(".devstatus-dialog-content")
        }
    },
    createView: function(summaryBuildCount, dialogBuildCount) {
        return new JIRA.DevStatus.DetailDialogBuildView({
            count: summaryBuildCount,
            dataType: "build",
            issueKey: this.issueKey,
            tabs: {
                bamboo: {name: "Bamboo", count: dialogBuildCount}
            }
        });
    },
    assertNoNoPermissionMessage: function() {
        ok(this.getDetailDialog().content.find(".no-permission-to-view-all").length === 0, "There is NO no-permission-to-view message");
    },
    assertNoConnectionErrorWarning: function() {
        ok(this.getDetailDialog().content.find(".aui-message.warning:has(.connection-error)").length === 0, "There is NO connection error message");
    },
    assertNoErrorMessagesInCanvas: function() {
        this.assertNoNoPermissionMessage();
        this.assertNoConnectionErrorWarning();
    }
});

test("Test lock screen renders if no data would be displayed", function() {
    var applicationType = 'fecru';
    var data = {detail: [{projects: []}] };
    var view = this.createView(7, 7);

    var $fixture = AJS.$('#qunit-fixture');
    $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
    view.$el = $fixture;
    view.renderSuccess = this.sandbox.spy();
    view._handleFetchSucceeded(applicationType, data);

    ok(view.renderSuccess.neverCalledWith(applicationType, data.detail));
    var contentContainer = view.getContentContainer(applicationType);
    var noPermissionToViewAll = contentContainer.find(".no-permission-to-view-all");
    ok(noPermissionToViewAll.find(".lock-image").length > 0, "has lock image");
    equal(noPermissionToViewAll.text(), "You don't have access to view all related builds. Please contact your administrator.");
});

test("Test sorting of projects and plans", function() {
    var view = this.createView(7, 7);
    deepEqual(view.sortProjectsAndPlans([
        {name: 'z', plans: [
            {name: 'a', build: {finishedDate: '2013-10-22T12:00:00.000+0100'}},
            {name: 'c', build: {finishedDate: '2013-10-22T12:01:00.000+0100'}},
            {name: 'b', build: {finishedDate: '2013-10-22T12:02:00.000+0100'}}
        ]},
        {name: 'a', plans: [
            {name: 'a', build: {finishedDate: '2013-10-22T12:00:00.000+0100'}}
        ]},
        {name: 'M', plans: [
            {name: 'a', build: {finishedDate: '2013-10-22T12:01:00.000+0100'}},
            {name: 'c', build: {finishedDate: '2013-10-22T12:02:00.000+0100'}},
            {name: 'b', build: {finishedDate: '2013-10-22T12:03:00.000+0100'}}
        ]}
    ]), [
        {name: 'M', plans: [
            {name: 'b', build: {finishedDate: '2013-10-22T12:03:00.000+0100'}},
            {name: 'c', build: {finishedDate: '2013-10-22T12:02:00.000+0100'}},
            {name: 'a', build: {finishedDate: '2013-10-22T12:01:00.000+0100'}}
        ]},
        {name: 'z', plans: [
            {name: 'b', build: {finishedDate: '2013-10-22T12:02:00.000+0100'}},
            {name: 'c', build: {finishedDate: '2013-10-22T12:01:00.000+0100'}},
            {name: 'a', build: {finishedDate: '2013-10-22T12:00:00.000+0100'}}
        ]},
        {name: 'a', plans: [
            {name: 'a', build: {finishedDate: '2013-10-22T12:00:00.000+0100'}}
        ]}
    ]);
});

test("Test build count", function() {
    var view = this.createView();
    equal(view.getBuildCount([
    ]), 0, "no projects");
    equal(view.getBuildCount([
        {name: 'a', plans: [
            {key: 'A-A'}
        ]}
    ]), 1, "single project, single build");
    equal(view.getBuildCount([
        {name: 'z', plans: [
            {key: 'Z-A'},
            {key: 'Z-C'},
            {key: 'Z-B'}
        ]}
    ]), 3, "single project, multiple builds");
    equal(view.getBuildCount([
        {name: 'a', plans: [
            {key: 'A-A'}
        ]},
        {name: 'z', plans: [
            {key: 'Z-A'},
            {key: 'Z-C'},
            {key: 'Z-B'}
        ]}
    ]), 4, "multiple projects, multiple builds");
});

test("Test has artifacts", function() {
    var view = this.createView();
    ok(!view.hasArtifacts([]), "no projects");
    ok(!view.hasArtifacts([
         {name: 'a', plans: [
         ]}
    ]), "single project, no plans");
    ok(!view.hasArtifacts([
        {name: 'a', plans: [
            {key: 'Z-A'},
            {key: 'Z-C'}
        ]}
    ]), "single project, multiple plans, no build info");
    ok(!view.hasArtifacts([
        {name: 'a', plans: [
            {key: 'Z-A', build: {}},
            {key: 'Z-C', build: {}}
        ]}
    ]), "single project, multiple plans, no artifact info");
    ok(!view.hasArtifacts([
        {name: 'a', plans: [
            {key: 'Z-A', build: { artifacts: [] }},
            {key: 'Z-C', build: { artifacts: [] }}
        ]}
    ]), "single project, multiple plans, empty artifact info");
    ok(view.hasArtifacts([
        {name: 'a', plans: [
            {key: 'Z-A', build: { artifacts: [] }},
            {key: 'Z-C', build: { artifacts: ['one'] }}
        ]}
    ]), "single project, multiple plans, got artifact");
    ok(view.hasArtifacts([
        {name: 'a', plans: [
            {key: 'Z-A', build: { artifacts: [] }},
            {key: 'Z-C', build: { artifacts: [] }}
        ]},
        {name: 'b', plans: [
            {key: 'X-A', build: { artifacts: [] }},
            {key: 'X-C', build: { artifacts: ['one'] }}
        ]}
    ]), "multiple projects, multiple plans, got artifact");
});

test("Test dialog title for single build", function() {
    var view = this.createView(1, 1);

    view.show();
    var dialog = this.getDetailDialog();
    equal(dialog.heading.text(), "DEV-1: 1 build")
});

test("Test dialog title for multiple builds", function() {
    var view = this.createView(2, 1);

    view.show();
    var dialog = this.getDetailDialog();
    equal(dialog.heading.text(), "DEV-1: 2 builds")
});

test("Test build detail build dialog with success request", function() {
    expect(160);
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);

    view.show();
    var dialog = this.getDetailDialog();
    var spinner = dialog.content.find(".status-loading");
    var activePane = dialog.content.find(".tabs-pane.active-pane");
    var projects = [
        {
            name: "Bamboo",
            plans: [
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-19", result: "Passed in 2 hours", status: "SUCCESS", latestBuild: 5}
                }
            ]
        },
        {
            name: "Bamboo AAA",
            plans: [
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-19", result: "Passed in under 1 second", status: "SUCCESS", latestBuild: 5, fixesBuildNumber: 4}
                },
                {
                    disabled: false,
                    isBranch: true,
                    build: {date: "2013-11-17", result: "Failed in 1 second", status: "FAILED", latestBuild: 3}
                },
                {
                    disabled: true,
                    isBranch: false,
                    build: {date: "2013-11-15", result: "9 tests passed in 80 seconds", status: "SUCCESS", latestBuild: 3}
                }
            ]
        },
        {
            name: "Bamboo",
            plans: [
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-19", result: "Passed in 2 hours", status: "SUCCESS", latestBuild: 5}
                }
            ]
        },
        {
            name: "Bamboo 2",
            plans: [
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-13", result: "1 test failed in 27 hours", status: "FAILED", latestBuild: 5}
                },
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-09", result: "Build incomplete", status: "UNKNOWN", latestBuild: 88}
                },
                {
                    disabled: true,
                    isBranch: false,
                    build: {date: "2013-11-09", result: "1 test passed in 6 minutes", status: "SUCCESS", latestBuild: 3}
                }
            ]
        },
        {
            name: "Bamboo BBB",
            plans: [
                {
                    disabled: false,
                    isBranch: false,
                    build: {date: "2013-11-13", result: "1 test failed in 1 minute", status: "FAILED", latestBuild: 5}
                },
                {
                    disabled: false,
                    isBranch: true,
                    build: {date: "2013-11-11", result: "1 of 3 tests failed in 88 minutes", status: "FAILED", latestBuild: 3}
                },
                {
                    disabled: true,
                    isBranch: false,
                    build: {date: "2013-11-09", result: "1 test passed in 1 hour", status: "SUCCESS", latestBuild: 3}
                }
            ]
        }
    ];

    ok(spinner.is(":visible"), "A spinner is visible  when there is an onflight ajax request");
    ok(spinner.children().length > 0);
    ok(activePane.hasClass("loading"), "Dialog content is faded out when there is an onflight ajax request");

    AJS.TestContractManager.respondToRequest("detail3LO-bamboo", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("bambooDetailBuild-multiple", this.server.requests[1]);

    this.assertNoErrorMessagesInCanvas();

    var buildContainers = dialog.content.find(".detail-builds-container");

    ok(!spinner.is(":visible"), "A spinner is no longer visible when the onflight ajax request has completed");
    ok(!activePane.hasClass("loading"), "Dialog content is no longer faded out when ajax has completed");
    equal(buildContainers.length, projects.length, "Number of project tables rendered into the dialog is correct");

    _.each(buildContainers, _.bind(function(element, index) {
        var el = AJS.$(element);
        var messagePrefix = "Project[" + index + "]: ";

        var projectLink = el.find(".project-link");
        equal(projectLink.text(), projects[index].name, messagePrefix + "the project name exist and is correct");
        strictEqual(projectLink.attr("target"), undefined, messagePrefix + "clicking on project link opens in the same tab");
        projectLink.click();
        ok(JIRA.DevStatus.BuildsAnalytics.fireDetailProjectClicked.called,
           messagePrefix + "analytic event is fired correctly when clicking on project link");
        JIRA.DevStatus.BuildsAnalytics.fireDetailProjectClicked.reset();

        var buildList = el.find(".builds-table tr.plan-row");
        equal(buildList.length, projects[index].plans.length, messagePrefix + "list of builds exist and contains the correct number of rows");

        _.each(buildList, _.bind(function(element, buildRowIndex) {
            var buildRow = AJS.$(element);
            var messagePrefix = "Project[" + index + "], plan[" + buildRowIndex + "]: ";
            var expectedPlanData = projects[index].plans[buildRowIndex];
            var expectedBuildData = expectedPlanData.build;

            var plan = buildRow.find(".plan");
            var planLink = plan.find(".plan-link");
            strictEqual(planLink.attr("target"), undefined, messagePrefix + "clicking on plan link opens in the same tab");
            planLink.click();
            ok(JIRA.DevStatus.BuildsAnalytics.fireDetailPlanClicked.called,
               messagePrefix + "analytic event is fired correctly when clicking on plan link");
            JIRA.DevStatus.BuildsAnalytics.fireDetailPlanClicked.reset();

            equal(plan.find(".aui-lozenge.disabled").length, expectedPlanData.disabled ? 1 : 0, "disabled lozenge present or not");
            equal(plan.find(".aui-icon.aui-iconfont-devtools-branch-small").length, expectedPlanData.isBranch ? 1 : 0, "branch icon present or not");

            var latestBuild = buildRow.find(".build");

            var latestBuildIcon = latestBuild.find(".aui-icon");
            if (expectedBuildData.status == "SUCCESS") {
                ok(latestBuildIcon.hasClass("aui-iconfont-approve"), messagePrefix + "has correct icon");
            } else if (expectedBuildData.status == "FAILED") {
                ok(latestBuildIcon.hasClass("aui-iconfont-error"), messagePrefix + "has correct icon");
            } else if (expectedBuildData.status == "UNKNOWN") {
                ok(latestBuildIcon.hasClass("aui-iconfont-devtools-task-cancelled"), messagePrefix + "has correct icon");
            }

            var latestBuildLink = latestBuild.find(".build-link");
            equal(latestBuildLink.text(), "#" + expectedBuildData.latestBuild, messagePrefix + "latest build number is correct");
            strictEqual(latestBuildLink.attr("target"), undefined, messagePrefix + "clicking on build link opens in the same tab");
            latestBuildLink.click();
            ok(JIRA.DevStatus.BuildsAnalytics.fireDetailBuildClicked.called,
               messagePrefix + "analytic event is fired correctly when clicking on build link");
            JIRA.DevStatus.BuildsAnalytics.fireDetailBuildClicked.reset();

            var buildResult = buildRow.find(".result");
            equal(buildResult.text(), expectedBuildData.result, messagePrefix + "the build result exist and is correct");

            var fixesBuildLink = buildResult.find("a");
            equal(fixesBuildLink.length, 0, "fixes build link not present at all times");

            var timestamp = buildRow.find(".timestamp .livestamp");
            ok(timestamp.length != 0, messagePrefix + "time stamp is rendered into the table");
            equal(timestamp.attr("title"), expectedBuildData.date, messagePrefix + "tooltip is correctly appended onto the timestamp element");
        }, this));

    }, this));
});