AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-review-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-contract-test-resource");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-and-dialog-resources");

module("JIRA.DevStatus.DetailDialogReviewView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "DEV-1";
        this.issueId = 10000;
        this.server = sinon.fakeServer.create();
        this.templateSpy = undefined;

        JIRA.DevStatus.QunitTestUtils.spyAnalyticMethods("Reviews", this.sandbox);
        JIRA.DevStatus.ReviewsAnalytics.fireDetailReviewClicked = sinon.spy();
    },
    teardown : function () {
        this.sandbox.restore();
        this.server.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
        //JIRA form dialog appends overflow: hidden to the body of the page. This is to remove it.
        AJS.$("body").css("overflow", "initial");
    },
    getDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            heading: dialog.find(".devstatus-dialog-heading"),
            content: dialog.find(".devstatus-dialog-content"),

            getReviewContainer: function() {
                return this.content.find(".detail-reviews-container");
            },
            getRows: function() {
                return this.getReviewContainer().find("tbody tr");
            },
            getColumns: function(row) {
                var $row = AJS.$(row);
                return {
                    review: $row.find(".review"),
                    title: $row.find('.title'),
                    state: $row.find(".state"),
                    author: $row.find(".author"),
                    reviewers: $row.find('.reviewer-container .vcs-user'),
                    extraReviewersLozenge: $row.find('.reviewer-container .extrareviewers-tooltip'),
                    reviewerAvatarCount: $row.find('.reviewer img').length,
                    truncatedReviewerCount: parseInt($row.find('.reviewer').text()),
                    commentCount: parseInt($row.find(".comment .count").text()),
                    timestamp: $row.find(".timestamp"),
                    overDue: $row.find('.timestamp .overdue').length > 0
                }
            },
            getNoPermissionToViewAllMessage: function() {
                return this.content.find(".no-permission-to-view-all");
            },
            getExtraReviewerInlineDialog: function(content) {
                return {
                    rowsElement: content.find("tr"),
                    rowsValue: function() {
                        var values = [];
                        _.each(this.rowsElement, function(row) {
                            var $row = AJS.$(row);
                            values.push({
                                name: $row.find(".reviewer-name").text(),
                                hasAvatar: $row.find(".reviewer-avatar img").length > 0
                            });
                        });
                        return values;
                    }

                }
            }
        }
    },
    createView: function(count) {
        var view = new JIRA.DevStatus.DetailDialogReviewView({
            count: count,
            issueKey: this.issueKey,
            issueId: this.issueId,
            tabs: {
                fecru: {name:"FishEye / Crucible", count: count}
            },
            dataType: "review",
            reviewersThreshold: 2
        });
        this.templateSpy = sinon.spy(view, 'template');
        return view;
    },
    assertNoNoPermissionMessage: function() {
        ok(this.getDetailDialog().content.find(".no-permission-to-view-all").length === 0, "There is NO no-permission-to-view message");
    },
    assertNoConnectionErrorWarning: function() {
        ok(this.getDetailDialog().content.find(".aui-message.warning:has(.connection-error)").length === 0, "There is NO connection error message");
    },
    assertNoErrorMessagesInCanvas: function() {
        this.assertNoNoPermissionMessage();
        this.assertNoConnectionErrorWarning();
    },
    assertColumnContent: function (dialog, row, reviewId, title, state, author, reviewerCount, extraReviewerCount, commentCount, timestamp) {
        var columns = dialog.getColumns(dialog.getRows()[ row]);
        equal(columns.review.text(), reviewId, 'Review id rendered for ' + reviewId);
        ok(columns.title.text().match(title), 'Review title rendered for ' + reviewId);
        equal(columns.state.text(), state, 'Review status rendered for ' + reviewId);
        // NOTE: title has been renamed by the tipsy to "original-title".
        equal(columns.author.find('.extra-content-in-title').attr('original-title'), author, 'Author rendered for ' + reviewId);
        ok(columns.author.find('img').length > 0, 'Avatar rendered for ' + reviewId);
        equal(columns.reviewerAvatarCount, reviewerCount, 'Review avatars rendered for  ' + reviewId);
        deepEqual(columns.truncatedReviewerCount, extraReviewerCount, 'Extra reviewer badge rendered ' + reviewId);
        deepEqual(columns.commentCount, commentCount, 'Comment count rendered for ' + reviewId);
        ok(columns.timestamp.text().match(timestamp), 'Timestamp rendered for ' + reviewId);
        return columns;
    }
});

test("Test lock screen renders if no data would be displayed", function() {
    var applicationType = 'fecru';
    var data = {detail: [{reviews: []}] };
    var view = this.createView();

    var $fixture = AJS.$('#qunit-fixture');
    $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
    view.$el = $fixture;
    view.renderSuccess = this.sandbox.spy();
    view._handleFetchSucceeded(applicationType, data);

    ok(view.renderSuccess.neverCalledWith(applicationType, data.detail));
    var contentContainer = view.getContentContainer(applicationType);
    var noPermissionToViewAll = contentContainer.find(".no-permission-to-view-all");
    ok(noPermissionToViewAll.find(".lock-image").length > 0, "has lock image");
    equal(noPermissionToViewAll.text(), "You don't have access to view all related reviews. Please contact your administrator.");
});

test("Test content table is not rendered when JSON response empty", function() {
    var view = this.createView(1);
    var jiraDialog = AJS.$('<div class="jira-dialog"></div>').appendTo(AJS.$("#qunit-fixture"));
    var container = AJS.$('<div class="devstatus-dialog-content"></div>').appendTo(jiraDialog);
    view.getContentContainer = this.sandbox.stub().returns(container);
    view.renderSuccess("fecru", []);

    var dialog = this.getDetailDialog();
    equal(dialog.getReviewContainer().length, 0, "Dialog container doesn't exists");
    equal(dialog.getRows().length, 0, "Table/rows are rendered into the dialog");
    equal(dialog.getNoPermissionToViewAllMessage().length, 1, "Contains no permission to view all message");
});

test("Test responses from multiple connected products are combined", function() {
    // having
    var view = this.createView(),
        reviews1 = [
            {id: 'CR-1', lastModified: '2013-11-20T11:49:10.512+0100', state: 'Review'},
            {id: 'CR-2', lastModified: '2013-11-20T23:59:10.512+0100', state: 'Review'}
        ],
        reviews2 = [
            {id: 'CR-3', lastModified: '2013-10-20T11:49:10.512+0100', state: 'Review'},
            {id: 'CR-4', lastModified: '2013-12-20T11:49:10.512+0100', state: 'Review'}
        ];

    // when
    view.renderSuccess('fecru', [{ reviews: reviews1 }, { reviews: reviews2 } ]);

    // then
    deepEqual(this.templateSpy.getCall(0).args[0].reviews, [reviews2[1], reviews1[1], reviews1[0], reviews2[0]], 'Reviews combined before template call');
});

test("Test sorting reviews by lastModified", function () {
    // having
    var view = this.createView(),
        reviews = [
        {id: 'CR-1', lastModified: '2013-11-20T11:49:10.512+0100', state: 'Review'},
        {id: 'CR-2', lastModified: '2013-11-20T23:59:10.512+0100', state: 'Review'},
        {id: 'CR-3', lastModified: '2013-10-20T11:49:10.512+0100', state: 'Review'},
        {id: 'CR-4', lastModified: '2013-12-20T11:49:10.512+0100', state: 'Review'},
        {id: 'CR-5', lastModified: '2013-11-20T11:49:10.512+0100', dueDate: '2014-12-20T11:49:10.512+0100', state: 'Review'},
        {id: 'CR-6', lastModified: '2013-11-20T11:49:10.512+0100', dueDate: '2012-12-20T11:49:10.512+0100', state: 'Review'}
    ];

    // when
    view.renderSuccess('fecru', [{ reviews: reviews }]);

    // then
    deepEqual(this.templateSpy.getCall(0).args[0].reviews, [reviews[5], reviews[4], reviews[3], reviews[1], reviews[0], reviews[2]], 'Reviews sorted before template call');
});

test("Test complete and rejected reviews don't have due dates", function() {
    // having
    var view = this.createView(),
        reviewBase = {id: 'CR-1', lastModified: '2013-11-20T11:49:10.512+0100', dueDate: '4999-11-20T11:49:10.512+0100', reviewers: []},
        inReview = _.extend({state: 'Review'}, reviewBase),
        closed = _.extend({state: 'Closed'}, reviewBase),
        rejected = _.extend({state: 'Rejected'}, reviewBase);

    // when
    view.renderSuccess('fecru', [{reviews: [inReview]}]);
    view.renderSuccess('fecru', [{reviews: [closed]}]);
    view.renderSuccess('fecru', [{reviews: [rejected]}]);

    // then
    deepEqual(this.templateSpy.getCall(0).args[0].reviews, [inReview], 'Review in progress due date not stripped');
    deepEqual(this.templateSpy.getCall(1).args[0].reviews, [_.omit(closed, 'dueDate')], 'Review closed due date stripped');
    deepEqual(this.templateSpy.getCall(2).args[0].reviews, [_.omit(rejected, 'dueDate')], 'Review rejected due date stripped');
});

test("Test sorting reviewers by completion status", function () {
    // having
    var view = this.createView(),
        reviewers = [
            {name: 'd', complete: true},
            {name: 'c', complete: false},
            {name: 'b', complete: true},
            {name: 'a', complete: false}
        ],
        reviews = [
        {
            id: 'CR-1',
            state: 'Review',
            reviewers: reviewers
        }
    ];

    // when
    var sorted = _(reviews).map(view._updateReviewers)[0].reviewers;

    // then
    deepEqual(sorted, [reviewers[2], reviewers[0], reviewers[3], reviewers[1]], 'Reviewers sorted before template call');
});

test("Test author is excluded from reviewers list", function () {
    var view = this.createView(),
        reviewers = [
            {name: 'amy'},
            {name: 'bernice'}
        ],
        reviews = [
            {
                id: 'CR-1',
                state: 'Review',
                reviewers: reviewers,
                author: {name: 'amy', avatar: 'http://foo/bar', emailAddress: 'amy@example.com'}
            }
        ];

    var actual = _(reviews).map(view._updateReviewers)[0].reviewers;
    ok(_.contains(actual, {name: 'amy'}) === false, 'Author should not be in the reviewers list');
});

test("Test transforming reviewers", function() {
    var view = this.createView(),
        moderator = {name: 'c'},
        reviewers = [
            {name: 'b', complete: false},
            {name: 'a', complete: true}
        ],
        reviews = [
            {
                id: 'CR-1',
                state: 'Review',
                reviewers: reviewers,
                moderator: moderator
            }
        ];

    // when
    var sorted = _(reviews).map(view._updateReviewers)[0].reviewers;

    // then
    deepEqual(sorted, [moderator, reviewers[1], reviewers[0]], 'Moderator inserted into reviewers');
});

test("Test transforming reviewers doesn't add moderator if he's the author", function() {
    var view = this.createView(),
        moderator = {name: 'a', avatar: 'http://foo/bar', emailAddress: 'moder@tor.com'},
        reviewers = [
            {name: 'b', complete: false}
        ],
        reviews = [
            {
                id: 'CR-1',
                state: 'Review',
                reviewers: reviewers,
                moderator: moderator,
                author: moderator
            }
        ];

    // when
    var sorted = _(reviews).map(view._updateReviewers)[0].reviewers;

    // then
    deepEqual(sorted, [reviewers[0]], "Moderator not inserted into reviewers if he's a reviewer as well");
});

test("Test filtering reviews", function() {
    // having
    var view = this.createView(),
        reviews = [
            {id: 'CR-1', state: 'Draft'},
            {id: 'CR-2', state: 'Review'},
            {id: 'CR-3', state: 'Approval'},
            {id: 'CR-4', state: 'Closed'},
            {id: 'CR-5', state: 'Dead'},
            {id: 'CR-6', state: 'Unknown'},
            {id: 'CR-6', state: 'Summarize'},
            {id: 'CR-7', state: 'Rejected'}

        ];

    // when
    view.renderSuccess('fecru', [{ reviews: reviews }]);

    // then
    deepEqual(this.templateSpy.getCall(0).args[0].reviews, [reviews[1], reviews[2], reviews[3], reviews[6], reviews[7]], 'Reviews filtered before template call');
});

test("Test event sent when review id clicked", function() {
    var view = this.createView(0);
    view.show();
    view.renderSuccess('fecru', [
        { reviews: [
            {id: 'CR-2', state: 'Review', commentCount: 666}
        ] }
    ]);

    AJS.$('.review-link').click();
    ok(JIRA.DevStatus.ReviewsAnalytics.fireDetailReviewClicked.calledOnce, 'analytics event fired when clicking review id');

    AJS.$('.comment-link').click();
    ok(JIRA.DevStatus.ReviewsAnalytics.fireDetailReviewClicked.calledTwice, 'analytics event fired when clicking comments link');
});

test("Test review detail renders dialog with success request", function() {
    expect(58);
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);
    view.show();

    var dialog = this.getDetailDialog();
    var spinner = dialog.content.find(".status-loading");
    var activePane = dialog.content.find(".tabs-pane.active-pane");

    ok(spinner.is(":visible"), "A spinner is visible  when there is an onflight ajax request");
    ok(spinner.children().length > 0);
    ok(activePane.hasClass("loading"), "Dialog content is faded out when there is an onflight ajax request");

    AJS.TestContractManager.respondToRequest("detail3LO-fecru", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailReviews-multiple", this.server.requests[1]);

    this.assertNoErrorMessagesInCanvas();

    ok(!spinner.is(":visible"), "A spinner is no longer visible when the onflight ajax request has completed");
    ok(!activePane.hasClass("loading"), "Dialog content is no longer faded out when ajax has completed");
    equal(dialog.getRows().length, 5, "Number of pull request rows rendered into the table is correct");

    var reviewPerm7 = this.assertColumnContent(dialog, 0, 'PERM-7', 'patch #2815684', 'Review', 'System Administrator', 3, NaN, 1, /^Due/);
    ok(reviewPerm7.reviewers.eq(0).find('.approved').length > 0, 'Review completion mark rendered correctly');
    ok(reviewPerm7.reviewers.eq(0).find('.fade').length > 0, 'Review completion fade rendered correctly');
    ok(reviewPerm7.overDue, 'PERM-7 overdue');

    this.assertColumnContent(dialog, 1, 'CR-244', 'DEF-1 Row row row your boat', 'Review', 'Geoff Crain', 2, 7, 2, /^Due/);

    this.assertColumnContent(dialog, 2, 'CR-256', 'DEF-1: 1', 'Approval', 'Matt Quail', 2, NaN, NaN, /^Due/);

    var reviewDef10 = this.assertColumnContent(dialog, 3, 'DEF-10', 'DEF-1: 1', 'Closed', 'Matt Quail', 2, 2, NaN, /^(?!Due).*$/);
    ok(!reviewDef10.overDue, 'DEF-10 not overdue');

    var reviewDef4 = this.assertColumnContent(dialog, 4, 'DEF-4', '.*', 'Closed', 'Geoff Crain', 0, NaN, 101, /^(?!Due)/);
    ok(!reviewDef4.overDue, 'DEF-4 not overdue');
});

test("Test reviewer tooltip in detail dialog", function() {
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);
    view.show();

    var dialog = this.getDetailDialog();

    AJS.TestContractManager.respondToRequest("detail3LO-fecru", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("detailReviews-multiple", this.server.requests[1]);

    var rowWithRewiewers = 1;

    //Clicking on number lozenge on the reviewer column brings up the inline dialog
    ok(!_.isObject(view.reviewerInlineDialogView.activeExtraReviewersToolTip), "activeExtraReviewersToolTip is NOT created when the pull request link is NOT hovered");
    dialog.getColumns(dialog.getRows()[rowWithRewiewers]).extraReviewersLozenge.mouseenter().mousemove();
    ok(_.isObject(view.reviewerInlineDialogView.activeExtraReviewersToolTip), "activeExtraReviewersToolTip is created when the pull request link is hovered");

    var reviewInlineDialog = dialog.getExtraReviewerInlineDialog(view.reviewerInlineDialogView.activeExtraReviewersToolTip.$content);
    equal(reviewInlineDialog.rowsElement.length, 7, "number of reviewers in tooltip is correct");
    deepEqual(
        _.pluck(reviewInlineDialog.rowsValue(), "name"),
        ["Brendan Humphreys", "Conor MacNeill", "George Filippoff", "Matt Quail", "Peter McNeil", "Wojciech Seliga", "joe lowercase"],
        "tooltip content is correct");

    dialog.getColumns(dialog.getRows()[rowWithRewiewers]).extraReviewersLozenge.mouseleave();
    ok(!view.reviewerInlineDialogView.activeExtraReviewersToolTip.visible, "activeExtraReviewersToolTip is hidden when mouse is removed");
});

test("Test extra reviewers are spliced after sorting", function() {
    // having
    var view = this.createView(),
        reviews = [{reviews: [
            {
                id: 'CR-1',
                state: 'REVIEW',
                reviewers: [
                    {name: 'a', complete: false},
                    {name: 'b', complete: false},
                    {name: 'c', complete: true},
                    {name: 'd', complete: true},
                ],
                moderator: {
                    name: 'm'
                }}
        ]}];

    // when
    var prepared = view._prepareReviews(reviews);
    var extraReviewers = JSON.parse(prepared[0].extraReviewers);
    var reviewers = prepared[0].reviewers;

    // then
    deepEqual(_.pluck(reviewers, 'name'), ['Moderator: m', 'c', 'd', 'a', 'b'], "reviewers should be sorted: moderator first, then completed then notcompleted");
    deepEqual(_.pluck(extraReviewers, 'name'), ['d', 'a', 'b'], "extraReviewers should be the tail of the all-reviewer list");
});