Backbone.define("JIRA.DevStatus.DetailDialogCommitView", JIRA.DevStatus.BaseDetailDialogView.extend({
    template: JIRA.Templates.DevStatus.DetailDialog.commit,

    events: {
        "click a.repository-link": "_onClickRepositoryLink",
        "click a.changesetid": "_onClickChangesetId",
        "click .filecount a": "_onClickFiles",
        "click .file-expand a": "_onClickExpand",
        "click .create-review-commit a": "_onClickCreateReview",
        "click .filename a": "_onClickFile",
        "mouseenter .reviews-link": "_onHoverReviewsToolTip",
        "click": "_onClickReviewsToolTip"
    },

    initialize: function(options) {
        this.analytics = JIRA.DevStatus.CommitsAnalytics;
        this.issueId = options.issueId;

        var dialogToOpen = AJS.Meta.get("fusion-open-detail-dialog");
        var initialTab;
        if (JIRA.DevStatus.URL.isCreateReviewDetailDialogLink(dialogToOpen)) {
            initialTab = JIRA.DevStatus.URL.getCreateReviewDetailDialogApplicationType(dialogToOpen);
        }

        JIRA.DevStatus.BaseDetailDialogView.prototype.initialize.call(this, _.extend({
            type: "commit",
            width: 1000,
            height: 400,
            initialTab: initialTab
        }, options));
    },

    concatResults: function (result) {
        return _.reduce(result, function (memo, data) {
            var repositories = data.repositories || [];
            _.each(repositories, function (repositories) {
                repositories.showInstance = data._instance && !data._instance.singleInstance;
                repositories.instance = data._instance;
            });
            return memo.concat(repositories);
        }, []);
    },

    _clipEllipsis: function() {
        function getWidth(el) {
            return el.outerWidth();
        }

        this.$el.find(".detail-content-container .file .ellipsis").each(function(i, content) {
            var elem = AJS.$(content);
            var width = getWidth(elem);
            if (width == 0) {
                return;
            }
            var link = elem.find("a");

            if (getWidth(link) > width) {
                var slicedText = link.text();
                while (getWidth(link) > width)
                {
                    slicedText = slicedText.slice(1);
                    link.text('\u2026' + slicedText);
                }
            }
        });
    },

    renderSuccess: function(applicationType, result) {
        var repositories = this.concatResults(result);
        var container = this.getContentContainer(applicationType);
        this.transformedRepositories = this.transformRepositories(repositories);
        var canCreateReview = this.setupCreateReviewDialog(this.transformedRepositories);

        container.html(this.template({
            applicationType: applicationType,
            repositories: this.transformedRepositories,
            canCreateReview: canCreateReview,
            livestampRelativizeThreshold: this.options.livestampRelativizeThreshold
        }));

        this.renderNoPermissionToViewAllWarningAtBottom(applicationType, this.getUniqueCommitCount(repositories));
        JIRA.DevStatus.Date.addTooltip(container);
        this.setupBranchesTooltip();
        this.createReviewDialog && this.createReviewDialog.renderSuccess();

        return this;
    },

    setupCreateReviewDialog: function (repositories) {
        var instances = _.chain(repositories)
                         .filter(function(item)
                            {
                                return item.showCreateReview;
                            })
                         .pluck('instance')
                         .value();

        var canCreate = false;

        if (instances.length) {
            this.createReviewDialog = new JIRA.DevStatus.CreateReviewFormDialog({
                analytics: this.analytics,
                applicationType: this._getActiveApplicationType(),
                issueId: this.issueId,
                instances: instances
            });
            canCreate = true;
        }

        var dialogToOpen = AJS.Meta.get("fusion-open-detail-dialog");
        if (JIRA.DevStatus.URL.isCreateReviewDetailDialogLink(dialogToOpen) &&
            JIRA.DevStatus.URL.getCreateReviewDetailDialogApplicationType(dialogToOpen) === this._getActiveApplicationType()) {
            AJS.Meta.set("fusion-open-detail-dialog", undefined);
            if (canCreate) {
                this.createReviewDialog.dialog.show();
            }
        }

        return canCreate;
    },

    _onHoverReviewsToolTip: function(e) {
        var self = this;
        e.preventDefault();

        var target = AJS.$(e.target);
        var id = "commit-reviews-popup";
        this.activeReviewsToolTip = AJS.InlineDialog(target, id,
            function(content, trigger, showPopup) {
                var reviewData = self._getCommitDetails(AJS.$(trigger)).reviews;
                var $trigger = AJS.$(trigger);
                JIRA.DevStatus.CommitsAnalytics.fireDetailReviewsShown(self._getActiveApplicationType());
                content.html(JIRA.Templates.DevStatus.DetailDialog.reviewsInlineDialog({
                    reviewData: reviewData
                }));
                content.find('a').click(function(e) {
                    self._onClickReview(e);
                });

                showPopup();
            }, {
                cacheContent: false,
                hideDelay: 200,
                showDelay: 50,
                onHover: true,
                width: false //Stop AUI from hardcoding inline dialog width
            }
        );
    },

    _onClickReviewsToolTip: function(e) {
        if (this.activeReviewsToolTip) {
            e.stopPropagation();
        }
    },


    setupBranchesTooltip: function() {
        var self = this;
        AJS.$('.branches-link').tooltip({
                title: function() {
                    var branchData = self._getCommitDetails(AJS.$(this)).branches;
                    JIRA.DevStatus.CommitsAnalytics.fireDetailBranchesShown(self._getActiveApplicationType());
                    return JIRA.Templates.DevStatus.DetailDialog.branchesTooltip({
                        branchData: branchData
                    }) },
                html: true
            }
        );
    },

    getTitle: function(count, issue) {
        var totalCount = _.chain(this.model.get("tabs"))
            .map(function(tab) {return tab.count;})
            .reduce(function(memo, num) {return memo + num;}, 0)
            .value();
        var duplicatedCount = Math.max(totalCount - count, 0);
        return AJS.$.trim(AJS.I18n.getText("devstatus.detail.panel.title.commit", count, issue, duplicatedCount));
    },

    getOAuthMessageInFooter: function(instances) {
        return AJS.I18n.getText('devstatus.authentication.message.commit', instances.length);
    },

    getOAuthMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.authenticate.commit.title');
    },

    getConnectionMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.connection-problem.commit.title');
    },

    getNoPermissionToViewAllMessageInCanvas: function() {
        return AJS.I18n.getText('devstatus.authentication.no-permission-to-view-all.commit.title');
    },

    hasData: function(detail) {
        return this.concatResults(detail).length > 0;
    },

    /**
     * Prepares repositories json data to be rendered.
     */
    transformRepositories: function(repositories) {
        repositories = this.sortRepositoriesAndCommits(repositories);
        var self = this;
        var optFields = [['reviews', 'showReviews'], ['branches', 'showBranches'], ['files', 'showFiles'], ['createReviewUrl', 'showCreateReview']];
        _.each(optFields, function(args) {
            repositories = self._setRepositoryFromFromCommitField(repositories, args[0], args[1]);
        });
        repositories = this.prepareReviews(repositories);
        return repositories;
    },

    _setRepositoryFromFromCommitField: function(repositories, field, flag) {
        _.each(repositories, function (repository) {
            repository[flag] = _.find(repository.commits, function (commit) {
                return commit[field] != null;
            }) != null;
        });
        return repositories;
    },

    /**
     * Checks if the repositories have review info, and orders the reviews by status
     */
    prepareReviews: function (repositories) {
        var self = this;
        _.each(repositories, function (repository) {
            _.each(repository.commits, self._orderCommitReviews);
        });
        return repositories;
    },

    /**
     * Orders the reviews in the commit
     */
    _orderCommitReviews: function (commit) {
        if (commit.reviews && commit.reviews.reviews) {
            var reviewPriorities = ['REVIEW', 'APPROVAL', 'SUMMARIZE', 'REJECTED', "CLOSED"];
            commit.reviews.reviews = _.chain(commit.reviews.reviews)
                .filter(function (review) {
                    return _.contains(reviewPriorities, review.state);
                })
                .sortBy(function (review) {
                    return _.indexOf(reviewPriorities, review.state)
                })
                .value();
        }
    },

    /**
     * Sort repositories alphabetically (case insensitive) and commits from newest to oldest.
     */
    sortRepositoriesAndCommits: function(repositories) {
        repositories = _.sortBy(repositories, function(repo) {
            return repo.name.toLowerCase();
        });
        _.each(repositories, function(repository) {
            // Dates should _always_ be in ISO format - so ASCII sorting will work
            repository.commits = _.sortBy(repository.commits, 'authorTimestamp').reverse();

        });
        return repositories;
    },

    /**
     * Calculate the total number of unique commits in all repos.
     */
    getUniqueCommitCount: function (repositories) {
        return _.union.apply(null, _.map(repositories, function (repo) {
            // project non-null commit ids
            return _.filter(_.pluck(repo.commits, 'id'), function(commitId) { return commitId;});
        })).length;
    },

    /**
     * Gets the repository object corresponding to the passed dom element
     */
    _getRepoDetails: function ($cell) {
        return this.transformedRepositories[parseInt($cell.parents('.detail-commits-container').data('repository-index'))];
    },

    /**
     * Gets the commit object corresponding to the passed dom element
     */
    _getCommitDetails: function ($cell) {
        return this._getRepoDetails($cell).commits[parseInt($cell.parents('tr').data('commit-index'))];
    },

    _onClickRepositoryLink: function() {
        JIRA.DevStatus.CommitsAnalytics.fireDetailRepoClicked(this._getActiveApplicationType());
    },

    _onClickChangesetId: function() {
        JIRA.DevStatus.CommitsAnalytics.fireDetailCommitClicked(this._getActiveApplicationType());
    },

    _onClickExpand: function(e) {
        e.preventDefault();
        var expand = this._getFileExpand(AJS.$(e.target));
        if (expand.toggle(expand.getRows())) {
            JIRA.DevStatus.CommitsAnalytics.fireDetailFilesExpandedClicked(this._getActiveApplicationType());
        }
        this._clipEllipsis();
    },

    _onClickFiles: function(e) {
        e.preventDefault();
        var expanded = this._getFileExpand(AJS.$(e.target)).toggle(
            // Go up to find the parent commit row and move across to the corresponding file row
            AJS.$(e.target).closest('.commitrow').next('.filerow')
        );
        this._clipEllipsis();

        if (expanded) {
            JIRA.DevStatus.CommitsAnalytics.fireDetailFileExpandedClicked(this._getActiveApplicationType());
        }
    },

    _onClickCreateReview: function(e) {
        JIRA.DevStatus.CommitsAnalytics.fireDetailCreateReviewClicked(this._getActiveApplicationType(), false);
    },

    _onClickReview: function(e) {
        JIRA.DevStatus.CommitsAnalytics.fireDetailReviewClicked(this._getActiveApplicationType());
    },

    _onClickFile: function(e) {
        JIRA.DevStatus.CommitsAnalytics.fireDetailFileClicked(this._getActiveApplicationType());
    },

    /**
     * Encapsulates the logic for toggling a single file now, or all of them, and updating the 'show all' label with
     * the appropriate text.
     *
     * @param $target this should be from the current event target so the per-repository container can be used
     * @returns {{getRows: getRows, toggle: toggle}}
     * @private
     */
    _getFileExpand: function($target) {
        // This is the containing element for a single repository, which each have their own expanding link
        var $container = $target.closest('.detail-commits-container');
        // Returns whether there are non-hidden elements in a given set of jQuery elements
        function hasVisibleRows($el) {
            return $el.length > $el.filter('.hidden').length
        }
        return {

            /**
             * @returns {*} all of the file rows for this repository
             */
            getRows: function() {
                return $container.find('.filerow');
            },

            /**
             * @param $el a list of file row elements to toggle based on their collective visible states.
             *            If any are visible then hide them, otherwise show them _all_
             * @returns {boolean} true if the rows were made visible, false otherwise
             */
            toggle: function($el) {
                var hide = hasVisibleRows($el);
                $el.toggleClass('hidden', hide);
                // Update the expand label now that we have toggled the file rows
                this._updateExpandLabel();
                return !hide;
            },

            /**
             * Updates the 'show all' label based on the current state of the file rows.
             * If _any_ of them are visible then (continue to) show 'hide', otherwise revert back to 'show all'.
             * @private
             */
            _updateExpandLabel: function() {
                $container.find('.file-expand a').text(hasVisibleRows(this.getRows())
                    ? AJS.I18n.getText('devstatus.detail.panel.commit.files.expand.hide')
                    : AJS.I18n.getText('devstatus.detail.panel.commit.files.expand.show')
                );
            }
        }
    }
}));
