AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-review-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-contract-test-resource");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");

module("JIRA.DevStatus.DetailDialogCommitView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "DEV-1";
        this.issueId = 123;
        this.server = sinon.fakeServer.create();

        this.repoClickedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailRepoClicked");
        this.commitClickedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailCommitClicked");
        this.branchesShownStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailBranchesShown");
        this.reviewsShownStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailReviewsShown");
        this.reviewClickedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailReviewClicked");
        this.fileExpandedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailFileExpandedClicked");
        this.filesExpandedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailFilesExpandedClicked");
        this.fileClickedStub = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, "fireDetailFileClicked");
        this.createReviewClicked = this.sandbox.stub(JIRA.DevStatus.CommitsAnalytics, 'fireDetailCreateReviewClicked');
    },
    teardown : function () {
        this.sandbox.restore();
        this.server.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
        //JIRA form dialog appends overflow: hidden to the body of the page. This is to remove it.
        AJS.$("body").css("overflow", "initial");
        AJS.Meta.set("fusion-open-detail-dialog", undefined);
    },
    getCommitDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            heading: dialog.find(".devstatus-dialog-heading"),
            content: dialog.find(".devstatus-dialog-content")
        }
    },
    getCommitRows: function($el) {
        return $el.find('tbody tr.commitrow');
    },
    createView: function(count, fixture) {
        return new JIRA.DevStatus.DetailDialogCommitView({
            count: 10,
            el: fixture,
            issueId: this.issueId,
            issueKey: this.issueKey,
            tabs: {
                bitbucket: {name:"Bitbucket", count: count || 9}
            }
        });
    },
    assertNoNoPermissionMessage: function() {
        ok(this.getCommitDetailDialog().content.find(".no-permission-to-view-all").length === 0, "There is NO no-permission-to-view message");
    },
    assertNoConnectionErrorWarning: function() {
        ok(this.getCommitDetailDialog().content.find(".aui-message.warning:has(.connection-error)").length === 0, "There is NO connection error message");
    },
    assertNoErrorMessagesInCanvas: function() {
        this.assertNoNoPermissionMessage();
        this.assertNoConnectionErrorWarning();
    },
    renderViewFromContracts: function(contracts, count) {
        var view = this.createView(count || 5);
        view.show();
        for (var i = 0; i < contracts.length; i++) {
            AJS.TestContractManager.respondToRequest(contracts[i], this.server.requests[i]);
        }
    },
    renderFromData: function (result, applicationType) {
        var applicationType = applicationType || 'fecru';
        var $fixture = AJS.$('#qunit-fixture');
        $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
        var view = this.createView(1, $fixture);
        view.renderSuccess(applicationType, result);
        return view;
    }
});

test("Test bitbucket detail commits repositories and commits are ordered", function() {
    var view = this.createView();
    deepEqual(view.sortRepositoriesAndCommits([
        {name: 'z', commits: []},
        {name: 'a', commits: []},
        {name: 'M', commits: [
            {authorTimestamp: '2013-10-22T15:33:50.000+1100'},
            {authorTimestamp: '2011-10-22T15:33:50.000+1100'},
            {authorTimestamp: '2012-10-22T15:33:50.000+1100'}
        ]}
    ]), [
        {name: 'a', commits: []},
        {name: 'M', commits: [
            {authorTimestamp: '2013-10-22T15:33:50.000+1100'},
            {authorTimestamp: '2012-10-22T15:33:50.000+1100'},
            {authorTimestamp: '2011-10-22T15:33:50.000+1100'}
        ]},
        {name: 'z', commits: []}
    ]);
});

test("Test commit detail dialog renders reviews and branches", function () {
    // having
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewsAndBranches"]);

    // when
    var $rows = this.getCommitRows(AJS.$('.devstatus-detail-table'));
    var $row0 = $rows.eq(0); // 8f70c6e
    var $row1 = $rows.eq(1); // 9dba66c
    var $row2 = $rows.eq(2); // 155a0e9
    var $row0branches = $row0.find('.branches-link');
    var $row1branches = $row1.find('.branches-link');
    var $row2reviews = $row2.find('.reviews-link');

    // then
    equal($row0branches.find('.aui-iconfont-devtools-branch-small').length, 1, "Branch icon rendered");
    equal($row0branches.find('.branch-count').length, 0, "Branch count not rendered");
    equal($row0.find('.reviews-link').length, 0, "Review lozenge not rendered");

    equal($row1branches.find('.aui-iconfont-devtools-branch-small').length, 1, "Branch icon rendered");
    equal($row1branches.find('.branch-count').text(), "21", "Branch count rendered");

    equal($row2reviews.length, 1, "Review link rendered");
    equal($row2reviews.text(), "Review", "Review link status rendered");

});

test("Test commit detail dialog renders instance names", function () {
    // having
    var view = this.renderFromData(
        [
            {
                repositories: [
                    {name: 'instance1-repo1'},
                    {name: 'instance1-repo2'}
                ],
                _instance: {
                    name: 'instance1'
                }
            },
            {
                repositories: [
                    {name: 'instance2-repo1'}
                ],
                _instance: {
                    name: 'instance2'
                }
            }
        ]
    );

    // when
    var $sources = view.$el.find('.repository-instance');

    // then
    equal($sources.length, 3, "Rendered repository instance names");
    equal($sources.eq(0).text(), '(instance1)', "Rendered repository instance name");
    equal($sources.eq(1).text(), '(instance1)', "Rendered repository instance name");
    equal($sources.eq(2).text(), '(instance2)', "Rendered repository instance name");
});

test("Test repository instance names aren't rendered when there's only one", function() {
    // having
    var view = this.createView();

    // when
    var singleInstance = view.concatResults([
        {
            repositories: [
                {name: 'instance1-repo1'},
                {name: 'instance1-repo2'}
            ],
            _instance: {
                name: 'instance1',
                singleInstance: true
            }
        }
    ]);

    // then
    ok(!singleInstance[0].showInstance, "Set to not show instance for repos from a single instance");
    ok(!singleInstance[1].showInstance, "Set to not show instance for repos from a single instance");
});

test("Test commit detail dialog renders default avatar if none provided", function () {
    // having
    var avatarUrl = 'http://localhost/foobar',
        view = this.renderFromData(
            [
                {
                    repositories: [
                        {
                            name: 'instance1-repo1',
                            avatar: avatarUrl
                        },
                        {
                            name: 'instance1-repo2'
                        }
                    ]
                }
            ]
        );

    // when
    var $avatars = view.$el.find('.project-avatar');

    // then
    equal($avatars.length, 2, "Rendered repository avatars");
    equal($avatars.eq(0).find('img').attr('src'), avatarUrl, "Rendered provided repository avatar image");
    equal($avatars.eq(1).find('.aui-iconfont-devtools-repository').length, 1, "Rendered default repository avatar if no image provided");
});

test("Test commit detail dialog uses repository avatar description for project tooltip", function () {
    var view = this.renderFromData(
            [
                {
                    repositories: [
                        {
                            name: 'instance1-repo1',
                            avatarDescription: 'project'
                        },
                        {
                            name: 'instance1-repo2'
                        }
                    ]
                }
            ]
        );
    var $projects = view.$el.find('.project-avatar');

    // then
    equal($projects.eq(0).attr('title'), 'project');
    equal($projects.eq(1).attr('title'), undefined);
});

test("Test commit detail dialog renders files", function () {
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-files"]);

    var $rows = this.getCommitRows(AJS.$('.devstatus-detail-table'));

    function getFileRow(i, message, matcher) {
        var $row = $rows.eq(i);
        equal($row.find('.filecount').text(), message);
        var $fileRow = $row.next();
        matcher($fileRow.find('.file'));
    }
    function checkNoLines(files) {
        return function($files) {
            equal($files.length, files.length);
            $files.each(function(i) {
                var $file = AJS.$(this);
                equal($file.find('.changetype').text().toUpperCase(), files[i].type);
                equal($file.find('.filename').text(), files[i].name);
            });
        };
    }
    function checkWithLines(files) {
        return function($files) {
            equal($files.length, files.length);
            $files.each(function(i) {
                var $file = AJS.$(this);
                if (files[i].type) {
                    equal($file.find('.lines').text().toUpperCase(), files[i].type);
                } else {
                    equal($file.find('.lines .added').text(), '+' + files[i].added);
                    equal($file.find('.lines .removed').text(), '-' + files[i].removed);
                }
                equal($file.find('.filename').text(), files[i].name);
            });
        };
    }
    getFileRow(0, '10 files', checkNoLines([
        {type: 'ADDED', name: 'a'},
        {type: 'COPIED', name: 'b'},
        {type: 'MOVED', name: 'c'},
        {type: 'DELETED', name: 'd'},
        {type: 'MODIFIED', name: 'e'}
    ]));
    getFileRow(1, '', checkNoLines([]));
    getFileRow(2, '3 files', checkWithLines([
        {type: 'ADDED', name: 'a'},
        {type: 'DELETED', name: 'b'},
        {name: 'c', added: 3, removed: 8}
    ]));
    getFileRow(3, '1 file', checkNoLines([
        {type: 'MODIFIED', name: 'plugins/branch-utils/src/test/java/com/atlassian/stash/internal/branch/list/aheadbehind/ShowBranchAheadBehindCountingOutputHandlerTest.java'}
    ]));
});

test("Test commit detail dialog file count", function () {
    equal(AJS.$(JIRA.Templates.DevStatus.DetailDialog.filesColumn({count: 1})).text(), "1 file");
    equal(AJS.$(JIRA.Templates.DevStatus.DetailDialog.filesColumn({count: 99})).text(), "99 files");
    equal(AJS.$(JIRA.Templates.DevStatus.DetailDialog.filesColumn({count: 100})).text(), "100+ files");
});

test("Test commit detail dialog toggle files", function () {
    var self = this;
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-files"]);

    var $rows = this.getCommitRows(AJS.$('.devstatus-detail-table')).filter(function() {
        // Filter out the empty rows - are not included in the toggle
        return AJS.$(this).find('.filecount').text() !== "";
    });
    var $expand = $rows.closest('.detail-commits-container').find('.file-expand a');
    function checkExpand(expanded) {
        equal($expand.text(), expanded ? 'Hide files' : 'Show all files', 'Expand show/hide state');
    }
    function checkFileHidden(i, hidden) {
        equal($rows.eq(i).next().hasClass('hidden'), hidden, 'File row visibility: ' + i);
    }
    function toggleFiles(i) {
        $rows.eq(i).find('.filecount a').click();
    }

    ok(true, 'Everything should be closed');
    checkExpand(false);
    $rows.each(function(i) {
        checkFileHidden(i, true);
    });

    ok(true, 'Toggle one file at a time');
    $rows.each(function(i) {
        toggleFiles(i);
        ok(self.fileExpandedStub.calledOnce, 'Analytic event fired when showing files for a single commit');
        self.fileExpandedStub.reset();
        checkFileHidden(i, false);
        checkExpand(true);
    });

    ok(true, 'Hide one');
    toggleFiles(0);
    checkFileHidden(0, true);
    checkExpand(true);
    ok(!self.fileExpandedStub.called, 'Analytic event not fired when hiding files for a single commit');

    ok(true, 'Hide them all');
    $expand.click();
    ok(!this.filesExpandedStub.called, 'Analytic event not fired when hiding all files');
    checkExpand(false);
    $rows.each(function(i) {
        checkFileHidden(i, true);
    });

    ok(true, 'Expand them all');
    $expand.click();
    ok(this.filesExpandedStub.called, 'Analytic event fired when showing all files');
    checkExpand(true);
    $rows.each(function(i) {
        checkFileHidden(i, false);
    });

    ok(true, 'Hide one file');
    toggleFiles(0);
    checkFileHidden(0, true);
    checkExpand(true);

    AJS.$('.filename:visible a').eq(0).click();
    ok(this.fileClickedStub.called);
});

test("Test lock screen renders if no data would be displayed", function() {
    var applicationType = 'fecru';
    var data = {detail: [{repositories: []}] };
    var view = this.createView();

    var $fixture = AJS.$('#qunit-fixture');
    $fixture.html('<div id="tab-content-' + applicationType + '"><div class="detail-content-container"></div></div>');
    view.$el = $fixture;
    view.renderSuccess = this.sandbox.spy();
    view._handleFetchSucceeded(applicationType, data);

    ok(view.renderSuccess.neverCalledWith(applicationType, data.detail));
    var contentContainer = view.getContentContainer(applicationType);
    var noPermissionToViewAll = contentContainer.find(".no-permission-to-view-all");
    ok(noPermissionToViewAll.find(".lock-image").length > 0, "has lock image");
    equal(noPermissionToViewAll.text(), "You don't have access to view all related commits. Please contact your administrator.");
});

test("Filenames are correctly clipped", function() {
    var longName = "long-name/averylongnameindeed/indeeditissuchaverylongnameimrathershocked/"
        + "whyisitthatthefilenameisstillgoing/thisisincrediblehave/you/ever/seen/AnythingLikeIt.java";
    var view = this.renderFromData([{
        repositories: [{
                name: 'instance1-repo1',
                avatarDescription: 'project',
                commits: [{
                        fileCount: 1,
                        author: "mclovin",
                        files: [
                            {
                                changeType: "BLOGGED",
                                url: "http://example.com",
                                path: longName
                }]}]
        }]
    }]);

    var $rows = this.getCommitRows(view.$el.find('.devstatus-detail-table'));

    var row = $rows.eq(0);
    ok(view.$el.find('.filerow').hasClass("hidden"));

    // Make the width small enough so we will have to truncate.
    view.$el.find('.file .filename').css("white-space", "nowrap").css("width", "300px");
    row.find('.filecount a').click();

    // For some strange reason this flaky. It isn't important for this test but still strange.
    // ok(!view.$el.find('.filerow').hasClass("hidden"));
    var file = view.$el.find('.file .filename a').eq(0);

    // We've truncated!
    ok(file.text().length < longName.length);
});

test("Test commit detail dialog renders branch tooltip", JIRA.DevStatus.QunitTestUtils.withFakeTimer(function (clock) {
    // having
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewsAndBranches"]);
    var $row1 = this.getCommitRows(AJS.$('.devstatus-detail-table')).eq(1); // 9dba66c
    var $branchesLink = $row1.find('.branches-link');

    // when
    $branchesLink.trigger('mouseover');
    clock.tick(500);

    // then
    var $branchTooltip = AJS.$('.branch-tooltip');
    equal($branchTooltip.length, 1, "Branch tooltip rendered");
    ok(this.branchesShownStub.calledWith('bitbucket'), "Analytic event fired when branches tooltip rendered");

    var $branchNames = $branchTooltip.find('.branch-name-tooltip');
    equal($branchNames.length, 5, "All branches rendered");
    equal($branchNames.eq(0).text(), 'master', "Branch name rendered");
    equal($branchNames.eq(1).text(), "manybranches-test-190fds-df-very-long-branch-name-very-long-long", "Branch name rendered");
    equal($branchNames.eq(2).text(), "manybranches-test-4", "Branch name rendered");
    equal($branchNames.eq(3).text(), "manybranches-test-3", "Branch name rendered");
    equal($branchNames.eq(4).text(), "manybranches-test-2", "Branch name rendered");

    equal($branchTooltip.find('.branch-tooltip-summary').text(), 'and 16 more', "Branch summary rendered");
    $branchTooltip.parent().parent().remove();
}));

test("Test commit detail dialog renders review popup", JIRA.DevStatus.QunitTestUtils.withFakeTimer(function (clock) {
    // having
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewsAndBranches"]);
    var $row2 = this.getCommitRows(AJS.$('.devstatus-detail-table')).eq(2); // 155a0e9
    var $reviewsLink = $row2.find('.reviews-link');

    // when
    $reviewsLink.mouseenter().mousemove();
    clock.tick(500);

    // then
    var $reviewPopup = AJS.$('#inline-dialog-commit-reviews-popup');
    equal($reviewPopup.length, 1, "Review popup rendered");
    ok(this.reviewsShownStub.calledWith('bitbucket'), "Analytic event fired when review popup rendered");

    var $reviews = $reviewPopup.find('.inline-review-row');
    equal($reviews.length, 2, "All reviews rendered");
    equal($reviews.eq(0).find('.inline-review-id').text(), "CR-256", "Review id rendered");
    equal($reviews.eq(0).find('.inline-review-state').text(), "Review", "Review state rendered");
    equal($reviews.eq(1).find('.inline-review-id').text(), "DEF-10", "Review id rendered");
    equal($reviews.eq(1).find('.inline-review-state').text(), "Closed", "Review state rendered");

    $reviews.eq(0).find('a').click();
    ok(this.reviewClickedStub.calledWith('bitbucket'), 'Analytic event fired when review clicked');

    var $summary = $reviewPopup.find('.inline-review-summary');
    equal($summary.text(), 'and 18 more', "Review summary rendered");
    equal($summary.find('a').attr('href'), 'http://lpater-dev:8060/foo/changelog/gitsample?cs=155a0e9308d7e4b06ad1ec146244980839f61951', "Review summary link rendered");
    $summary.find('a').click();
    ok(this.reviewClickedStub.calledTwice, 'Analytic event fired when review clicked');

    $reviewPopup.remove();
}));

test("Test reviews are sorted by priority", function() {
    // having
    var view = this.createView(),
        commit = {
            "id": "cs1",
            "reviews": {
                "totalCount": 9,
                "reviews": [
                    {
                        "id": "CR-6",
                        "state": "CLOSED"
                    },
                    {
                        "id": "CR-5",
                        "state": "REJECTED"
                    },
                    {
                        "id": "CR-1",
                        "state": "REVIEW"
                    },
                    {
                        "id": "CR-3",
                        "state": "APPROVAL"
                    },
                    {
                        "id": "CR-4",
                        "state": "SUMMARIZE"
                    },
                    {
                        "id": "CR-2",
                        "state": "REVIEW"
                    }
                ]
            }
        };

    // when
    view._orderCommitReviews(commit);

    // then
    deepEqual(_.reduce(commit.reviews.reviews,
        function(memo, item) {
            memo.push(item.id);
            return memo;
        }, []), ["CR-1", "CR-2", "CR-3", "CR-4", "CR-5", "CR-6"]);

});

test("Test reviews in invalid states are filtered", function() {
    // having
    var view = this.createView(),
        commit = {
            "id": "cs1",
            "reviews": {
                "totalCount": 9,
                "reviews": [
                    {
                        "id": "CR-6",
                        "state": "DRAFT"
                    },
                    {
                        "id": "CR-5",
                        "state": "DEAD"
                    },
                    {
                        "id": "CR-1",
                        "state": "REVIEW"
                    },
                    {
                        "id": "CR-3",
                        "state": "fooo"
                    },
                    {
                        "id": "CR-4"
                    }
                ]
            }
        };

    // when
    view._orderCommitReviews(commit);

    // then
    equal(commit.reviews.reviews.length, 1, "Reviews in invalid states were filtered");
    equal(commit.reviews.reviews[0].id, 'CR-1', "Reviews in valid states were preserved");
});

test("Test checking if repositories have reviews", function() {
    // having
    var view = this.createView(),
        repositories = [
            {name: 'a', commits: []},
            {name: 'b', commits: [{id: '1'}]},
            {name: 'c', commits: [{id: '1', reviews: null}]},
            {name: 'd', commits: [{id: '1', reviews: undefined}]},
            {name: 'e', commits: [{id: '1', reviews: {}}]},
            {name: 'f', commits: [{id: '1', reviews: {totalCount: 0, reviews: []}}]},
            {name: 'g', commits: [{id: '1', reviews: {totalCount: 100, reviews: [
                {id: "CR-1"}
            ]}}]},
            {name: 'g', commits: [
                {id: '0'},
                {id: '1', reviews: {totalCount: 100, reviews: [
                    {id: "CR-1"}
                ]}}
            ]}
        ];

    // when
    view.transformRepositories(repositories);

    // then
    ok(!repositories[0].showReviews, "don't show reviews if no commits");
    ok(!repositories[1].showReviews, "dont't show reviews if no commits with review prop");
    ok(!repositories[2].showReviews, "don't show reviews if review prop null");
    ok(!repositories[3].showReviews, "don't show reviews if review prop undefined");
    ok(repositories[4].showReviews, "show reviews if review object present");
    ok(repositories[5].showReviews, "show reviews if review object present");
    ok(repositories[6].showReviews, "show reviews if review object present");
    ok(repositories[7].showReviews, "show reviews if review object in at least one commit");
});

test("Test checking if repositories have branches", function() {
    // having
    var view = this.createView(),
        repositories = [
            {name: 'a', commits: []},
            {name: 'b', commits: [{id: '1'}]},
            {name: 'c', commits: [{id: '1', branches: null}]},
            {name: 'd', commits: [{id: '1', branches: undefined}]},
            {name: 'e', commits: [{id: '1', branches: {}}]},
            {name: 'f', commits: [{id: '1', branches: {totalCount: 0, branches: []}}]},
            {name: 'g', commits: [{id: '1', branches: {totalCount: 100, branches: [
                {id: "CR-1"}
            ]}}]},
            {name: 'g', commits: [
                {id: '0'},
                {id: '1', branches: {totalCount: 100, branches: [
                    {id: "CR-1"}
                ]}}
            ]}
        ];

    // when
    view.transformRepositories(repositories);

    // then
    ok(!repositories[0].showBranches, "don't show branches if no commits");
    ok(!repositories[1].showBranches, "dont't show branches if no commits with branches prop");
    ok(!repositories[2].showBranches, "don't show branches if branches prop null");
    ok(!repositories[3].showBranches, "don't show branches if branches prop undefined");
    ok(repositories[4].showBranches, "show branches if branches object present");
    ok(repositories[5].showBranches, "show branches if branches object present");
    ok(repositories[6].showBranches, "show branches if branches object present");
    ok(repositories[7].showBranches, "show branches if branches object in at least one commit");
});

test("Test checking if repositories have files", function() {
    var view = this.createView(),
        repositories = [
            {name: 'a', commits: [{id: '1'}]},
            {name: 'b', commits: [{id: '1', files: []}]}
        ];

    view.transformRepositories(repositories);

    ok(!repositories[0].showFiles, "don't show files if commit object has no files");
    ok(repositories[1].showFiles, "show files if commit object has files");
});

test("Test get unique commit count", function() {
    var view = this.createView();
    equal(view.getUniqueCommitCount([
    ]), 0, "no repo");
    equal(view.getUniqueCommitCount([
        {name: 'M'}
    ]), 0, "one repo no commits");
    equal(view.getUniqueCommitCount([
        {name: 'M', commits: []}
    ]), 0, "one repo empty commits");
    equal(view.getUniqueCommitCount([
        {name: 'M', commits: [ {id: undefined}, {id: null}, {id: '3'}, {name: 'name'} ]}
    ]), 1, "one repo with null id");
    equal(view.getUniqueCommitCount([
        {name: 'M', commits: [ {id: '1'}, {id: '2'}, {id: '3'} ]}
    ]), 3, "one repo");
    equal(view.getUniqueCommitCount([
        {name: 'z', commits: []},
        {name: 'a', commits: []},
        {name: 'M', commits: [ {id: '1'}, {id: '2'}, {id: '3'} ]}
    ]), 3, "one repo with empty repo");
    equal(view.getUniqueCommitCount([
        {name: 'z', commits: []},
        {name: 'a', commits: [ {id: '4'}]},
        {name: 'M', commits: [ {id: '1'}, {id: '2'}, {id: '3'} ]}
    ]), 4, "two repos with no overlap");
    equal(view.getUniqueCommitCount([
        {name: 'z', commits: []},
        {name: 'a', commits: [ {id: '1'}, {id: '4'}]},
        {name: 'M', commits: [ {id: '1'}, {id: '2'}, {id: '3'} ]}
    ]), 4, "two repos with overlap");
    equal(view.getUniqueCommitCount([
        {name: 'z', commits: [ {id: '5'}, {id: '3'}]},
        {name: 'a', commits: [ {id: '1'}, {id: '4'}]},
        {name: 'M', commits: [ {id: '1'}, {id: '2'}, {id: '3'} ]}
    ]), 5, "three repos with overlap");
});

test("Test bitbucket detail commits dialog with success request", function() {
    expect(28);
    JIRA.DevStatus.Date.format = 'YYYY-MM-DD';
    var view = this.createView(5);

    view.show();
    var dialog = this.getCommitDetailDialog();
    var spinner = dialog.content.find(".status-loading");
    var activePane = dialog.content.find(".tabs-pane.active-pane");
    var repoNames = ["repoAWESOME", "repoGOOD", "repoone"];
    var commitCount = [3, 1, 2];

    ok(spinner.is(":visible"), "A spinner is visible  when there is an onflight ajax request");
    ok(spinner.children().length > 0);
    ok(activePane.hasClass("loading"), "Dialog content is faded out when there is an onflight ajax request");

    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[0]);
    AJS.TestContractManager.respondToRequest("bitbucketDetailCommit-multipleRepo", this.server.requests[1]);

    this.assertNoErrorMessagesInCanvas();

    var commitContainers = dialog.content.find(".detail-commits-container");
    var mergeRow = dialog.content.find("tr[data-changesetid='jklmnalsotoolong']");

    ok(!spinner.is(":visible"), "A spinner is no longer visible when the onflight ajax request has completed");
    ok(!activePane.hasClass("loading"), "Dialog content is no longer faded out when ajax has completed");
    equal(commitContainers.length, repoNames.length, "Number of commit tables rendered into the dialog is correct");

    _.each(commitContainers, _.bind(function(element, index) {
        var el = AJS.$(element);
        var repoLink = el.find(".repository-link");
        equal(repoLink.text(), repoNames[index], "The repository name exist and is correct: " + index);

        repoLink.click();
        ok(this.repoClickedStub.called, "Analytic event is fired correctly when clicking on repo link: " + index);

        var commitList = this.getCommitRows(el.find(".commits-table"));
        equal(commitList.length, commitCount[index], "List of commits exist and contains the correct number: " + index);

        var changesetlink = commitList.find(".changesetid");
        changesetlink.eq(0).click();
        ok(this.commitClickedStub.called, "Analytic event is fired correctly when clicking on commit link: " + index);

        var timestamp = commitList.find(".timestamp .livestamp");
        ok(timestamp.length != 0, "Time stamp is rendered into the table");
        equal(timestamp.attr("title"), '2013-09-23', "Tooltip is correctly appended onto the timestamp element: " + index);
    }, this));

    ok(mergeRow.hasClass("merge"), "Merge commit is correctly displayed");
    equal(mergeRow.find(".aui-lozenge.merge").length, 1, "Merge row is correctly appended with a merge lozenge");
});

test("Test bitbucket detail commits dialog with success request with one repo", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "bitbucketDetailCommit-oneRepo"], 2);

    this.assertNoErrorMessagesInCanvas();

    var commitContainer = this.getCommitDetailDialog().content.find(".detail-commits-container");
    equal(commitContainer.length, 1, "One commit table is rendered into the dialog");
    equal(commitContainer.find(".repository-link").text(), "repoone", "The repository name exist and is correct");
});

/// Tests for auth and error handling in BaseDetailDialogView.js
test("Test bitbucket detail commits dialog with success request and errors", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "bitbucketDetailCommit-multipleRepoWithErrors"], 2);

    equal(this.server.requests.length, 3, "A third request was sent");
    ok(this.server.requests[2].url.indexOf('/provider/auth-status') > 0, "The third request was to re-fetch auth status");
    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[2]);

    this.assertNoNoPermissionMessage();

    var dialog = this.getCommitDetailDialog();
    equal(dialog.content.find(".detail-commits-container").length, 3, "There are still commits visible with errors");
    equal(dialog.content.find(".authentication-problem h2").length, 0,
            "No authenticate message is shown when there is data");
    equal(dialog.content.find(".connection-problem h2").length, 0,
            "No connection message is shown in full canvas when there is data");
    equal(dialog.content.find(".aui-message.warning").length, 1,
            "Connection message is prepended to commit table when there is connection problem");
    var errorText = dialog.content.find(".aui-message.warning").text();
    ok(errorText.indexOf("JIRA is having difficulty contacting") >= 0 &&
            errorText.indexOf("If this condition persists, please contact your ") >= 0,
            "Correct connection message is prepended to commit table when there is connection problem");
});

test("Test bitbucket detail commits dialog with only authentication errors", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit3LO-stashUnauthorizedError"]);

    equal(this.server.requests.length, 3, "A third request was sent");
    ok(this.server.requests[2].url.indexOf('/provider/auth-status') > 0, "The third request was to re-fetch auth status");
    AJS.TestContractManager.respondToRequest("detailCommit3LO-stash", this.server.requests[2]);

    var dialog = this.getCommitDetailDialog();
    equal(dialog.content.find(".detail-commits-container").length, 0, "There is no commits table on ajax request without data");
    equal(dialog.content.find(".authentication-problem h2").text(), "Authenticate to see related commits",
            "Authentication message is shown when there's no data and with authentication problem");
});

test("Test bitbucket detail commits dialog with only connection errors", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit3LO-stashOtherError"]);

    equal(this.server.requests.length, 2, "No auth status re-fetch request was sent");

    var dialog = this.getCommitDetailDialog();
    equal(dialog.content.find(".detail-commits-container").length, 0, "There is no commits table on ajax request without data");
    equal(dialog.content.find(".connection-problem h2").text(), "Unable to retrieve commit information",
            "Connection message is shown in full canvas when there's no data and with connection problem");
    // not testing the detailed error message with instances, as that should have been covered by SummaryErrorView.js tests
});

test("Test bitbucket detail commits dialog with success but no permission to view any commits", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "bitbucketDetailCommit-empty"], 1);

    var dialog = this.getCommitDetailDialog();
    equal(dialog.content.find(".detail-commits-container").length, 0, "There is no commits table on fail ajax request");
    equal(dialog.content.find(".no-permission-to-view-all").text(), "You don't have access to view all related commits. Please contact your administrator.",
            "No permission message is shown in full canvas where there's no data");
});

test("Test bitbucket detail commits dialog with success but no permission to view some commits", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "bitbucketDetailCommit-multipleRepo"], 6);

    this.assertNoConnectionErrorWarning();

    var dialog = this.getCommitDetailDialog();
    ok(dialog.content.find(".detail-commits-container").length > 0, "There are commits visible");
    equal(dialog.content.find(".no-permission-to-view-all").text(), "You don't have access to view all related commits. Please contact your administrator.",
            "No permission message is shown in full canvas where there's no data");
});

test("Test bitbucket detail commits dialog with fail request", function() {
    this.renderViewFromContracts(["detailCommit3LO-stash", "bitbucketDetailCommit-failRequest"]);

    var dialog = this.getCommitDetailDialog();
    equal(dialog.content.find(".detail-commits-container").length, 0, "There is no commits table on fail ajax request");
    equal(dialog.content.find(".connection-problem h2").text(), "Unable to retrieve commit information",
        "A message is shown when ajax request failed");
});

test("Test review creation for changeset", function() {
    // having
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewCreation"]);
    var dialog = this.getCommitDetailDialog();

    // when
    var $createReviewCs = dialog.content.find('.commits-table .create-review-commit a');
    $createReviewCs.click();

    // then
    equal($createReviewCs.length, 1, 'Create review link rendered');
    equal($createReviewCs.eq(0).attr('href'), 'http://fecru/cru/create?repo=gitsample&csid=3da7cf6a980221195e9d09dd80c30c3a002a0c54', 'Correct href rendered');
    ok(this.createReviewClicked.calledWith('bitbucket', false))
});

test("Test review creation for single instance", function() {
    // having
    JIRA.Issue.getIssueId = this.sandbox.stub().returns(123);
    this.sandbox.stub(JIRA.DevStatus.Navigate, 'navigate');

    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewCreation"]);
    var dialog = this.getCommitDetailDialog();

    // when
    var $createReviewCs = dialog.content.find('.create-review-instance a');
    equal($createReviewCs.length, 1, 'Create review link rendered');
    $createReviewCs.eq(0).click();

    // then
    equal(this.server.requests.length, 3, "Request for targets sent");
    AJS.TestContractManager.respondToRequest("createReviewDialog-targets", this.server.requests[2]);

    equal(AJS.$('#devstatus-cta-create-review-dialog:visible').length, 0, 'No dialog shown');
    ok(JIRA.DevStatus.Navigate.navigate.called, 'Navigate called');
    equal(JIRA.DevStatus.Navigate.navigate.getCall(0).args[0], 'http://lpater-dev:8060/foo/cru/create?issueKey=DEF-1&issueSummary=1&issueDescription=This+is+an+issue+descriptopn',
        'Navigated to the right url');
    ok(this.createReviewClicked.calledWith('bitbucket', true));
});

test("Test review creation for multiple instances", function() {
    // having
    JIRA.Issue.getIssueId = this.sandbox.stub().returns(123);
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewCreation-twoInstances"]);
    var dialog = this.getCommitDetailDialog();

    // when
    var $createReviewCs = dialog.content.find('.create-review-instance a');
    equal($createReviewCs.length, 1, 'Create review link rendered');
    $createReviewCs.eq(0).click();

    equal(this.server.requests.length, 3, "Request for targets sent");
    AJS.TestContractManager.respondToRequest("createReviewDialog-targets", this.server.requests[2]);
    AJS.$('#devstatus-cta-create-review-dialog .target').eq(0).click();

    // then
    equal(AJS.$('#devstatus-cta-create-review-dialog').length, 1, 'Choose instance dialog shown');
    equal(AJS.$('#devstatus-cta-create-review-dialog .target').length, 2, 'Two instances shown in the picker');
    equal(AJS.$('#devstatus-cta-create-review-dialog .target .title').eq(0).text(), 'lpater-dev', 'Instance name shown');
    equal(AJS.$('#devstatus-cta-create-review-dialog .target .title').eq(1).text(), 'lpater-dev-2', 'Instance name shown');
    ok(!AJS.$('#devstatus-cta-create-review-dialog .target').eq(0).hasClass('selected'), 'Primary instance picked by default');
    ok(AJS.$('#devstatus-cta-create-review-dialog .target').eq(1).hasClass('selected'), 'Primary instance picked by default');


    ok(this.createReviewClicked.calledWith('bitbucket', true));
});

test("Test review creation works by following link - single instance", function() {
    // having
    AJS.Meta.set("fusion-open-detail-dialog", "create-review-bitbucket");
    JIRA.Issue.getIssueId = this.sandbox.stub().returns(123);
    this.sandbox.stub(JIRA.DevStatus.Navigate, 'navigate');

    // when
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewCreation", "createReviewDialog-targets"]);
    var dialog = this.getCommitDetailDialog();

    // then
    ok(JIRA.DevStatus.Navigate.navigate.called, 'Navigate called');
    equal(JIRA.DevStatus.Navigate.navigate.getCall(0).args[0], 'http://lpater-dev:8060/foo/cru/create?issueKey=DEF-1&issueSummary=1&issueDescription=This+is+an+issue+descriptopn',
        'Navigated to the right url');
});

test("Test review creation works by following link - multiple instances", function() {
    // having
    AJS.Meta.set("fusion-open-detail-dialog", "create-review-bitbucket");
    JIRA.Issue.getIssueId = this.sandbox.stub().returns(123);

    // when
    this.renderViewFromContracts(["detailCommit3LO-stash", "detailCommit-reviewCreation-twoInstances", "createReviewDialog-targets"]);
    var dialog = this.getCommitDetailDialog();

    // then
    equal(AJS.$('#devstatus-cta-create-review-dialog:visible').length, 1, 'Choose instance dialog shown');
});

test("Test create review instances not shown if no link from remote", function() {
    // when
    var view = this.renderFromData([
        {
            repositories: [{
                name: 'foo',
                commits: []
            }],
            _instance: {
                name: 'theta'
            }
        },
        {
            repositories: [{
                name: 'foo',
                commits: [{
                    "author": "lpater"
                }]
            }],
            _instance: {
                name: 'zeta'
            }
        },
        {
            repositories: [{
                name: 'foo',
                commits: [{
                    "createReviewUrl": "http://review",
                    "author": "lpater"
                }]
            }],
            _instance: {
                name: 'eta'
            }
        },
        {
            repositories: [],
            _instance: {
                name: 'teta'
            }
        }
    ]);

    // then
    deepEqual(_.pluck(view.createReviewDialog.instances, 'name'), ['eta']);
});

test("Heading text for commits contains no duplicates", function() {
    var view = this.createView();
    view.show();
    var dialog = this.getCommitDetailDialog();
    equal(dialog.heading.text(), this.issueKey + ": 10 unique commits");
});

test("Heading text for commits contains duplicates", function() {
    var view = this.createView();
    view.model.set("tabs", {
        bitbucket: {name:"Bitbucket", count: 7},
        stash: {name:"Stash", count: 6}
    });
    view.show();
    var dialog = this.getCommitDetailDialog();
    equal(dialog.heading.text(), this.issueKey + ": 10 unique commits (and 3 duplicates)");
});
