AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-qunit-test-utils");

module("JIRA.DevStatus.BaseDetailDialogView", {
    setup: function() {
        this.sandbox = sinon.sandbox.create();
        this.issueKey = "AWESOME-1";
        this.model = this._mockOfClass(JIRA.DevStatus.BaseDetailDialogModel);
        JIRA.DevStatus.QunitTestUtils.spyAnalyticMethods("Summary", this.sandbox);
    },
    teardown : function () {
        this.sandbox.restore();
        JIRA.DevStatus.QunitTestUtils.cleanAllDialogs();
        //JIRA form dialog appends overflow: hidden to the body of the page. This is to remove it.
        AJS.$("body").css("overflow", "initial");
    },
    getDevStatusDetailDialog: function() {
        var dialog = AJS.$(".jira-dialog");
        return {
            content: dialog.find(".devstatus-dialog-content"),
            footer: dialog.find(".buttons-container.form-footer"),
            getUnableToViewAllMessage: function() {
                return this.content.find(".no-permission-to-view-all").text();
            }
        }
    },
    createView: function(options) {
        if (!options) {options = {};}
        var view = new JIRA.DevStatus.BaseDetailDialogView(options);
        view.analytics = JIRA.DevStatus.SummaryAnalytics;
        return view;
    },
    createDefaultView: function(count) {
        return this.createView({
            type: "commit",
            model: this.model,
            count: count || 2,
            issueKey: this.issueKey
        })
    },
    _mockOfClass: function (constructor) {
        var mock = {};
        _.each(_.functions(constructor.prototype), function (functionName) {
            mock[functionName] = sinon.stub();
        });

        return mock;
    },
    assertVisible: function(div, message) {
        ok(div.is(':visible') && !div.hasClass('hidden'), message);
    },
    assertInvisible: function(div, message) {
        ok(!div.is(':visible') || div.hasClass('hidden'), message);
    },
    defaultTabOrder: ["bitbucket", "fecru", "stash"],
    defaultTabs: {
        bitbucket: {
            count: 5000,
            name: "Bitbucket"
        },
        fecru: {
            count: 1,
            name: "Fisheye / Crucible"
        },
        stash: {
            count: 99,
            name: "Stash"
        }
    },
    sourceStash1: {
        "typeName": "Stash",
        "type": "stash",
        "name": "Stash Dev One",
        "baseUrl": "https://stash1.dev.atlassian.com",
        "applicationLinkId": "62153ed0-1c05-3f55-886f-d46708ffcc91"
    },
    sourceStash2: {
        "typeName": "Stash",
        "type": "stash",
        "name": "Stash Dev Two",
        "baseUrl": "https://stash2.dev.atlassian.com",
        "applicationLinkId": "62153ed0-1c05-3f55-886f-d46708ffcc92"
    },
    sourceStash3: {
        "typeName": "Stash",
        "type": "stash",
        "name": "Stash Dev Three",
        "baseUrl": "https://stash3.dev.atlassian.com",
        "applicationLinkId": "62153ed0-1c05-3f55-886f-d46708ffcc93"
    },
    sourceStash4: {
        "typeName": "Stash",
        "type": "stash",
        "name": "Stash Dev Four",
        "baseUrl": "https://stash4.dev.atlassian.com",
        "applicationLinkId": "62153ed0-1c05-3f55-886f-d46708ffcc94"
    }
});

test("Test default values are set when not provided", function() {
    var view = this.createView();

    ok(_.isObject(view.model), "A detail dialog model is created if not provided");
    ok(view.options.width != null, "Width parameter is defined if not provided");
});

test("Test default values are NOT set when width is provided", function() {
    view = this.createView({
        model: this.model,
        width: 9000
    });
    equal(view.model, this.model, "Provided dialog model parameter is respected");
    equal(view.options.width, 9000, "Provided width parameter is respected");
});

test("Tabs are ordered and keep data attributes", function() {
    this.model.get.withArgs("tabs").returns({y: {foo: 10}, a: {}, z: {bar: 20}, b: {}});
    console.log(this.createDefaultView()._convertTabsForSoy());
    deepEqual(this.createDefaultView()._convertTabsForSoy(), [{type: 'a'}, {type: 'b'}, {type: 'y', foo: 10}, {type: 'z', bar: 20}]);
});

test("Tabs are not rendered when there is only one product", function() {
    var tabsPerApp = {
        bitbucket: {
            count: 5000
        }
    };
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView(1);

    view.show();

    ok(this.model.switchTab.calledWith("bitbucket"), "The initial ajax call is made on show()");

    var dialog = this.getDevStatusDetailDialog();
    var tabsMenu = dialog.content.find(".tabs-menu");
    equal(tabsMenu.find(".menu-item").length, 1, "There is only one tab");
    ok(tabsMenu != null, "Tabs menu element exist");
    this.assertInvisible(tabsMenu, "Tabs menu is not visible");

});

test("Tabs are rendered when there are more than one product", function() {
    expect(13);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView(99999);

    view.show();

    ok(this.model.switchTab.calledWith("bitbucket"), "The initial ajax call is made on show()");

    var dialog = this.getDevStatusDetailDialog();

    var tabsMenu = dialog.content.find(".tabs-menu");
    var menuItem = tabsMenu.find(".menu-item");
    ok(tabsMenu != null, "Tabs menu element exist");
    this.assertVisible(tabsMenu, "Tabs menu is visible");

    equal(menuItem.length, 3, "There are three tabs");
    _.each(menuItem, function(element, index) {
        var app = appsOrder[index];
        var el = AJS.$(element);
        equal(el.text(), tabsPerApp[app].name + tabsPerApp[app].count, "Tabs content is correct with app name + count");

        el.children().click();
        ok(instance.model.switchTab.calledWith(app, false), "Tab clicks are propagated properly back to the model without forceFetch");
        instance.model.switchTab.reset();
    });

    view.shouldRefreshOnTabSwitch = function() {return true;};
    _.each(menuItem, function(element, index) {
        var app = appsOrder[index];
        var el = AJS.$(element);

        el.children().click();
        ok(instance.model.switchTab.calledWith(app, true), "Tab clicks are propagated properly back to the model with forceFetch");
        instance.model.switchTab.reset();
    });
});

test("Tabs are rendered when switched away before data is back", function() {
    expect(7);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView(99999);

    view.show();

    ok(this.model.switchTab.calledWith("bitbucket"), "The initial ajax call is made on show()");
    view.renderSuccess = sinon.stub();
    view._renderError = sinon.stub();

    view._handleFetchSucceeded("stash", {detail: ["stash-data"]});
    ok(view.renderSuccess.calledWith("stash", ["stash-data"]), "stash pane is rendered first");
    ok(view._renderError.calledWith("stash", undefined, true), "error is handled on stash first");
    view._handleFetchSucceeded("bitbucket", {detail: ["bb-data1", "bb-data2"], errors: "error"});
    ok(view.renderSuccess.calledWith("bitbucket", ["bb-data1", "bb-data2"]), "bitbucket pane is rendered later");
    ok(view._renderError.calledWith("bitbucket", "error", true), "error is handled on bitbucket later");

    view._handleFetchSucceeded("fecru", {detail: [], errors: "error"});
    ok(view.renderSuccess.neverCalledWith("fecru", undefined), "fecru pane is NOT rendered");
    ok(view._renderError.calledWith("fecru", "error", false), "error is handled on bitbucket later");
});

test("Authentication status footer is not rendered when no errors", function() {
    expect(3);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": true, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertInvisible(statusDiv, "Status div is not visible");
});

test("Authentication status footer is rendered when there is one auth error", function() {
    expect(6);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": false, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following application:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 1, 'One unauth instance is shown');
    ok(instanceDivs.text(), this.sourceStash1.name, 'Unauth instance name is correct');
});

test("Authentication screen shown when there is no data", function() {
    expect(2);

    var view = this.createDefaultView();

    view.renderSuccess = sinon.stub();
    view._renderNoPermissionToViewAllWarning = sinon.stub();

    view._handleFetchSucceeded("stash", {detail: []});
    ok(view.renderSuccess.neverCalledWith("stash", ["stash-data"]), "stash pane is rendered first");

    ok(view._renderNoPermissionToViewAllWarning.calledWith("stash"), "No Permission called");
});

test("Authentication screen not shown when there is data", function() {
    expect(2);

    var view = this.createDefaultView();

    view.renderSuccess = sinon.stub();
    view._renderNoPermissionToViewAllWarning = sinon.stub();

    view._handleFetchSucceeded("stash", {detail: ["things"]});
    ok(view.renderSuccess.calledWith("stash", ["things"]), "stash pane is rendered first");

    ok(view._renderNoPermissionToViewAllWarning.neverCalledWith("stash"), "No Permission called");
});

test("Authentication screen not shown when hasData overridden", function() {
    expect(2);

    var view = this.createDefaultView();

    view.renderSuccess = sinon.stub();
    view._renderNoPermissionToViewAllWarning = sinon.stub();
    view.hasData = function() {return false;}

    view._handleFetchSucceeded("stash", {detail: ["things"]});
    ok(view.renderSuccess.neverCalledWith("stash", ["things"]), "stash pane is rendered first");

    ok(view._renderNoPermissionToViewAllWarning.calledWith("stash"), "No Permission called");

});


test("Authentication status footer is NOT rendered when there is one incorrectly configured link with auth error", function() {
    expect(3);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": false, "configured": false }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertInvisible(statusDiv, "Status div is not visible");
});

test("Authentication status footer is rendered when there is one auth error with non-errors", function() {
    expect(6);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash2, "authenticated": true, "configured": true},
        { "source": this.sourceStash1, "authenticated": false, "configured": true },
        { "source": this.sourceStash3, "authenticated": true, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following application:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 1, 'One unauth instance is shown');
    ok(instanceDivs.text(), this.sourceStash1.name, 'Unauth instance name is correct');
});

test("Authentication status footer is NOT rendered when there is one incorrectly configured link with other correctly configured links", function() {
    expect(3);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash2, "authenticated": true, "configured": true},
        { "source": this.sourceStash1, "authenticated": false, "configured": false },
        { "source": this.sourceStash3, "authenticated": true, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertInvisible(statusDiv, "Status div is not visible");
});

test("Authentication status footer is rendered when there is one incorrectly configured link with other correctly configured links, but with auth error", function() {
    expect(6);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash2, "authenticated": true, "configured": true},
        { "source": this.sourceStash1, "authenticated": false, "configured": false },
        { "source": this.sourceStash3, "authenticated": false, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following application:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 1, 'One unauth instance is shown');
    ok(instanceDivs.text(), this.sourceStash1.name, 'Unauth instance name is correct');
});

test("Authentication status footer is rendered when there are two auth errors", function() {
    expect(6);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": false, "configured": true },
        { "source": this.sourceStash2, "authenticated": false, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following applications:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 2, 'Two unauth instances are shown');
    equal(instanceDivs.text(), this.sourceStash1.name + this.sourceStash2.name, 'Unauth instance names are correct');
});

test("Authentication status footer is rendered when there are three auth errors", JIRA.DevStatus.QunitTestUtils.withFakeTimer(function (clock) {
    expect(11);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": false, "configured": true },
        { "source": this.sourceStash2, "authenticated": false, "configured": true },
        { "source": this.sourceStash3, "authenticated": false, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following applications:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 2, 'Two unauth instances are shown');
    equal(instanceDivs.text(), this.sourceStash1.name + this.sourceStash2.name, 'Shown unauth instance names are correct');
    var moreLink = statusDiv.find('.more-instances');
    equal(moreLink.length, 1, 'The link for more unauth instances is shown');
    moreLink.click();
    clock.tick(1); // to make inline dialog appear
    // inline dialog dom element is created only after the click
    var inlineDialogDiv = AJS.$(document).find('#inline-dialog-instance-list-popup');
    equal(inlineDialogDiv.length, 1, 'Inline dialog for more instances is created');
    this.assertVisible(inlineDialogDiv, 'Inline dialog is visible after clicking on more link');
    var popupInstancesDivs = inlineDialogDiv.find('.instance');
    equal(popupInstancesDivs.length, 1, 'One instance in the inline dialog');
    equal(popupInstancesDivs.text(), this.sourceStash3.name, "Shown unauth instance name in popup is correct");
}));

test("Authentication status footer is rendered when there are four auth errors", JIRA.DevStatus.QunitTestUtils.withFakeTimer(function (clock) {
    expect(11);
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView();

    view.show();

    ok(this.model.fetchAuthenticationStatuses.calledWith(), "The fetch auth ajax call is made on show()");

    var authResult = [
        { "source": this.sourceStash1, "authenticated": false, "configured": true },
        { "source": this.sourceStash2, "authenticated": false, "configured": true },
        { "source": this.sourceStash3, "authenticated": false, "configured": true },
        { "source": this.sourceStash4, "authenticated": false, "configured": true }
    ];
    view._handleAuthSucceeded(authResult);

    var dialog = this.getDevStatusDetailDialog();

    var statusDiv = dialog.footer.find(".oauth-status");
    ok(statusDiv != null && statusDiv.length > 0, "Status div element exist");
    this.assertVisible(statusDiv, "Status div is visible");
    ok(statusDiv.text().indexOf('You might be able to see more information by authenticating with the following applications:') >= 0, "Status div contains right message title");
    var instanceDivs = statusDiv.find('.instance');
    equal(instanceDivs.length, 2, 'Two unauth instances are shown');
    equal(instanceDivs.text(), this.sourceStash1.name + this.sourceStash2.name, 'Shown unauth instance names are correct');
    var moreLink = statusDiv.find('.more-instances');
    equal(moreLink.length, 1, 'The link for more unauth instances is shown');
    moreLink.click();
    clock.tick(1); // to make inline dialog appear
    // inline dialog dom element is created only after the click
    var inlineDialogDiv = AJS.$(document).find('#inline-dialog-instance-list-popup');
    equal(inlineDialogDiv.length, 1, 'Inline dialog for more instances is created');
    this.assertVisible(inlineDialogDiv, 'Inline dialog is visible after clicking on more link');
    var popupInstancesDivs = inlineDialogDiv.find('.instance');
    equal(popupInstancesDivs.length, 2, 'Two instances in the inline dialog');
    equal(popupInstancesDivs.text(), this.sourceStash3.name + this.sourceStash4.name, "Shown unauth instance names in popup are correct");
}));

test("Test analytics are fired when clicked on product tab", function() {
    expect(1 + (3 * 3));
    var instance = this;
    var appsOrder = this.defaultTabOrder;
    var tabsPerApp = this.defaultTabs;
    this.model.get.withArgs("tabs").returns(tabsPerApp);
    var view = this.createDefaultView(99999);
    view.show();

    var menuItem = this.getDevStatusDetailDialog().content.find(".tabs-menu").find(".menu-item");
    equal(menuItem.length, 3, "There are three tabs");
    _.each(menuItem, function(element, index) {
        var app = appsOrder[index];
        var el = AJS.$(element);

        el.children().click();
        ok(instance.model.switchTab.calledWith(app, false), "Tab clicks are propagated properly back to the model without forceFetch");
        ok(JIRA.DevStatus.SummaryAnalytics.fireDetailTabClicked.called,
            "Analytic event is fired correctly when clicking on product tab in detail dialog");
        ok(JIRA.DevStatus.SummaryAnalytics.fireDetailTabClicked.calledWith(app, undefined),
            "Analytic event is fired correctly when clicking on product tab in detail dialog");
        instance.model.switchTab.reset();
    });
});

test("Test sanitised pane id", function() {
    // It shouldn't matter that the applicationId is a CSS selector
    var id = "bad.id";
    var tabs = {};
    tabs[id] = {
        count: 1,
        name: "Invalid"
    };
    this.model.get.withArgs("tabs").returns(tabs);
    var view = this.createDefaultView(99999);
    view.show();
    ok(view._getPane(id).length === 1);
});

test("Anon user always see a message to login if count is different regardless of error case", function() {
    var contentMap = {};
    var anonStub = this.sandbox.stub(JIRA.Users.LoggedInUser, "isAnonymous");
    this.model.get.withArgs("tabs").returns(this.defaultTabs);

    var view = this.createDefaultView();
    view.show();

    // No error, not anon = generic message
    this.model.get.withArgs("contentMap").returns({stash: {}});
    anonStub.returns(false);
    view.renderNoPermissionToViewAllWarningAtBottom("stash", 1);
    equal(this.getDevStatusDetailDialog().getUnableToViewAllMessage(), "Unable to retrieve development information Please contact your administrator.");
    view.getContentContainer("stash").empty();

    // No error, anon = login message
    this.model.get.withArgs("contentMap").returns({stash: {}});
    anonStub.returns(true);
    view.renderNoPermissionToViewAllWarningAtBottom("stash", 1);
    equal(this.getDevStatusDetailDialog().getUnableToViewAllMessage(), "Unable to retrieve development information You might be able to view more by authenticating with JIRA.");
    view.getContentContainer("stash").empty();

    // Contains error, not anon = no message
    this.model.get.withArgs("contentMap").returns({stash: {errors: {empty: false}}});
    anonStub.returns(false);
    view.renderNoPermissionToViewAllWarningAtBottom("stash", 1);
    equal(this.getDevStatusDetailDialog().getUnableToViewAllMessage(), "");
    view.getContentContainer("stash").empty();

    // Contains error, anon = login message
    this.model.get.withArgs("contentMap").returns({stash: {errors: {empty: false}}});
    anonStub.returns(true);
    view.renderNoPermissionToViewAllWarningAtBottom("stash", 1);
    equal(this.getDevStatusDetailDialog().getUnableToViewAllMessage(), "Unable to retrieve development information You might be able to view more by authenticating with JIRA.");
    view.getContentContainer("stash").empty();
});

test("Incapable error is not shown in detail dialog", function() {
    expect(4 * 4);
    this.model.get.withArgs("tabs").returns(this.defaultTabs);
    var view = this.createDefaultView(99999);
    view.show();

    var container = "bitbucket";
    var errorData = [{error: 'unathorized', instance: {baseUrl: 'sss', name: 'squirtle'}},
                     {error: 'random', instance: {baseUrl: 'ccc', name: 'charmander'}},
                     {error: 'incapable', instance: {baseUrl: 'ppp', name: 'pikachu'}}];

    function assertNoIncapableInstanceInError() {
        var content = view.getContentContainer(container);
        var errorInstances = content.find(".instance");
        ok(errorInstances.length === 2);
        equal(errorInstances.eq(0).text(), "squirtle");
        equal(errorInstances.eq(1).text(), "charmander");
        ok(content.text().indexOf("pikachu") === -1, "There is no mention of the incapable instance at all in detail dialog");
    }

    _.each([true, false], function(hasDataFlag) {
        _.each([true, false], function(ajaxFailedFlag) {
            view._renderError(container, errorData, hasDataFlag, ajaxFailedFlag);
            assertNoIncapableInstanceInError();
            view.getContentContainer(container).empty();
        });
    });
});
