Backbone.define('JIRA.DevStatus.BaseDetailDialogModel', Backbone.Model.extend({
    TAB_PREFIX: "#tab-menu-",
    PANE_PREFIX: "#tab-content-",

    properties: [
        "tabs",
        "contentMap", //map of instance id to content json
        "oauthStatuses", // authentication status of each remote instance
        "selectedTab",
        "selectedPane"
    ],

    namedEvents: [
        // events related to detailed data rest call
        "fetchRequested",
        "fetchFailed",
        "fetchSucceeded",
        // events related to auth status rest call
        "fetchAuthRequested",
        "fetchAuthFailed",
        "fetchAuthSucceeded"
    ],

    initialize: function (data) {
        this.issueKey = data.issueKey;
        this.issueId = data.issueId;
        this.type = data.dataType;
        this.set("contentMap", {});
        this.set("tabs", data.tabs);
    },

    switchTab: function(applicationType, forceFetch) {
        this.set("selectedTab", this.TAB_PREFIX + applicationType);
        this.set("selectedPane", this.PANE_PREFIX + applicationType);

        if (this.get("contentMap")[applicationType] == null || forceFetch) {
            this._fetchContents(applicationType);
        }
    },

    _fetchContents: function(applicationType) {
        this.trigger("fetchRequested", applicationType);
        this._handleFetch(applicationType, AJS.$.ajax({
            url: AJS.contextPath() + "/rest/dev-status/1.0/issue/detail",
            data: {
                issueId: this.issueId,
                applicationType: applicationType,
                dataType: this.type
            }
        }).promise());
    },

    _handleFetch: function(applicationType, fetchXhr) {
        var instance = this;
        //noinspection JSUnusedLocalSymbols
        fetchXhr.done(function(result) {
            instance._fireSuccessResult(applicationType, result);
        }).fail(function(xhr, status) {
            instance._fireFailResult(applicationType, xhr);
        });
    },

    _fireSuccessResult: function(applicationType, result) {
        this._updateContents(applicationType, result);
        this.trigger("fetchSucceeded", applicationType, result);
    },

    _fireFailResult: function(applicationType, xhr) {
        this._updateContents(applicationType);
        this.trigger("fetchFailed", applicationType, xhr);
    },

    _updateContents: function(applicationType, content) {
        this.get("contentMap")[applicationType] = content;
    },

    /**
     * Initiate the rest call to fetch authentication statuses based on the application types from the current tabs.
     */
    fetchAuthenticationStatuses: function() {
        var applicationTypes = _.keys(this.get("tabs") || {});
        if (!_.isEmpty(applicationTypes)) {
            var instance = this;
            //noinspection JSUnusedLocalSymbols
            AJS.$.ajax({
                        url: AJS.contextPath() + "/rest/dev-status/1.0/provider/auth-status",
                        data: {
                            applicationType: applicationTypes
                        }}).promise()
                    .done(function(result) {
                        if (result && result.data) {
                            instance._updateOAuthStatuses(result.data);
                            instance.trigger("fetchAuthSucceeded", result.data);
                        } else {
                            instance._updateOAuthStatuses();
                            instance.trigger("fetchAuthFailed", null, result);
                        }
                    })
                    .fail(function(xhr, status) {
                        instance._updateOAuthStatuses();
                        instance.trigger("fetchAuthFailed", xhr);
                    });
        }
    },

    /**
     * Handle the change of oauth status of an instance belonging to an application type.
     * If it's the current tab, triggers the refresh of the current tab by starting the fetch of detailed data.
     * If it's not the current tab but it's content is cached, invalidate the cache.
     */
    oauthStatusChanged: function(applicationType) {
        if (_.has(this.get("tabs"), applicationType)) {
            var currentApplicationType = this._getCurrentApplicationType();
            if (currentApplicationType === applicationType) {
                this._fetchContents(currentApplicationType);
            } else {
                // clear the cache, regardless of whether it's there
                this.get("contentMap")[applicationType] = null;
            }
        }
    },

    _getCurrentApplicationType: function() {
        var selectedTab = this.get("selectedTab");
        if (selectedTab) {
            return selectedTab.slice(this.TAB_PREFIX.length);
        }
        return null;
    },

    _updateOAuthStatuses: function (content) {
        this.set("oauthStatuses", content);
    }
}));
