/**
 * @class JIRA.DevStatus.DevStatusModule
 * @extends Backbone.Model
 */
Backbone.define('JIRA.DevStatus.DevStatusModule', Backbone.Model.extend({
    livestampRelativizeThreshold: 1000 * 60 * 60 * 24 * 7, // one week

    initialize: function (options) {
        _.defaults(this, options);

        this._listeners = new JIRA.DevStatus.Util.Listeners();
        this.createBranchFormDialog = new JIRA.DevStatus.CreateBranchFormDialog(options);
        this.devSummaryJson = null;
    },

    /**
     *
     * @param options
     * @return {JIRA.DevStatus.DevStatusModule} this
     */
    initializePanel: function (options) {
        // stop listening to any events that this module has registered for (before they become garbage)
        this._listeners.stopListening();

        this._getContainers(options);
        this.labsOptIn = JIRA.DevStatus.Labs.LabsOptInModel.createFromDOM(this.parentContainer);
        // On page load, only show tooltip if it hasn't been shown & the opt in to labs panel
        // is not going be shown.
        options.showTooltip = options.showTooltip && !(this.labsOptIn.isAllowed() && !(this.labsOptIn.isOptedIn() || this.labsOptIn.isDismissed()));
        this._createCreateBranchView(options);

        this.devSummaryJson = this.devStatusContainer.find('.dev-summary.json-blob').data('json');
        this._initLabsOptInView();
        this._initTryLabsView();

        this._createDevStatusData(options);
        this._createAnalyticsModel(options);
        this._createSummaryViews(options);

        this._setupDataEventHandlers();

        this.devStatusPanelVisibilityToggler = options.devStatusPanelVisibilityToggler;

        if (!options.phaseTwoDisabled) {
            // only show cached summary or start to fetch summary when phase 2 is enabled
            this._startDevStatus(); // kick start phase 2 features
        } else {
            // if phase 2 is not enabled, finish the show/hide check immediately
            //  no animation, and is final
            this._postRender(false, true);
        }
        this.labsOptIn.on("change:optedIn", this._onChangeLabsOptIn, this);
        if (this.labsOptIn.isAllowed() && this.labsOptIn.isOptedIn()) {
            this._onChangeLabsOptIn(this.labsOptIn, true);
        }

        return this;
    },

    _createCreateBranchView: function(options) {
        this.createBranchView = new JIRA.DevStatus.CreateBranchView({
            el: this.createBranchContainer,
            showTooltip: options.showTooltip
        });
    },

    _createDevStatusData: function(options) {
        this.devStatusData = new JIRA.DevStatus.DevStatusData({
            issueId: JIRA.Issue.getIssueId()
        });
    },

    _createSummaryViews: function(options) {
        var that = this;
        var dataAttrs = this.devStatusContainer.data();
        this._summaryModules = _.clone(this.statusPanelContainer.find('.status-panel').map(function() {
            var $el = AJS.$(this);
            return new JIRA.DevStatus[$el.data('module')]({
                livestampRelativizeThreshold: that.livestampRelativizeThreshold,
                dataAttrs: dataAttrs, // pass all top level data attributes for dev status
                el: $el,
                labsOptIn: that.labsOptIn,
                analyticsModel: that.analyticsModel
            });
        }));

        this._errorModule = new JIRA.DevStatus.SummaryErrorModule({
            el: this.messagePanelContainer,
            dataAttrs: dataAttrs
        });
    },

    _createAnalyticsModel: function(options) {
        this.analyticsModel = new JIRA.DevStatus.AnalyticsModel({
            devStatusContainer: this.devStatusContainer,
            devStatusData: this.devStatusData
        });
    },

    _getContainers: function(options) {
        this.devStatusContainer = options.container;
        this.parentContainer = options.container.closest(this.parentContainerSelector);
        this.createBranchContainer = options.container.find(this.linkSelector);
        this.statusPanelContainer = options.container.find(this.statusPanelSelector);
        this.headerContainer = this.parentContainer.find(options.headerSelector);
        this.messagePanelContainer = options.container.find(".message-panel");
    },

    _onLoadingStarted: function() {
        this._startLoadingModuleData();
    },

    _onLoadingSuccess: function(devStatusData, isCache, isFinal) {
        this._renderModules({
            success: true,
            renderData: this.devStatusData.get("aggregatedData")
        });
        this._postRender(!isCache, isFinal);
    },

    _onLoadingFailure: function() {
        this._renderModules({
            success: false,

            // FUSE-1026: it's better to display stale data rather than have a blank development panel
            renderData: this.devSummaryJson ? this.devSummaryJson.cachedValue : undefined
        });
        this._postRender(false, true);
    },

    /**
     * Current model is have a single spinner controlled by DevStatusModule.
     *
     * When data comes back from DevStatusData, pass it along from one view to the next.
     * The order of calling these view is hardcoded
     */
    _setupDataEventHandlers: function() {
        this._listeners.startListening(this.devStatusData, "beforeRequest", this._onLoadingStarted, this);
        this._listeners.startListening(this.devStatusData, "requestSuccess", this._onLoadingSuccess, this);
        this._listeners.startListening(this.devStatusData, "requestFail", this._onLoadingFailure, this);
    },

    /**
     * Kick start the data fetching from rest end point.
     * Result will be handled by the event handlers configured in _setupDataEventHandlers.
     * @private
     */
    _startDevStatus: function() {
        var needAjax = !this.devSummaryJson || this.devSummaryJson.isStale === true;
        // always update the page with the cached summary if it is there
        if (this.devSummaryJson) {
            this.devStatusData.setAggregateData(this.devSummaryJson.cachedValue, true, !needAjax);
        }

        if (needAjax) {
            this._startLoadingModuleData();
            this.devStatusData.retrieveAggregateData();
        }
    },

    _onChangeLabsOptIn: function(model, labsOptedIn) {
        if (labsOptedIn === true) {
            // enable labs features
        } else {
            // hide labs features
        }
    },

    _initLabsOptInView: function() {
        this.labsOptInView = new JIRA.DevStatus.Labs.LabsOptInView({
            el: this.parentContainer.find('.labs-on-off-container'),
            labsOptIn: this.labsOptIn
        }).render();
    },

    _initTryLabsView: function() {
        // slightly hackish, but this part of JIRA is not pluggable so we need
        // to append our own container here so we can then keep the view clean
        var tryLabsContainer = AJS.$('<div class="try-labs-container"></div>').appendTo(this.headerContainer);

        new JIRA.DevStatus.Labs.TryLabsView({
            el: tryLabsContainer,
            labsOptIn: this.labsOptIn
        }).render();
    },

    /**
     * Hide or show the dev status panel depending on if there is no visible data. Fires analytics events.
     *
     * @private
     */
    _postRender: function(animateToggling, isFinal) {
        var isAnythingSummaryRendered = this._isAnySummaryViewVisibleOnPanel() || this._isAnyErrorViewVisible();
        if (isAnythingSummaryRendered) {
            var issueAnalytic = this.analyticsModel.getIssue();
            JIRA.DevStatus.SummaryAnalytics.fireSummaryShownEvent(
                issueAnalytic.isAssignee,
                issueAnalytic.issueType,
                issueAnalytic.issueStatus,
                this.analyticsModel.getSummary()
            );
            this._openDetailDialogUrlLink();
        }

        this._togglePanelEmptyStatus(!this._isAnySummaryViewVisibleOnPanel(true));

        var container = this.parentContainer;
        function markPanelAsFinal() {
            if (isFinal) {
                // used by test to be certain that the panel is done with animation and good for assertThis assertThat
                container.addClass('js-animation-completed');
            }
        }
        // check if anything is rendered on the panel, including any summary view or cta (e.g., create branch)
        var isAnythingRendered = isAnythingSummaryRendered || (this.createBranchContainer.size() > 0);
        if (this.devStatusPanelVisibilityToggler && (container.is(':visible') !== isAnythingRendered)) {
            // toggle the visibility of the dev status panel, animate it if not cache
            this.devStatusPanelVisibilityToggler(this.parentContainer, isAnythingRendered, animateToggling, markPanelAsFinal);
        } else {
            markPanelAsFinal();
        }
    },

    _getModuleByType: function (dialogToOpen) {
        return _.find(this._summaryModules, function (module) {
            return module.data && module.data.getType && _.isEqual(module.data.getType(), dialogToOpen);
        });
    },

    _openDetailDialogUrlLink: function() {
        var dialogToOpen = AJS.Meta.get("fusion-open-detail-dialog");
        if (dialogToOpen) {
            var moduleToClick;
            if (JIRA.DevStatus.URL.isCreateReviewDetailDialogLink(dialogToOpen)) {
                // preserve META so the commit dialog can continue with creating the review
                moduleToClick =  this._getModuleByType('repository');
            } else {
                moduleToClick = this._getModuleByType(dialogToOpen);
                if (moduleToClick) {
                    AJS.Meta.set("fusion-open-detail-dialog", undefined);
                }
            }

            if (moduleToClick) {
                var summaryViewToClick = moduleToClick.view;
                if (summaryViewToClick && summaryViewToClick.isVisible()) {
                    AJS.Meta.set("fusion-analytics-new-context-link", true);
                    summaryViewToClick.getSummaryLink().click();
                }
            }
        }
    },

    /**
     * Toggle the empty-status class on the status panel container to indicate whether anything is being rendered into
     * the status panel
     *
     * @param toggle flag to set
     */
    _togglePanelEmptyStatus: function(toggle) {
        if (this.statusPanelContainer) {
            this.statusPanelContainer.toggleClass("empty-status", toggle);
        }
    },

    /**
     * Returns whether any summary module is rendered in the development panel.
     *
     * @return {boolean}
     * @private
     */
    _isAnySummaryViewVisibleOnPanel: function() {
        // return true if any of the summary views is visible
        return !!_.find(this._summaryModules, function(module) {
            return module.isViewVisible();
        });
    },

    /**
     * Returns whether any errors are visible in the development panel.
     *
     * @returns {boolean}
     * @private
     */
    _isAnyErrorViewVisible: function() {
        return this._errorModule.isViewVisible();
    },

    /**
     * Calls <code>render</code> on each of of this DevStatusModule's summary modules, passing in
     * <code>fetchResult.summaryData</code>. Use <code>fetchResult.success</code> to signal that there was an error
     * fetching data from the server.
     *
     * @param {object}  fetchResult
     * @param {boolean} fetchResult.success if the fetch operation was successful
     * @param {object}  fetchResult.renderData the summary data to render
     * @return {*}
     * @private
     */
    _renderModules: function(fetchResult) {
        _.each(this._summaryModules, function(module) {
            module.render(fetchResult.renderData);
        });

        this._errorModule.render(fetchResult.success ? fetchResult.renderData : undefined)
    },

    /**
     * Calls <code>startLoading()</code> on each module.
     *
     * @private
     */
    _startLoadingModuleData: function() {
        _.each(this._summaryModules, function(module) {
            module.startLoading();
        });
    }
}));
