/**
 * The purpose of this model is to keep track of all related analytics data that can be passed around from
 * one module to another
 */
Backbone.define('JIRA.DevStatus.AnalyticsModel', Backbone.Model.extend({
    /*
     Override for conversion from SummaryModel typeId to analytic key
     */
    TO_ANALYTIC_KEY_OVERRIDE: {
        repository: "commit"
    },

    properties: [
        /*
         Contains data related to the issue. This currently includes:
          - isAssignee
          - isAssignable
          - issueStatus
          - issueType
         */
        "issue",

        /*
         Contains data related to dev status summary. This includes data on the existence of certain information.
         For example:
         {
            commit: true,
            branch: true,
            pullrequest: true
         }
         Anything that is not included in the above, is considered to be non-existence in the summary panel
         */
        "summary"
    ],

    /**
     * @param options
     * @param options.devStatusContainer {Element} jQuery element for the dev status container
     * @param options.devStatusData {JIRA.DevStatus.DevStatusData}
     *          Analytics model need listen to changes to aggregate data and calculate necessary summary
     */
    initialize: function (options) {
        this.devStatusContainer = options.devStatusContainer;
        this.devStatusData = options.devStatusData;
        this._getAllAnalyticData();
    },

    getIssue: function() {
        return this.get("issue");
    },

    getSummary: function() {
        return this.get("summary");
    },

    _getAllAnalyticData: function() {
        this.set("issue", this._getAnalyticIssueData());
        this._initListenerToData();
    },

    _getAnalyticIssueData: function() {
        return {
            isAssignee: this.devStatusContainer.data("is-assignee"),
            isAssignable: this.devStatusContainer.data("is-assignable"),
            issueStatus: this.devStatusContainer.data("issue-status"),
            issueType: this.devStatusContainer.data("issue-type")
        };
    },

    _getAnalyticSummaryData: function(aggregateData) {
        var instance = this;
        var summary = {};
        _.each(aggregateData.summary, function(value, key) {
            // Convert {branch: {overall: {count: 5}}} into {branch: true}
            summary[instance.TO_ANALYTIC_KEY_OVERRIDE[key] || key] = value.overall.count > 0;
        });
        return summary;
    },

    _initListenerToData: function() {
        var instance = this;
        this.devStatusData.on("requestSuccess", function() {
            instance.set(
                "summary",
                instance._getAnalyticSummaryData(instance.devStatusData.get("aggregatedData"))
            );
        });
    }
}));
