/**
 * A fair amount of the autocomplete control code was taken from Chris Darroch's work in these places:
 *
 * Autocomplete example: https://bitbucket.org/chrisdarroch/autocomplete-controls/src/9021ccf4c7de9349349e67799ff9f8b65cc5f66c/views/index.erb?at=master
 * JIRA @mentions: jira-components/jira-webapp/src/main/webapp/includes/jira/mention/Mention.js
 */
define('widget/user-picker/overlay-view', [
    './picker-overlay.less',
    './picker-overlay.soy',
    'underscore',
    'zepto',
    'backbone',
    'feature/users/user-model',
    'util/presenter',
    'require',
],
function (
    css,
    Templates,
    _,
    $,
    Backbone,
    UserModel,
    presenter,
    require
) {
    const ProgressiveDataSet = require('@atlassian/aui/src/js/aui/progressive-data-set').default;
    const ResultSet = require('@atlassian/aui/src/js/aui/result-set').default;
    const ResultsList = require('@atlassian/aui/src/js/aui/results-list').default;
    const QueryInput = require('@atlassian/aui/src/js/aui/query-input').default;

    const MESSAGES = {
        401: AJS.I18n.getText('jira.mobile.error.user.picker.logged.out'),
        loggedout: AJS.I18n.getText('jira.mobile.error.user.picker.logged.out'),
        'default': AJS.I18n.getText('jira.mobile.error.user.picker.failure')
    };

    const OverlayView = Backbone.View.extend({
        className: 'overlay-content user-picker-overlay',

        events: {
            'submit form': 'returnUser',
            'click .cancel': 'hide',
            'click .results-list-container li': 'preventDefault',
            'click .input-clear': 'clearInput',
            'touchend .results-list-container': 'keepFocus',
            'touchstart .input-clear': 'keepFocus',
            'touchend .input-clear': 'keepFocus',
            'touchend .main-action[disabled]': 'keepFocus',
            'blur #user-picker-name': 'inputBlur',
            'keydown #user-picker-name': 'inputKeydown'
        },

        // Picked up the Presenter to make sure focus is fired after this view has been added to the DOM
        autofocus: null,

        initialize: function (options) {
            const instance = this;
            _.bindAll(this);

            this.callback = (options || {}).callback || function () {};
            this.cancelCallback = (options || {}).cancelCallback || function () {};
            this.user = options.defaultUser || null;
            this.extraUsers = options.extraUsers || [];
            this.fallbackUser = null;
            if (this.extraUsers.length) {
                let fallback = _.filter(this.extraUsers, function (model) {
                    return model.isDefaultFallback === true;
                }) || null;
                if (fallback && fallback.length && !(fallback[0] instanceof  Backbone.Model)) {
                    fallback = new UserModel(fallback[0]);
                }
                this.fallbackUser = fallback;
            }
            this.isTyping = false;
            this.scrollTop = window.scrollY;

            // Other parts of JMob auto-scroll to window top when the device is rotated.
            // Make sure the user picker goes to the top as well.
            $(window).on('orientationchange', this.resetScrollTop);

            // Set up autocomplete components
            const bootstrapUsers = this.extraUsers.concat(this.user || []);
            const dataSet = this.dataSet = new ProgressiveDataSet(bootstrapUsers, {
                model: UserModel,
                queryEndpoint: options.dataSrc,
                queryParamKey: 'username'
            });
            dataSet.ajaxOptions = {showLoading: false};
            dataSet.matcher = function(model, query) {
                let matches = false;
                matches = matches || instance._stringPartStartsWith(model.get("name"), query);
                matches = matches || instance._stringPartStartsWith(model.get("displayName"), query);
                return matches;
            };
            /**
             * Override the dataSet fetch method so it works with Backbone 0.9.10 and has better error handling
             * Changed {add:true} to {update:true, replace:false} and added {error: function}
             */
            dataSet.fetch = function(query) {
                const data = this.getQueryData(query);
                const params = {
                    update: true,
                    remove: false,
                    data: data,
                    error: function (collection, xhr) {
                        xhr.errorHandled = true;
                        // If the response didn't contain an error field, then let's assume it was a connection error
                        const appData = require('util/app-data');
                        let message;
                        // The "Assignable user" endpoint returns 404 if the user is not logged in and they can't see the issue.
                        // Need to make sure the HTTP header checks have already been done by util/ajax first though.
                        _.defer(function () {
                            if (xhr.status == 404 && !appData.getUsername()) {
                                message = MESSAGES.loggedout;
                            } else {
                                message = MESSAGES[xhr.status] || MESSAGES['default'];
                            }
                            queryResult.$el.html(Templates.error({
                                error: message
                            })).show();
                        });
                    }
                };
                const remote = Backbone.Collection.prototype.fetch.call(this, params);
                return remote;
            };

            this.queryInput = null; // Created during render()
            const resultSet = this.resultSet = new ResultSet({
                source: dataSet
            });

            const queryResult = this.queryResult = new ResultsList({
                model: resultSet,
                className: 'aui-dropdown2 aui-style-default'
            });
            // Override the default implementation to avoid view.make() which isn't in our version of Backbone
            queryResult.render = function() {
                const listData = resultSet.collection.map(function (model) {
                    return model.toJSON();
                });
                this.$el.html(Templates.suggestionList({
                    users: listData
                }));
                return this;
            };
            // Override to not show results when text input is empty
            queryResult._shouldShow = function(query) {
                return query !== '';
            }
            queryResult.on('selected', function (model) {
                queryResult.hide();
                instance.user = model;
                instance.setTyping(false);
                instance.updateDisplay();
                window.scrollTo(0, instance.scrollTop);
            });
        },

        render: function () {
            this.$el.empty();

            const opts = {
                titleText: this.options.titleText,
                actionText: this.options.actionText,
                cancelText: this.options.cancelText
            };
            if (this.user) {
                opts.defaultUser = this.user.toJSON();
            }
            this.$el.html(Templates.overlay(opts));
            this.$el.css('top', this.scrollTop || 0);

            // Cache some common references
            this.$container = this.$('.picker-input-container');
            this.$input = this.$('#user-picker-name');
            this.$avatar = this.$container.children('.user-avatar');
            this.$clear = this.$container.children('.input-clear');
            this.$submit = this.$('.main-action');

            // Set up the autocomplete
            this.queryInput = new QueryInput({
                el: this.$el.find('#user-picker-name')
            });
            this.queryInput.on('change', this.dataSet.query);
            this.$el.find('.results-list-container').append(this.queryResult.el);

            this.updateDisplay();
            this.autofocus = this.$input;

            return this;
        },

        /**
         * Disable any non-overlay form fields, so the next/previous form buttons on iOS are disabled
         */
        onShow: function () {
            this.$pageFields = $('form').find(':enabled');
            this.$pageFields.prop('disabled', true);
        },

        /**
         * Re-enable any previously disabled form fields
         */
        onClose: function () {
            this.$pageFields.prop('disabled', false);
            this.$pageFields = null;
            $(window).off('orientationchange', this.resetScrollTop);
        },

        resetScrollTop: function () {
            this.scrollTop = 0;
            this.$el.css('top', this.scrollTop);
        },

        refreshScrollTop: function () {
            this.scrollTop = window.scrollY;
            this.$el.css('top', this.scrollTop);
        },


        /*** Autocomplete methods and handlers ***/

        setTyping: function (isTyping) {
            isTyping = !!isTyping;
            if (isTyping !== this.isTyping) {
                this.$container.toggleClass('is-typing', isTyping);
            }
            this.isTyping = isTyping;
            this.$submit.prop('disabled', isTyping || !this.user || this.$input.val() === '');
        },

        _stringPartStartsWith: function (text, startsWith) {
            text = $.trim(text || "").toLowerCase();
            startsWith = (startsWith || "").toLowerCase();
            const nameParts = text.split(/\s+/);

            if (!text || !startsWith) return false;
            if (text.indexOf(startsWith) === 0) return true;

            return _.any(nameParts, function(word) {
                return word.indexOf(startsWith) === 0;
            });
        },


        /*** UI state controls ***/

        updateDisplay: function (options) {
            options = options || {};
            const hasUser = !!this.user;
            if (hasUser) {
                this.$input.val(this.user.get('displayName'));
                this.$avatar.attr('src', this.user.get('avatarUrls')['48x48']);
            } else if (options.clearInput !== false) {
                this.$input.val('');
            }
            this.$submit.prop('disabled', !hasUser);
            this.$container.toggleClass('has-user', hasUser);
        },

        checkValidUser: function () {
            const text = this.$input.val();
            // Default to Automatic if empty
            if (this.fallbackUser && text === '') {
                this.user = this.fallbackUser;
                return true;
            }
            // If field is not empty, make sure what the user has typed is a valid user
            let foundModel;
            const compText = text.toLowerCase();
            // First find any cached user models matching the text
            // For this purpose we're assuming that any matching user models have already been loaded
            const userModels = this.dataSet.filter(function (model) {
                const userName = model.get('name').toLowerCase();
                const displayName = model.get('displayName').toLowerCase();
                return (compText === userName || compText === displayName);
            });
            if (userModels.length) {
                // If there's more than one matching user, prefer username over display name
                if (userModels.length > 1) {
                    foundModel = _.find(userModels, function (model) {
                        return compText === model.get('name').toLowerCase();
                    });
                }
                // If there's only one user, OR
                // If no user had an exact username match (i.e. duplicate display names), pick the first one
                if (!foundModel) {
                    foundModel = userModels[0];
                }
            }
            // Clear out any cached user if no matching model could be found, this will cause the main action
            // button to be disabled
            if (!foundModel) {
                this.user = null;
                return false;
            }
            this.user = foundModel;
            return true;
        },


        /*** UI event handlers ***/

        /**
         * Return a user object to the caller that created the overlay
         */
        returnUser: function (e) {
            e.preventDefault();
            this.callback(this.user);
            presenter.hideOverlay();
        },

        /**
         * Hide the overlay with no further action
         */
        hide: function (e) {
            e.preventDefault();
            this.cancelCallback();
            presenter.hideOverlay();
        },

        preventDefault: function (e) {
            e.preventDefault();
        },

        keepFocus: function (e) {
            this.shouldKeepFocus = true;
        },

        clearInput: function (e) {
            e.preventDefault();
            this.$input.val('').focus();
            this.setTyping(true);
            this.queryResult.hide();
        },

        inputBlur: function (e) {
            const that = this;
            if (this.shouldKeepFocus) {
                e.preventDefault();
                this.$input.focus();
                _.defer(function () {
                    that.shouldKeepFocus = false;
                });
            } else {
                this.setTyping(false);
                this.checkValidUser();
                this.updateDisplay({
                    clearInput: false
                });
                _.defer(function () {
                    that.queryResult.hide();
                });
            }
        },

        inputKeydown: function (e) {
            _.defer(this.inputType);
        },

        inputType: function (e) {
            const value = this.$input.val();
            if (!this.user || value !== this.user.get('displayName')) {
                this.setTyping(true);
            }
            if (value === '') {
                this.queryResult.hide();
            }
        }
    });

    return OverlayView;
});
