AJS.test.require("com.atlassian.jira.jira-projects-plugin:subnavigator-test", function() {
    "use strict";

    var jQuery = require("jquery");
    var _ = require("underscore");
    var Items = require("jira/projects/components/subnavigator/entities/items");
    var Content = require("jira/projects/components/subnavigator/views/content");

    var ITEM_1 = { id: "item1", label: "Item 1", description: "Item 1" };
    var ITEM_2 = { id: "item2", label: "Item 2", description: "Item 2" };
    var ITEM_3 = { id: "item3", label: "Item 3", description: "Item 3" };

    var TIMEOUT = 300;

    var ID = "id";

    module("jira/projects/components/subnavigator/views/content", {
        setup: function() {
            this.$el = jQuery("<div/>").appendTo('#qunit-fixture');

            jQuery('#qunit-fixture').append("<span class='tipsy'/>");

            this.model = new Items({itemGroups: [[ITEM_1]]});
            this.view = new Content({
                el: this.$el,
                model: this.model,
                id: ID,
                tooltipTimeout: TIMEOUT
            });
        },

        teardown: function() {
            if (this.clearTimeoutSpy) {
                this.clearTimeoutSpy.restore();
            }
        },

        assertScheduledTooltipsAreCanceled: function(timeoutId) {
            sinon.assert.calledOnce(this.clearTimeoutSpy);
            sinon.assert.calledWith(this.clearTimeoutSpy, timeoutId);
        }
    });

    test("Removes the selected item from the list when rendering its template if 'hideSelectedItem' is true or undefined", function() {
        this.sandbox.useFakeTimers();

        this.model.set("itemGroups", [[ITEM_1, ITEM_2],[ITEM_3]]);
        this.model.selectItem(ITEM_2.id);

        var itemGroups = this.view.serializeData().itemGroups;

        deepEqual(itemGroups, [[ITEM_1], [ITEM_3]]);
    });

    test("Marks the selected item as selected when rendering its template if 'hideSelectedItem' is false", function() {
        this.sandbox.useFakeTimers();
        var view = new Content({
            el: this.$el,
            model: this.model,
            id: ID,
            tooltipTimeout: TIMEOUT,
            hideSelectedItem: false
        });

        this.model.set("itemGroups", [[ITEM_1, ITEM_2],[ITEM_3]]);
        this.model.selectItem(ITEM_2.id);

        var itemGroups = view.serializeData().itemGroups;

        var selectedItem2 = _.clone(ITEM_2);
        selectedItem2.selected = true;
        deepEqual(itemGroups, [[ITEM_1, selectedItem2 ], [ITEM_3]]);
    });

    test("It renders dropdown items as normal links when 'hideSelectedItem' is true", function() {
        this.sandbox.useFakeTimers();
        var view = new Content({
            el: this.$el,
            model: this.model,
            id: ID,
            tooltipTimeout: TIMEOUT,
            hideSelectedItem: true
        });

        this.model.set("itemGroups", [[ITEM_1],[ITEM_3]]);

        view.render();

        notOk(view.$el.find("a").hasClass("aui-dropdown2-radio"),
            "When 'hideSelectedItem' is true dropdown links shouldn't have 'aui-dropdown2-radio' class"
        );

    });

    test("When an Item is clicked on the Dropdown2 component triggers the 'click' event", function() {
        this.sandbox.useFakeTimers();

        var onClick = this.spy();
        this.view.listenTo(this.view, "click", onClick);

        this.view.render();
        this.view.$el.find("a").trigger("click");

        sinon.assert.calledOnce(onClick);
        sinon.assert.calledWith(onClick, ITEM_1.id);
    });

    test("When an Item is checked on the Dropdown2 component removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();

        this.view.$el.find("a").trigger("aui-dropdown2-item-check");

        equal(jQuery(".tipsy").length, 0);
    });

    test("Select on an item prevents scheduled tooltips to be displayed", function() {
        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("a").trigger("click");

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse leaving an li element removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();

        this.view.$el.find("li").mouseleave();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse leaving an li element prevents scheduled tooltips to be displayed", function() {
        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("li").mouseleave();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering a li element removes previous tooltips that were shown", function() {
        this.sandbox.useFakeTimers();
        this.view.render();

        this.view.$el.find("li").mouseenter();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse entering a li element cancels previous tooltips that were scheduled to be shown", function() {
        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.find("li").mouseenter();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering an li element shows a tooltip", function() {
        var clock = this.sandbox.useFakeTimers();

        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.view.$el.find("li").mouseenter();

        clock.tick(TIMEOUT);

        equal(jQuery(".tipsy").length, 1);
    });

    test("Hiding the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();

        this.view.$el.trigger("aui-dropdown2-hide");

        equal(jQuery(".tipsy").length, 0);
    });

    test("Hiding the view prevents scheduled tooltips to be displayed", function() {
        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$el.trigger("aui-dropdown2-hide");

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Rendering the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Rendering the view prevents scheduled tooltips to be displayed", function() {
        this.view.tooltipTimer = "timerId";
        this.view.render();

        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.render();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Uses the given id as a suffix when rendering the template", function() {
        this.sandbox.useFakeTimers();

        this.view.render();

        equal(this.view.$el.attr("id"), "subnav-opts-" + ID);
    });
});
