AJS.test.require("com.atlassian.jira.jira-projects-plugin:pinnablenavigator-test", function() {
    "use strict";

    var jQuery = require("jquery");

    var Items = require("jira/projects/components/pinnable-navigator/entities/items");
    var UnpinnedView = require("jira/projects/components/pinnable-navigator/views/navigator-view/unpinned-view");
    var Search = require("jira/projects/components/pinnable-navigator/views/search/search");
    var DarkFeatureModule = require("jira/projects/components/pinnable-navigator/dark-feature");

    var ITEM_1 = { id: "item1", label: "Item 1", description: "Item 1" };
    var ITEM_4 = { id: "item4", label: "Item 4", description: "Item 4", count: 5 };
    var LONGLIST = [ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_1,ITEM_4,ITEM_4,ITEM_4];

    var TIMEOUT = 500;

    var ID = "id";

    module("jira/projects/components/pinnable-navigator/views/navigator-view/unpinned-view", {
        setup: function() {
            jQuery('#qunit-fixture').append("<span class='tipsy'/>");

            this.$el = jQuery("<div/>");

            jQuery('#qunit-fixture').append(this.$el);
            this.clock = sinon.useFakeTimers();

            this.model = new Items({itemGroups: [{items: [ITEM_4]}]});
            this.view = new UnpinnedView({
                el: this.$el,
                model: this.model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage",
                emptyText: "Nothing here",
                searchView: new Search({
                    searchPlaceHolder: "Search...",
                    el: ".js-subnav-search-block",
                    model: this.model
                })
            });
        },
        getViewWithDeleteOptionOnly: function(){
            var model = new Items({itemGroups: [{allowDelete: true, items: [ITEM_4]}]});
            return new UnpinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage"
            });
        },
        getViewWithReorderOptionOnly: function(){
            var model = new Items({itemGroups: [{allowReorder: true, items: [ITEM_4]}]});
            return new UnpinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title"
            });
        },
        getViewWithoutAnyOptions: function(){
            var model = new Items({itemGroups: [{items: [ITEM_4]}]});
            return new UnpinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage"
            });
        },
        getViewWithButtonsList: function(){
            var model = new Items({itemGroups: [{type: "buttons", items: [ITEM_4]}]});
            return new UnpinnedView({
                el: this.$el,
                model: model,
                id: ID,
                tooltipTimeout: TIMEOUT,
                title: "Title",
                manageText: "Manage"
            });
        },
        teardown: function() {
            if (this.clearTimeoutSpy) {
                this.clearTimeoutSpy.restore();
            }
            this.clock.restore();
        },

        assertScheduledTooltipsAreCanceled: function(timeoutId) {
            sinon.assert.calledOnce(this.clearTimeoutSpy);
            sinon.assert.calledWith(this.clearTimeoutSpy, timeoutId);
        }
    });

    test("Clicking on an item triggers the 'itemSelected' event", function() {
        this.sandbox.useFakeTimers();

        var onClick = this.spy();
        this.view.listenTo(this.view, "itemSelected", onClick);

        this.view.render();
        this.view.putListintoView();
        this.view.$(".aui-dropdown2-section:first-child a").click();

        sinon.assert.calledOnce(onClick);
    });

    test("Clicking on an item removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();
        this.view.$(".aui-dropdown2-section:first-child a").click();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Clicking on an item prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$(".aui-dropdown2-section:first-child a").click();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse leaving an li element removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();
        this.view.$(".aui-dropdown2-section:first-child li").mouseleave();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse leaving an li element prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$(".aui-dropdown2-section:first-child li").mouseleave();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering a li element removes previous tooltips that were shown", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();
        this.view.$("li").mouseenter();

        equal(jQuery(".tipsy").length, 0);
    });

    test("Mouse entering a li element cancels previous tooltips that were scheduled to be shown", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.$(".aui-dropdown2-section:first-child li").mouseenter();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Mouse entering an li element shows a tooltip", function() {
        var clock = this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        jQuery('#qunit-fixture').append(this.view.$el);

        this.view.$(".aui-dropdown2-section:first-child li").mouseenter();

        clock.tick(TIMEOUT);

        equal(jQuery(".tipsy").length, 1);
        this.view.$el.remove();
    });

    test("Hiding the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();
        this.view.$el.trigger("aui-dropdown2-hide");

        equal(jQuery(".tipsy").length, 0);
    });

    test("Hiding the view prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.listItemsView.$el.trigger("aui-dropdown2-hide");

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Rendering the view removes the tooltips", function() {
        this.sandbox.useFakeTimers();
        this.view.render();
        this.view.putListintoView();
        equal(jQuery(".tipsy").length, 0);
    });

    test("Rendering the view prevents scheduled tooltips to be displayed", function() {
        this.view.render();
        this.view.putListintoView();
        this.view.listItemsView.tooltipTimer = "timerId";
        this.clearTimeoutSpy = sinon.spy(window, "clearTimeout");
        this.view.listItemsView.render();

        this.assertScheduledTooltipsAreCanceled("timerId");
    });

    test("Uses the given id as a suffix when rendering the template", function() {
        this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();
        equal(jQuery("#subnav-opts-" + ID).length, 1);
    });

    test("Keep menu open button is appeared in the panel", function() {
        this.view.render();
        this.view.putListintoView();
        equal(this.view.$(".js-pin-toggler").length, 1);
    });

    test("Clicking on Keep menu open button will fire 'navigatorPinned' event", function() {
        var onClick = this.spy();
        this.view.listenTo(this.view, "navigatorPinned", onClick);

        this.view.render();
        this.view.putListintoView();
        this.view.$(".js-pin-toggler").click();

        sinon.assert.calledOnce(onClick);
    });

    test("The count number will be appeared at the right of the item text if the item has count attribute", function() {
        this.view.render();
        this.view.putListintoView();
        equal(this.view.$el.find(".navigator-item-count-number").length, 1);
        equal(this.view.$el.find(".navigator-item-count-number").text(), "5");
    });

    test("The count number won't be appeared at the right of the item text if the item hasn't count attribute", function() {
        this.model.set("itemGroups", [{items: [ITEM_1]}]);

        this.view.render();
        this.view.putListintoView();
        equal(this.$el.find(".navigator-item-count-number").length, 0);
    });

    test("Manage button will be shown if one of allowDelete and allowReorder are true", function() {
        var view;

        view = this.getViewWithDeleteOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-manage-button").length, 1);

        view = this.getViewWithReorderOptionOnly();
        view.render();
        view.putListintoView();
        equal(view.$(".js-manage-button").length, 1);

        view = this.getViewWithoutAnyOptions();
        view.render();
        view.putListintoView();
        equal(view.$(".js-manage-button").length, 0);
    });

    test("Button lists items are rendered as normal", function () {
        var view = this.getViewWithButtonsList();
        view.render();
        view.putListintoView();
        equal(view.$(".button-list").size(), 0);
        equal(view.$(".aui-button").size(), 0);
        equal(view.$("a").size(), 2);
    });

    test("Prevents default on click, if itemSelected event is prevented", function () {
        this.sandbox.useFakeTimers();

        this.view.listItemsView.listenTo(this.view.listItemsView, "itemSelected", function (e) {
            e.preventDefault();
        });

        this.view.render();
        this.view.putListintoView();
        var e = jQuery.Event( "click" );

        this.view.listItemsView.$(".aui-dropdown2-section:first-child a").trigger(e);

        ok(e.isDefaultPrevented(), "expectedDefault to be prevented");

    });

    test("Does NOT prevented default on click, if itemSelected event is NOT prevented", function () {
        this.sandbox.useFakeTimers();

        this.view.render();
        this.view.putListintoView();
        var e = jQuery.Event("click");

        this.view.$(".aui-dropdown2-section:first-child a").trigger(e);

        ok(!e.isDefaultPrevented(), "expectedDefault to be prevented");

    });


    test("Empty groups don't render", function () {
        this.sandbox.useFakeTimers();

        var model = new Items({
            itemGroups: [
                {items: [ITEM_4]},
                {items: []}
            ]
        });
        this.view = new UnpinnedView({
            el: this.$el,
            model: model,
            id: ID,
            tooltipTimeout: TIMEOUT,
            title: "Title",
            manageText: "Manage"
        });

        this.view.render();
        this.view.putListintoView();

        equal(this.view.$el.children(".aui-dropdown2-section").size(), 2);

    });

    test("Button item groups which don't have 'isSelectable' class don't trigger itemselected", function () {
        var view = this.getViewWithDeleteOptionOnly();
        view.render();
        var onClick = this.spy();
        view.listenTo(this.view, "itemSelected", onClick);
        view.$(".button-list a:not('.isSelectable')").click();
        sinon.assert.callCount(onClick, 0);
    });

    test("The search block will be appeared when there are more than ten items", function () {
        this.stub(DarkFeatureModule, 'isSearchEnabled').returns(true);

        this.view.render();
        this.view.renderSearch();
        this.view.putListintoView();
        ok(!this.view.$(".search-form").length, "Expect there is no search block");

        this.view.model.set("itemGroups", [{items:[ITEM_4], id: "group1"}, {items:LONGLIST, id: "group2"}]);
        this.view.render();
        this.view.renderSearch();
        this.view.putListintoView();

        jQuery('#qunit-fixture').append(this.view.$el);

        ok(this.view.$(".search-form").length, "Expect there is search block");

        this.view.$(".js-subnav-filter-text").val("item 1").trigger("keydown");

        this.clock.tick(10);
        ok(this.view.$(".js-item-link:not(.js-pin-toggler)").length === 4, "Expect there are four items (Item 1) are shown");

        ok(!this.view.$(".js-clear-query").hasClass("hidden"), "Expect there is a clear icon");

        this.view.$(".js-clear-query").click();

        this.clock.tick(10);
        ok(this.view.$(".js-subnav-filter-text").val() === "", "Expect there is empty text after clicking on clear icon");
        ok(this.view.$(".js-item-link:not(.js-pin-toggler)").length === 11, "Expect there are full list");
    });
});
