AJS.test.require("com.atlassian.jira.jira-projects-plugin:pinnablenavigator-test", function() {
    "use strict";

    var _ = require("underscore");
    var Backbone = require("jira-projects-backbone");

    var Items = require("jira/projects/components/pinnable-navigator/entities/items");

    var ITEM_1 = { id: "item1", label: "Item 1" };
    var ITEM_2 = { id: "item2", label: "Item 2" };
    var ITEM_3 = { id: "item3", label: "Item 3" };

    var ITEMS = [{items: [ITEM_1], id: "group1"}, {items: [ITEM_2], id: "group2"}];
    var ITEMSWITHBUTTONS = [{items: [ITEM_1]}, {items: [ITEM_2]}, {items: [ITEM_3], type: "buttons"}];

    function listener() {
        return _.extend({}, Backbone.Events);
    }

    module("jira/projects/components/pinnable-navigator/entities/items", {
        setup: function() {
            this.model = new Items({});
        }
    });

    test("Is empty when given list of items is undefined", function() {
        this.model.set("itemGroups", undefined);

        strictEqual(this.model.isEmpty(), true);
    });

    test("Is empty when given list of items is empty", function() {
        this.model.set("itemGroups", []);

        strictEqual(this.model.isEmpty(), true);
    });

    test("Is not empty when the list of items contains something", function() {
        this.model.set("itemGroups", ["item1"]);

        strictEqual(this.model.isEmpty(), false);
    });

    test("Keeps a reference to the selected item when there is one with the given id", function() {
        this.model.set("itemGroups", ITEMS);

        this.model.selectItem(ITEM_2.id);

        equal(this.model.get("selectedItem"), ITEM_2);
    });

    test("Sets the selected item to the first one if there is no item with the given id", function() {
        this.model.set("itemGroups", ITEMS);

        this.model.selectItem("nonexistentItem");

        equal(this.model.get("selectedItem"), ITEM_1);
    });

    test("Sets the selected item to undefined if there is no item with the given id and the list of items is empty", function() {
        this.model.set("itemGroups", []);

        this.model.selectItem(ITEM_1.id);

        equal(this.model.get("selectedItem"), undefined);
    });

    test("Sets the selected item to undefined if there is no item with the given id and the list of items is undefined", function() {
        this.model.set("itemGroups", undefined);

        this.model.selectItem(ITEM_1.id);

        equal(this.model.get("selectedItem"), undefined);
    });

    test("Triggers a 'change:selectedItem' event when an item is selected", function() {
        var spy = this.spy();

        listener().listenTo(this.model, "change:selectedItem", spy);

        this.model.set("itemGroups", ITEMS);

        this.model.selectItem(ITEM_2.id);

        sinon.assert.calledOnce(spy);
    });

    test("Returns a flat version of the items list when asked for all items", function() {
        this.model.set("itemGroups", ITEMS);

        var allItems = this.model.getAllItems();

        deepEqual(allItems, [ITEM_1, ITEM_2]);
    });

    test("Returns all real items (except buttons such as New Queue or New Report)", function() {
        this.model.set("itemGroups", ITEMSWITHBUTTONS);

        var allItems = this.model.getAllItemsExceptButtons();

        deepEqual(allItems, [ITEM_1, ITEM_2]);
    });

    test("Returns all items which match with given name", function() {
        this.model.set("itemGroups", ITEMSWITHBUTTONS);

        var allItems = this.model.getAllItemsWithName("item 2");

        deepEqual(allItems, [{items: [ITEM_2]}, {items: [ITEM_3], type: "buttons"}]);
    });

    test("Test not manageable if no items in manageble group", function() {
        this.model.set("itemGroups", [
            {
                allowDelete: true,
                items: []
            },
            {
                items: [ITEM_1]
            }
        ]);

        ok(!this.model.isManageable(), "Expected group not to be manageable");
    });

    test("Test not manageable if no allowDelete or allowReorder", function() {
        this.model.set("itemGroups", [
            {
                items: []
            },
            {
                items: [ITEM_1]
            }
        ]);

        ok(!this.model.isManageable(), "Expected group not to be manageable");
    });

    test("Test is manageable", function() {
        this.model.set("itemGroups", [
            {
                items: []
            },
            {
                allowDelete: true,
                items: [ITEM_1]
            }
        ]);
        ok(this.model.isManageable(), "Expected group to be manageable");
    });

    test("Test hasDisplayableGroups", function () {
        this.model.set("itemGroups", [{items: []}]);
        ok(!this.model.hasDisplayableGroups());
        this.model.set("itemGroups", [{emptyText: "empty", items: []}]);
        ok(this.model.hasDisplayableGroups());
        this.model.set("itemGroups", [{items: [ITEM_1]}]);
        ok(this.model.hasDisplayableGroups());
    });
});
