/**
 * This object controls the normal dropdown scopefilter and collapsed  inline dialog scope filter. ie:
 * - it fetches and shares data between the scopefilter views mentioned above
 * - it listens to sidebar events and initializes them - sort of a route controller
 **/
define('jira/projects/sidebar/scopefilter/controller', [
    'jquery',
    'wrm/require',
    'wrm/data',
    'jira/api/projects',
    'wrm/context-path',
    'jira/util/logger',
    'jira/projects/libs/marionette',
    'jira/projects/sidebar/scopefilter/model',
    'jira/projects/sidebar/scopefilter/scopefilter-view',
    'jira/projects/sidebar/scopefilter/collapsed-view'
], function ($,
             wrmRequire,
             wrmData,
             Projects,
             contextPath,
             logger,
             Marionette,
             ScopeFilterModel,
             ScopeFilterDropdownView,
             CollapsedView) {
    'use strict';

    return Marionette.Controller.extend({
        initialize: function (options) {
            this.sidebarAPI = options.sidebarAPI;
            this.sidebar = this.sidebarAPI.getAUISidebar();
            this.$sidebarContentContainer = this.sidebarAPI.getContentContainer();

            if (!this.sidebarAPI.isProjectSidebar()) {
                return;
            }
            var scopeFilterData = wrmData.claim('scope-filter-data');
            this.scopeFilterModel = new ScopeFilterModel(scopeFilterData);
            this.dataFetched = false;
            // if we have data on page load it means we already rendered it on server side
            this.shouldRenderScopeFilterView = !this.scopeFilterModel.getSelectableScopes().length;

            if (this.sidebar.isCollapsed()) {
                // create the collapsed menu icon
                this.createCollapsedScopeFilterView()
            }
            else {
                this.createScopeFilterView();
            }
            // we want to listen to state change of the sidebar and initialize the view that wasn't initialized on page load
            this.listenToSidebarEvents();
        },
        createCollapsedScopeFilterView: function () {

            // Do not display this in JSD or any other project sidebar
            if(!this.scopeFilterModel.shouldDisplayWhenSidebarIsCollapsed()) {
                return;
            }

            this.collapsedView = new CollapsedView({
                model: this.scopeFilterModel,
                controller: this
            });
            this.collapsedView.render();
            this.$sidebarContentContainer.prepend(this.collapsedView.el);

            logger.trace('sidebar.scopefilter.collapsed');
        },
        createScopeFilterView: function () {
            if(!this.shouldRenderScopeFilterView){
                return;
            }

            this.scopeFilterDropdownView = new ScopeFilterDropdownView({
                el: '#sidebar-scope-filter-list',
                model: this.scopeFilterModel,
                controller: this
            });
        },
        // TBD maybe this mechanism should be inside the model like an overridden backbone fetch
        fetchScopeFilterList: function () {
            var deferred = $.Deferred();

            /**
             * if we haven't fetched scope list but we have items it means we have fall back to old plugin behaviour
             * (plugin did not specify separate web-item for selected item, hence we aren't lazy loading)
             * or we already fetched and claimed data, so we shouldn't claim it again since wrmData.claim will throw error
             */
            if (this.dataFetched || this.scopeFilterModel.getSelectableScopes().length) {
                return deferred.resolve();
            }
            var url = contextPath()
                + "/rest/projects/1.0/project/"
                + Projects.getCurrentProjectKey()
                + "/rapidviews";

            return $.ajax({
                contentType: "application/json",
                type: "GET",
                url: url
            }).done(function (data) {
                this.dataFetched = true;
                this.scopeFilterModel.setData(data);
            }.bind(this));
        },
        listenToSidebarEvents: function () {
            this.sidebar.on('collapse-start', function () {
                if (this.collapsedView) {
                    return;
                }
                this.createCollapsedScopeFilterView();
            }.bind(this));

            this.sidebar.on('expand-end', function () {
                if (this.collapsedView) {
                    this.collapsedView.destroy();
                    this.collapsedView = null;
                }
                if (this.scopeFilterDropdownView) {
                    return;
                }
                this.createScopeFilterView();
            }.bind(this));
        }
    });
});
