AJS.test.require(["com.atlassian.jira.jira-projects-plugin:reports-link-storage"], function(){
    "use strict";

    var Deferred = require("jira/jquery/deferred");
    var ProjectsAPI = require("jira/api/projects");
    var LAST_VIEWED_REPORT = "http://atlassian.com/report";

    module("jira/projects/sidebar/reports/link-storage", {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.stubSidebarApi();
            this.stubLocalStorage();
        },

        stubCurrentProject: function(projectKey) {
            this.context.mock("jira/api/projects", ProjectsAPI);
            this.stub(ProjectsAPI, "getCurrentProjectKey").returns(projectKey);
        },

        stubSidebarApi: function() {
            var sidebar = this.sidebarApi = {
                isProjectSidebar: sinon.stub(),
                getSelectedScopeFilterId: sinon.stub(),
                setReportsItemLink: sinon.spy()
            };
            this.context.mock("jira/api/projects/sidebar", {
                getSidebar: function() { return new Deferred().resolve(sidebar); }
            });
        },

        stubLocalStorage: function() {
            this.localStorageStub = {
                key: undefined,
                value: undefined,
                setItem: function(key, value) {
                    this.key = key;
                    this.value = value;
                },
                getItem: function(key) {
                    if (this.key === key) {
                        return this.value;
                    }
                }
            };
            this.context.mock("jira/data/local-storage", this.localStorageStub);
        }
    });

    test("Saves the last viewed report link on local storage when there is no scope filter", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.storeLastViewedReportLink(LAST_VIEWED_REPORT);

        equal(this.localStorageStub.key, "last.viewed.report.PRKEY");
        equal(this.localStorageStub.value, LAST_VIEWED_REPORT);
    });

    test("Saves the last viewed report link on local storage when there is scope filter", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);
        this.sidebarApi.getSelectedScopeFilterId.returns("scope-filter-id");

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.storeLastViewedReportLink(LAST_VIEWED_REPORT);

        equal(this.localStorageStub.key, "last.viewed.report.PRKEY.scope-filter-id");
        equal(this.localStorageStub.value, LAST_VIEWED_REPORT);
    });

    test("Changes the url of the reports link when storing the last viewed report", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.storeLastViewedReportLink(LAST_VIEWED_REPORT);

        sinon.assert.calledWith(this.sidebarApi.setReportsItemLink, LAST_VIEWED_REPORT);
    });

    test("Does not store anything on local storage when storing the last viewed report link if the sidebar is global", function () {
        this.sidebarApi.isProjectSidebar.returns(false);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.storeLastViewedReportLink(LAST_VIEWED_REPORT);

        equal(this.localStorageStub.key, undefined);
        equal(this.localStorageStub.value, undefined);
    });

    test("Does not change the url of the reports link when storing the last viewed report link if the sidebar is global", function () {
        this.sidebarApi.isProjectSidebar.returns(false);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.storeLastViewedReportLink(LAST_VIEWED_REPORT);

        sinon.assert.callCount(this.sidebarApi.setReportsItemLink, 0);
    });

    test("Changes the reports link to the value saved in local storage when restoring the link and there is scope filter", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);
        this.sidebarApi.getSelectedScopeFilterId.returns("scope-filter-id");

        this.localStorageStub.setItem("last.viewed.report.PRKEY.scope-filter-id", LAST_VIEWED_REPORT);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.restoreLastViewedReportLink();

        sinon.assert.calledWith(this.sidebarApi.setReportsItemLink, LAST_VIEWED_REPORT);
    });

    test("Changes the reports link to the value saved in local storage when restoring the link and there is no scope filter", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);

        this.localStorageStub.setItem("last.viewed.report.PRKEY", LAST_VIEWED_REPORT);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.restoreLastViewedReportLink();

        sinon.assert.calledWith(this.sidebarApi.setReportsItemLink, LAST_VIEWED_REPORT);
    });

    test("Does not change the reports link when restoring the link if there is nothing saved in local storage", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(true);

        this.localStorageStub.setItem("last.viewed.report.PRKEY", undefined);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.restoreLastViewedReportLink();

        sinon.assert.callCount(this.sidebarApi.setReportsItemLink, 0);
    });

    test("Does not change the reports link when restoring the link if the sidebar is global", function () {
        this.stubCurrentProject("PRKEY");
        this.sidebarApi.isProjectSidebar.returns(false);

        this.localStorageStub.setItem("last.viewed.report.PRKEY", LAST_VIEWED_REPORT);

        var reportLinkStorage = this.context.require("jira/projects/sidebar/reports/link-storage");
        reportLinkStorage.restoreLastViewedReportLink();

        sinon.assert.callCount(this.sidebarApi.setReportsItemLink, 0);
    });

});

