AJS.test.require("com.atlassian.jira.jira-projects-plugin:sidebar-last-visited", function() {
    "use strict";

    var wrmContextPath = require("wrm/context-path");
    var Projects = require("jira/api/projects");
    var Backbone = require("jira-projects-backbone");
    var _ = require("underscore");
    var Updater = require("jira/projects/sidebar/lastvisited/updater");

    module("jira/projects/sidebar/lastvisited/updater", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.server = sinon.fakeServer.create();
        },

        teardown: function() {
            this.server.restore();
            this.sandbox.restore();
        },

        mockSidebarWithSelectedItem: function() {
            return this.mockSidebar({
                getSelectedNavigationItem: function () {
                    return {
                        getId: function () {
                            return "some-selected-item";
                        }
                    };
                }
            });
        },

        mockSidebarWithoutSelectedItem: function() {
            return this.mockSidebar({
                getSelectedNavigationItem: function() {
                    return undefined;
                }
            });
        },

        mockSidebar: function(extensions) {
            return _.extend(extensions, Backbone.Events);
        },

        mockServerResponse: function(projectKey) {
            this.server.respondWith(
                "PUT",
                wrmContextPath() + "/rest/projects/1.0/project/" + encodeURIComponent(projectKey) + "/lastVisited",
                [
                    200,
                    { "Content-Type": "application/json" },
                    JSON.stringify({})
                ]
            );
        },

        buildLastVisitedUpdaterForProject: function(projectKey, sidebar, callback) {
            this.mockCurrentProject(projectKey);

            Updater.start(sidebar, callback);
        },

        mockCurrentProject: function(projectKey) {
            this.sandbox.stub(Projects, "getCurrentProjectKey").returns(projectKey);
        }
    });

    test("When the before:navigate:prevented event is triggered by the sidebar, it sends a request to update the last visited item", function() {
        var projectKey = "KEY";
        this.mockServerResponse(projectKey);
        var sidebar = this.mockSidebarWithoutSelectedItem();
        var successfulUpdateSpy = this.spy();
        this.buildLastVisitedUpdaterForProject(projectKey, sidebar, successfulUpdateSpy);
        var event = {
            emitter: {
                id: "some-id"
            }
        };

        sidebar.trigger("before:navigate:prevented", event);
        this.server.respond();

        sinon.assert.calledOnce(successfulUpdateSpy);
    });

    test("When the before:select event is triggered by the sidebar and it contains the isInitial flag, it sends a request to update the last visited item", function() {
        var projectKey = "KEY";
        this.mockServerResponse(projectKey);
        var sidebar = this.mockSidebarWithoutSelectedItem();
        var successfulUpdateSpy = this.spy();
        this.buildLastVisitedUpdaterForProject(projectKey, sidebar, successfulUpdateSpy);
        var event = {
            emitter: {
                id: "some-id"
            },
            isInitial: true
        };

        sidebar.trigger("before:select", event);
        this.server.respond();

        sinon.assert.calledOnce(successfulUpdateSpy);
    });

    test("When the sidebar is loaded with a selected item, the updater sends a request to update the last visited item to be the selected item", function() {
        var projectKey = "KEY";
        this.mockServerResponse(projectKey);
        var sidebar = this.mockSidebarWithSelectedItem();
        var successfulUpdateSpy = this.spy();
        this.buildLastVisitedUpdaterForProject(projectKey, sidebar, successfulUpdateSpy);

        this.server.respond();

        sinon.assert.calledOnce(successfulUpdateSpy);
    });


    test("When a non-project sidebar is loaded, the updater does not send any request.", function() {
        var projectKey = undefined;
        this.mockServerResponse(projectKey);
        var sidebar = this.mockSidebarWithSelectedItem();
        var successfulUpdateSpy = this.spy();

        this.buildLastVisitedUpdaterForProject(projectKey, sidebar, successfulUpdateSpy);

        equal(this.server.requests.length, 0, "No request was made to the server");
        sinon.assert.notCalled(successfulUpdateSpy);
    });
});
