define("jira/projects/sidebar/component/navigation-item", ["jira/util/browser", "jira/projects/libs/marionette"], function(browser, Marionette) {
    "use strict";

    return Marionette.CompositeView.extend({
        /**
         * @event JIRA.Projects.Sidebar.Component.NavigationItem#before:select
         * @type {EventObject}
         */

        /**
         * @event JIRA.Projects.Sidebar.Component.NavigationItem#select
         * @type {EventObject}
         */

        /**
         * @event JIRA.Projects.Sidebar.Component.NavigationItem#before:navigate
         * @type {EventObject}
         */

        /**
         * @event JIRA.Projects.Sidebar.Component.NavigationItem#before:deselect
         * @type {EventObject}
         */

        /**
         * @event JIRA.Projects.Sidebar.Component.NavigationItem#deselect
         * @type {EventObject}
         */

        ui: {
            link: "a.aui-nav-item"
        },

        events: {
            'simpleClick @ui.link:not([target="_blank"])': function (ev) {
                ev.preventDefault();
                this.navigate();
            }
        },

        /**
         * @classdesc Represents an individual item in a navigation group.
         *
         * @constructs
         * @extends JIRA.Projects.Libs.Marionette.CompositeView
         */
        initialize: function () {
            // Capture the ID of this item (if any).
            this.id = this.$el.find(">a").attr("data-link-id");

            // Ensure the `ui` and `events` hashes work as expected.
            this.bindUIElements();
        },

        onDestroy: function() {
            // This should be Marionette's responsibility, but it doesn't seem to ever get called (this might be a bug
            // in Marionette.
            this.unbind();
        },

        /**
         * Returns HTML element representing sidebar item.
         *
         * @returns {HTMLElement} HTML element representing sidebar item.
         */
        getElement: function() {
            return this.el;
        },

        /**
         * Navigates to the URL specified in the item's href.
         *
         * This method will trigger the preventable event 'before:navigate'. If it is prevented,
         * no navigation will occur.
         *
         * @fires JIRA.Projects.Sidebar.Component.NavigationItem#before:navigate
         *
         * @returns {boolean} False if before:select were prevented, true otherwise.
         */
        navigate: function () {
            // Select the element first, so the consumer has an opportunity to prevent
            // before:select and/or before:deselect
            if (!this.select()) {
                return
            }

            // Trigger before:navigate
            var navigateEvent = this.triggerPreventable("before:navigate");
            if (navigateEvent.isPrevented) {
                return false;
            }

            // Navigate to HREF
            var href = this.ui.link.attr("href");
            if (href) {
                browser.reloadViaWindowLocation(href);
            }

            return true;
        },

        /**
         * Selects the item.
         *
         * This method will throw two events, in this order:
         *
         *   * before:select (preventable)
         *   * select
         *
         * If the event 'before:select' is prevented, the element is not selected and the current selected item is not
         * deselected .
         *
         * @fires JIRA.Projects.Sidebar.Component.NavigationItem#before:select
         * @fires JIRA.Projects.Sidebar.Component.NavigationItem#select
         *
         * @returns {boolean} False if before:select were prevented, true otherwise.
         */
        select: function () {
            var selectEvent = this.triggerPreventable("before:select");
            if (selectEvent.isPrevented) {
                return false;
            }

            // Main operation
            this.$el.addClass("aui-nav-selected");

            // Trigger select
            this.trigger("select", selectEvent);

            return true;
        },

        /**
         * Deselects the item.
         *
         * This method will throw two events, in this order:
         *
         *   * before:deselect (preventable)
         *   * deselect
         *
         * If the event 'before:deselect' is prevented, the element is not deselected.
         *
         * @fires JIRA.Projects.Sidebar.Component.NavigationItem#before:deselect
         * @fires JIRA.Projects.Sidebar.Component.NavigationItem#deselect
         *
         * @returns {boolean} False if before:deselect was prevented, true otherwise.
         */
        deselect: function () {
            if (!this.isSelected()) {
                return true;
            }

            var event = this.triggerPreventable("before:deselect");
            if (event.isPrevented) {
                return false;
            }

            this.$el.removeClass("aui-nav-selected");
            this.$el.find('a').blur();
            this.trigger("deselect", event);
            return true;
        },

        /**
         * Checks if the element is selected.
         *
         * @returns {Boolean} Whether the element is selected.
         */
        isSelected: function () {
            return this.$el.hasClass("aui-nav-selected");
        },

        /**
         * Removes the badge of this navigation item
         */
        removeBadge: function() {
            this.$el.find('.aui-badge').remove();
        },

        getId: function() {
            return this.id;
        },

        getSelectedNavigationItem: function() {
            if (this.isSelected()) {
                return this;
            }
        },

        hasASelectedItem: function() {
            return this.isSelected();
        }
    });
});
