define('jira/projects/sidebar/component', [
    "jira/projects/sidebar/component/navigation-group",
    "jira/projects/libs/marionette",
    'jira/util/logger',
    'wrm/data',
    'jquery',
    'underscore'
], function(NavigationGroup, Marionette, logger, wrmData, jQuery, _) {
        'use strict';

        var isGlobalSidebar = Boolean(wrmData.claim('is-global-sidebar'));

        return Marionette.CompositeView.extend({
            childEvents: {
                'before:select': function(item, itemEvent) {
                    if (!this.hasASelectedItem()) {
                        itemEvent.isInitial = true;
                    }
                    this.retriggerPreventable('before:select', itemEvent);
                    if (!itemEvent.isPrevented) {
                        this.deselectAllGroups();
                    }
                },
                'select': function(child, ev) {
                    this.trigger('select', ev);
                },
                'before:deselect': function(item, itemEvent) {
                    this.retriggerPreventable('before:deselect', itemEvent);
                },
                'deselect': function(child, ev) {
                    this.trigger('deselect', ev);
                },
                'before:navigate': function(item, itemEvent) {
                    this.retriggerPreventable('before:navigate', itemEvent);
                },
                "before:navigate:prevented": function(item, itemEvent) {
                    this.trigger("before:navigate:prevented", itemEvent);
                }
            },

            initialize: function() {
                // There should already be html in the component, so just force bind events.
                this.render({force: true});
            },

            /**
             * Render an html string for a sidebar, setup sidebar structure and sub-components, and bind child component
             * events to the root.
             *
             * @param {Object} options - a dictionary of optional parameters.
             * @param {string} options.el - The html source for the new sidebar to render
             * @param {boolean} options.force - Whether or not to 'force' rebinding of events (this happens
             *                      automatically when a new 'el' is passed in).
             * @returns {Object} reference to 'this' to support chaining (Marionette convention).
             */
            render: function(options) {
                options = _.defaults({}, options, {force: false});

                var hasNewMarkup = 'el' in options;   // Allow people to pass virtually any value, jQuery will sanitise it.
                var shouldRender = hasNewMarkup || options.force === true;
                var shouldDetach = this.children.length > 0;

                if (!shouldRender) return this;

                var preRenderEvent = this.triggerPreventable('before:render');
                if (preRenderEvent.isPrevented) return this;

                if (shouldDetach) {
                    var preDetachEvent = this.triggerPreventable('before:detach');
                    if (preDetachEvent.isPrevented) return this;
                    this.destroyChildren({checkEmpty: false});
                    this.trigger('detach');
                }

                if (hasNewMarkup) {
                    var el = jQuery(options.el);
                    this.$el.replaceWith(el);
                    this.setElement(el);
                }

                this.$('.aui-sidebar-group').each(_.bind(function(index, sidebarGroup) {
                    var navigationGroup = new NavigationGroup({
                        el: sidebarGroup
                    });
                    var alreadyExists = !!this.getGroup(navigationGroup.id);
                    if (alreadyExists) {
                        logger.warn(
                            'Duplicated IDs detected. There are more than one NavigationGroup with id data-id="' + navigationGroup.id + '"'
                        );
                    }

                    this.proxyChildEvents(navigationGroup);
                    this.children.add(navigationGroup, navigationGroup.id);
                }, this));

                this.trigger('render');
                return this;
            },

            deselectAllGroups: function() {
                this.children.call('deselect');
            },

            replaceGroup: function(groupname, newGroup) {
                var oldGroup = this.getGroup(groupname);

                // Replace the DOM
                oldGroup.$el.replaceWith(newGroup.$el);

                // Replace the group in the internal list of groups. As we don't have a
                // replace method there, we need to remove the old group and add the new one
                // using the same `cid` to try to preserve the ordering.
                newGroup.cid = oldGroup.cid;
                this.children.remove(oldGroup);
                this.children.add(newGroup, newGroup.id);
            },

            getGroup: function(id) {
                return this.children.findByCustom(id);
            },

            getGroupAt: function(index) {
                return this.children.findByIndex(index);
            },

            getItem: function(id) {
                return this.getDefaultGroup().getItem(id);
            },

            /**
             * Returns HTML element representing sidebar.
             *
             * @returns {HTMLElement} HTML element representing sidebar.
             */
            getElement: function() {
                return this.el;
            },

            getDefaultGroup: function() {
                return this.getGroup('sidebar-navigation-panel');
            },

            isProjectSidebar: function() {
                return !isGlobalSidebar;
            },

            getSelectedScopeFilterId: function() {
                return this.$('.scope-filter a.scope-filter-trigger').attr('data-scope-filter-id');
            },

            setReportsItemLink: function(reportUrl) {
                var sidebarNavigationPanel = this.getGroup('sidebar-navigation-panel');

                // by default, tries to retrieve the project-centric link
                var reportLink = sidebarNavigationPanel.getItem('com.atlassian.jira.jira-projects-plugin:report-page');
                // in case it is not there (global board sidebar), let's get the one provided by Agile
                if (typeof reportLink === 'undefined') {
                    reportLink = sidebarNavigationPanel.getItem('com.pyxis.greenhopper.jira:global-sidebar-report');
                }

                if (reportLink) {
                    reportLink.ui.link.attr('href', reportUrl);
                }
            },

            getAUISidebar: function() {
                return AJS.sidebar('.aui-sidebar');
            },

            getContentContainer: function() {
                return this.$('.aui-sidebar-body .sidebar-content-container');
            },

            getSelectedNavigationItem: function() {
                return this.getDefaultGroup().getSelectedNavigationItem();
            },

            hasASelectedItem: function() {
                return this.getDefaultGroup().hasASelectedItem();
            },

            dim: function() {
                this.$el.attr({
                    "dimmed": "",
                    "aria-hidden": "true"
                });
            },

            undim: function() {
                this.$el.removeAttr("dimmed");
                this.$el.removeAttr("aria-hidden");
            }

        });
    });
