define("jira/projects/sidebar/analytics/hash", ["underscore"], function (_) {
    "use strict";

    var prefixes = [];

    function startsWith(string, prefix) {
        return string.indexOf(prefix) === 0;
    }

    function substringAfter(string, prefix) {
        return string.substring(startsWith(string, prefix) ? prefix.length : 0);
    }

    function hashCode (string) {
        var charCode;
        var hash = 0;
        if (!string) {
            return "";
        }

        for (var i = 0; i < string.length; i += 1) {
            charCode = string.charCodeAt(i);
            hash = ((hash * 32) - hash) + charCode;
            hash |= 0; // force 32-bit representation
        }

        return hash;
    }

    return {
        /**
         * A hashed version of the provided string (for anonymising analytics data).
         *
         * If the id starts with a 'whitelisted' prefix, the prefix + the hashCode
         * of the remainder will be returned. e.g. "prefix-foo -> prefix-101574"
         *
         * Hashing uses the same algorithm as Java's String.hashCode
         * (good for anonymity, not suitable for security purposes)
         *
         * @param string The string to hash
         * @returns {String} The whitelisted prefix of the string, followed by
         * the 'hash code' of the remainder
         */
        prefixedHash: function (string) {
            var prefix = _.find(prefixes, _.partial(startsWith, string)) || "";
            return {
                prefix: prefix,
                hash: prefix + hashCode(substringAfter(string, prefix))
            };
        },

        /**
         * Register an id prefix as safe for exclusion from the hashing process.
         *
         * e.g. If the prefix "prefix-" is registered then "prefix-foo" will
         * hash to "prefix-101574"
         *
         * @param prefix The prefix to register as safe
         */
        addPrefix: function (prefix) {
            if (prefix && !_.contains(prefixes, prefix)) {
                prefixes.push(prefix);
            }
        },

        /**
         * Clear all whitelisted prefixes (see {@link addPrefix})
         */
        clearPrefixes: function () {
            prefixes.length = 0;
        }
    };

});
