/* global Calendar */
define("jira/projects/release/submit-view", [
    "jira/projects/release/templates",
    "underscore",
    "jira/moment",
    "jira/util/formatter",
    "jira/api/projects",
    "jira/projects/abstract-list/submit-view",
    "jira/projects/release/submit-model-converter",
    "jira/projects/release/calendar-params-data"
], function(
    Templates,
    _,
    moment,
    formatter,
    ProjectsAPI,
    AbstractSubmitView,
    SubmitModelConverter,
    CalendarParams
) {
    "use strict";

    var EMPTY_ERROR = {};
    var IS_VALIDATION_CALL = {};

    return AbstractSubmitView.extend({
        template: Templates.addVersion,

        ui: {
            "form": "form",
            // input
            allInputs: "input[name]",
            nameInput: "input[name='name']",
            dateStartInput: "input[name='startDate']",
            dateStartCalendarTrigger: ".releases-add__date-start a",
            dateReleaseInput: "input[name='releaseDate']",
            dateReleaseCalendarTrigger: ".releases-add__date-release a",
            descriptionInput: "input[name='description']",
            // errors
            "err_nameInput": ".releases-add__name .error-container",
            "err_dateStartInput": ".releases-add__date-start .error-container",
            "err_dateReleaseInput": ".releases-add__date-release .error-container",
            // buttons
            addButton: ".releases-add__confirm button"
        },

        events: {
            "submit @ui.form": "_onSubmit",
            "input @ui.nameInput": "_validateForm",
            "change @ui.dateStartInput": "_validateStartDate",
            "change @ui.dateReleaseInput": "_validateReleaseDate"
        },

        validation: {
            "nameInput": ["_uniqueName", "_constraint", "_empty"]
        },

        onRender: function () {
            this._setupCalendar(this.ui.dateStartCalendarTrigger, this.ui.dateStartInput);
            this._setupCalendar(this.ui.dateReleaseCalendarTrigger, this.ui.dateReleaseInput);

            this._validateForm();
        },

        _setupCalendar: function (button, inputField) {
            var params = _.extend({}, CalendarParams, {
                button: button,
                inputField: inputField
            });
            Calendar.setup(params);
        },

        _getFormData: function () {
            var data = this.ui.form.serializeArray().reduce(function (r, param) {
                r[param.name] = param.value;
                return r;
            }, {});

            data.project = ProjectsAPI.getCurrentProjectKey();
            data.startDate = SubmitModelConverter.getDate(data.startDate);
            data.releaseDate = SubmitModelConverter.getDate(data.releaseDate);
            data.status = SubmitModelConverter.getDefaultStatus();

            return data;
        },

        _validateForm: function (event) {
            var allErrors = _.flatten([
                this._validateName(),
                this._validateStartDate(IS_VALIDATION_CALL),
                this._validateReleaseDate(IS_VALIDATION_CALL)
            ]);

            if (allErrors.length > 0) {
                this.ui.addButton.attr("disabled", "disabled");
            } else if (this.ui.addButton.is(':disabled')) {
                this.ui.addButton.removeAttr("disabled");

                // focus on add button, otherwise browser would skip to next element even though button is already enabled
                if (event && this.ui.descriptionInput.is(event.target)) {
                    this.ui.addButton.focus();
                }
            }

            return allErrors;
        },

        _onSubmit: function (event) {
            event.preventDefault();

            if (this._validateForm(event).length > 0) {
                return;
            }

            this.model.clear();
            this.model.set(this._getFormData());
            this.trigger("save", this.model);
            this.model.clear();

            if (!this.isDestroyed) {
                this.ui.allInputs.val("");
                this.ui.nameInput.focus();

                this._validateForm();
            }
        },

        _validateName: function () {
            return _.flatten(_.map(this.validation, function (validators, fieldName) {
                var value = this.ui[fieldName].val();
                var errors = _.flatten(validators.map(function (fn) {
                    return this[fn](value);
                }, this));

                var messageErrors = errors.filter(function (error) {
                    return error !== EMPTY_ERROR;
                });
                if (messageErrors.length > 0) {
                    this._showError(fieldName, messageErrors.join(', '));
                } else {
                    this._hideError(fieldName);
                }

                return errors;
            }, this));
        },

        _validateStartDate: function (event) {
            return this._validateDate("dateStartInput", "dateReleaseInput", true, event) || [];
        },

        _validateReleaseDate: function (event) {
            return this._validateDate("dateReleaseInput", "dateStartInput", false, event) || [];
        },

        _validateDate: function (fieldName, anotherFieldName, startDateFirst, isValidationFormCall) {
            var value = this.ui[fieldName].val().trim();
            var error;
            if (value.length == 0) {
                this._hideError(fieldName);
                if (isValidationFormCall !== IS_VALIDATION_CALL) {
                    this._validateForm();
                }
                return;
            }

            value = this._correctFormat(value);
            if (!value) {
                error = formatter.I18n.getText("project.page.release.add.invalid.date.format", CalendarParams.ifFormat);
            } else {
                this.ui[fieldName].val(value);
                var anotherValue = this.ui[anotherFieldName].val().trim();
                if (anotherValue.length > 0 && this._correctFormat(anotherValue)) {
                    this._hideError(anotherFieldName);

                    if (!this._compareDates(value, anotherValue, startDateFirst)) {
                        error = formatter.I18n.getText('project.page.release.add.start.release.date.order');
                    }
                }
            }

            this._showError(fieldName, error);

            if (error) {
                this.ui.addButton.attr("disabled", "disabled");
            } else if (isValidationFormCall !== IS_VALIDATION_CALL) {
                this._validateForm();
            }
            return error;
        },

        _showError: function (fieldName, error) {
            this.ui["err_" + fieldName]
                .html(error !== undefined ? error : '')
                .toggleClass('error', error !== undefined);
        },

        _hideError: function (fieldName) {
            this.ui["err_" + fieldName]
                .html('')
                .removeClass('error');
        },

        // validation functions
        _uniqueName: function (val) {
            const value = val ? val.trim().toLowerCase() : '';
            if (this.options.pageableItems.originalCollection.some(function (itemData) {
                    return (itemData.get("name") || '').trim().toLowerCase() === value;
                })) {
                return formatter.I18n.getText("project.page.release.add.error.already.exists");
            }
            return [];
        },

        _constraint: function (val) {
            if (val && val.length > 255) {
                return formatter.I18n.getText("project.page.release.add.limit.exceeded");
            }
            return [];
        },

        _empty: function (val) {
            if (!val || val.trim().length === 0) {
                return EMPTY_ERROR;
            }
            return [];
        },

        _compareDates: function (value1, value2, startDateFirst) {
            var parts1 = moment.splitDate(value1, CalendarParams.ifFormat);
            var parts2 = moment.splitDate(value2, CalendarParams.ifFormat);

            var date1 = new Date(parts1.year, parts1.month, parts1.day);
            var date2 = new Date(parts2.year, parts2.month, parts2.day);
            if (startDateFirst) {
                return date1 <= date2;
            }
            return date2 <= date1;
        },


        /**
         * Checks if given value is a correct date and also is it formatted.
         *
         * If it is true than returns formatted string.
         * If it isn't correct than return undefined.
         *
         * @param value - string with date.
         * @returns value or undefined if it's not a date.
         */
        _correctFormat: function (value) {
            var parts = moment.splitDate(value, CalendarParams.ifFormat);

            if (parts.month === -1 || isNaN(parts.day) || isNaN(parts.year) || parts.day === 0 || parts.year === 0) {
                return undefined;
            }
            var date = new Date(parts.year, parts.month, parts.day);
            if (date.getFullYear() != parts.year || date.getMonth() != parts.month) {
                return undefined;
            }
            return value.toLowerCase();
        }
    });
});
