define("jira/projects/page/release/release-page-app", [
    "jira/util/logger",
    "jira/projects/abstract-list/router",
    "jira/projects/abstract-list/navigation-utils",
    "jira/projects/libs/marionette",
    "jira/api/projects",
    "jira/data/local-storage",
    "wrm/data",
    "jira-projects-backbone",
    "jquery",
    "jira/projects/release/searchlayoutview",
    "jira/projects/release/submit-controller",
    "jira/projects/release/pageable-versions-collection",
    "jira/projects/release/filter-model",
    "jira/projects/abstract-list/submit-model-events",
    "jira/projects/release/config-data-model",
    "jira/dialog/form-dialog",
    "jira/util/events"
], function (
    logger,
    ALRouter,
    NavigationUtils,
    Marionette,
    Projects,
    localStorage,
    WRMData,
    Backbone,
    $,
    SearchLayoutView,
    SubmitController,
    PageableVersionsCollection,
    FilterModel,
    SubmitModelEvents,
    ConfigDataModel,
    FormDialog,
    Events
) {
    "use strict";

    var isProjectAdmin = WRMData.claim("com.atlassian.jira.projects.page.release:projectAdmin");

    var App = new Marionette.Application();

    function isOnProject() {
        return Projects.getCurrentProjectId() !== undefined;
    }

    App.on("start", function (options) {
        if (!isOnProject()) {
            return;
        }

        var projectId = Projects.getCurrentProjectId();
        var projectKey = Projects.getCurrentProjectKey();

        var storageStatusKey = "jira-projects:release-page-status-" + projectId;

        //required by tests
        var $head = $("head");
        $head.append($('<meta>').attr("name", "projectKey").attr("content", Projects.getCurrentProjectKey()));
        //required by delete version dialog
        $head.append($('<meta>').attr("name", "projectId").attr("content", Projects.getCurrentProjectId()));

        var $el = options.container;

        var releaseLearnMoreUrl = WRMData.claim("com.atlassian.jira.projects.page.release:learnMoreUrl");

        function saveStatus(status) {
            try {
                localStorage.setItem(storageStatusKey, status);
            } catch (exc) {
                // local storage may be disabled or full, do not do anything in this case
                logger.warn(exc);
            }
        }

        App.pageableItems = new PageableVersionsCollection(null, {
            data: options.data,
            projectKey: projectKey
        });

        App.filter = new FilterModel({
            pageableCollection: App.pageableItems,
            status: "unreleased",
            contains: ""
        });
        App.filter.filterCollection();
        App.filter.on("change:status", function (child, value) {
            saveStatus(value);
        });
        App.pageableItems.filter = App.filter;
        App.configDataModel = new ConfigDataModel();

        App.submitController = new SubmitController({
            pageableItems: App.pageableItems,
            isProjectAdmin: isProjectAdmin
        });

        App.addRegions({
            mainRegion: new Marionette.Region({el: $el})
        });
        App.mainRegion.on("show", function onMainRegionShow() {
            // Add page data for the version dialogs from releaseVersions.js
            // releaseVersions.js: re-read properties, specifically AJS.params.dateFormat that added in .calendarConfiguration
            AJS.populateParameters();
        });
        App.mainRegion.show(new SearchLayoutView({
            filter: App.filter,
            pageableItems: App.pageableItems,
            releaseLearnMoreUrl: releaseLearnMoreUrl,
            submitController: App.submitController,
            isProjectAdmin: isProjectAdmin,
            configDataModel: App.configDataModel
        }));
        App.mainRegion.currentView.on("progressDialogClicked", function () {
            App.trigger("progressDialogClicked");
        });
        App.mainRegion.currentView.on("navigate", onNavigate);

        App.router = new ALRouter();
        App.router.on("route:any", function () {
            var status = NavigationUtils.getQueryStringParam("status", true) || "unreleased";
            var contains = NavigationUtils.getQueryStringParam("contains", true) || "";
            var page = +NavigationUtils.getQueryStringParam("page", true) || 1;

            App.filter.set("status", status, {silent: true});
            saveStatus(status);
            App.filter.set("contains", contains, {silent: true});
            App.filter.filterCollection();
            App.mainRegion.currentView.trigger('filterChanged');
        });

        function onNavigate() {
            App.trigger("navigate");
        }

        App.filter.on("navigate", onNavigate);
        App.router.on("route:any", onNavigate);

        try {
            var storedStatus = localStorage.getItem(storageStatusKey);
            if (!NavigationUtils.getQueryStringParam("status", true) && isOnProject() && !!storedStatus) {
                App.filter.set("status", storedStatus, {silent: true});
                NavigationUtils.append("status", storedStatus);
            }
        } catch (exc) {
            // local storage may be disabled or full, do not do anything in this case
            logger.warn(exc);
        }

        if (Backbone.history && Backbone.History && !Backbone.History.started) {
            Backbone.history.start({
                root: NavigationUtils.getBackboneHistoryRoot(),
                pushState: NavigationUtils.pushStateSupported
            });
        }

        Events.bind(SubmitModelEvents.SERVER_ERROR, function (e, data) {
            var serverErrorConsole = $("#project-config-error-console");
            if (data && data.errorMessages) {
                // Replace any pre-existing messages
                serverErrorConsole.empty();
                new FormDialog({
                    id: "server-error-dialog",
                    content: function (callback) {
                        callback(JIRA.Templates.Common.serverErrorDialog({
                            message: data.errorMessages[0]
                        }));
                    }
                }).show();
            }
        });
    });

    return App;
});