define('jira/projects/release/filter-model', [
    "jira/projects/abstract-list/filter-model",
    "jira/projects/abstract-model/submit-model-states"
], function(ALFilterModel, SubmitModelStates) {
    "use strict";

    var AbstractFilterModel = ALFilterModel;

    var FilterModel = AbstractFilterModel.extend({
        filterAttributes: ['status', 'contains'],
        defaults: {
            status: "unreleased",
            contains: ""
        },

        initialize: function initialize(options) {
            AbstractFilterModel.prototype.initialize.call(this, options);

            this.on("change:status change:contains", this.filterCollection);
            this.on("change:status change:contains", this.navigate);
        },

        _getStatus: function _getStatus() {
            return this.get("status") || FilterModel.FILTER_NONE;
        },

        _getContains: function _getContains() {
            return (this.get("contains") || "").toLowerCase();
        },

        emptyContainsField: function () {
            this.set('contains', '');
        },

        filterCollection: function filterCollection() {
            var filteredElements = this.pageableCollection.originalCollection.models.filter(this.elementMatchFilter, this);

            this.pageableCollection.resetFilter(filteredElements);
        },

        _statusMatches: function _statusMatches(element) {
            var status = this._getStatus();
            if (status !== FilterModel.FILTER_ALL && status !== FilterModel.FILTER_NONE) {
                if (element.get('archived')) {
                    return this.containsStatus("archived");
                } else {
                    return (this.containsStatus("released") && element.get('released'))
                        || (this.containsStatus("unreleased") && !element.get('released'));
                }
            }
            return true;
        },

        elementMatchFilter: function elementMatchFilter(element) {
            if (element.get("state") === SubmitModelStates.IN_PROGRESS ||
                element.get("state") === SubmitModelStates.IN_PROGRESS_ERROR) {
                return true;
            }

            if (!this._statusMatches(element)) {
                return false;
            }

            var contains = this._getContains();
            return contains === "" || (
                    element.get('name').toLowerCase().indexOf(contains) > -1 ||
                    element.get('description').toLowerCase().indexOf(contains) > -1 ||
                    element.get('startDate').formatted.toLowerCase().indexOf(contains) > -1 ||
                    element.get('releaseDate').formatted.toLowerCase().indexOf(contains) > -1
                );
        },

        containsStatus: function (s) {
            var status = this.get("status") || FilterModel.FILTER_NONE;
            return status.indexOf(s) === 0 || status.indexOf("-" + s) > 0;
        }
    }, {
        FILTER_ALL: 'all',
        FILTER_NONE: 'no-filter'
    });

    return FilterModel;
});
