define("jira/projects/components/submitmodel", [
    "jira/util/logger",
    "jira/flag",
    "jira/ajs/ajax/ajax-util",
    "jira-projects-backbone",
    "underscore",
    "jquery",
    "wrm/context-path",
    "jira/projects/abstract-model/submit-model-states",
    "jira/projects/abstract-list/submit-model",
    "jira/projects/components/analytics"
], function(
    logger,
    jiraFlag,
    ajaxUtil,
    Backbone,
    _,
    $,
    contextPath,
    SubmitModelStates,
    AbstractSubmitModel,
    Analytics
) {
    var SubmitModel = AbstractSubmitModel.extend({
        defaults: _.extend({}, AbstractSubmitModel.prototype.defaults, {
            issuesCount: 0,
            archived: false
        }),

        urlRoot: contextPath() + "/rest/api/2/component",
        detailsPath: function() {
            return contextPath() + '/rest/projects/1.0/project/' + encodeURIComponent(this.get("project")) + '/component/' + this.id;
        },

        initialize: function() {
            Analytics.listenForActionsOnItem(this);
        },

        /**
         * Overrides default save handler to only save (send to server) attributes that have changed.
         * Also provides some default error handling.
         *
         * @override
         * @param attributes
         * @param options
         */
        save: function (attributes, options) {
            this.lastSaveAttributes = _.clone(attributes);
            options = options || {};

            // if it is a new model, we don't have to worry about updating only changed attributes because they are all new
            if (this.isNew()) {
                // call super
                return AbstractSubmitModel.prototype.save.call(this, attributes, options);
            } else if (attributes) {
                var deferred = new $.Deferred();

                this.set("state", SubmitModelStates.isStateError(this.get("state")) ? SubmitModelStates.IN_PROGRESS_ERROR : SubmitModelStates.IN_PROGRESS);

                var error = options.error;
                var success = options.success;

                Backbone.Model.prototype.save.call(this, [], {
                    attrs: attributes
                }).done(function (xhr) {
                    this.detailsSync().always(function () {
                        this.set("state", SubmitModelStates.SUCCESSFUL);
                        deferred.resolve();
                        if (success) {
                            success.call(this, this, xhr);
                        }
                    }.bind(this));
                }.bind(this))
                    .fail(function (xhr) {
                        this.set("state", SubmitModelStates.ERROR_UPDATE);
                        deferred.reject(xhr);
                        if( error ) {
                            var data = $.parseJSON(xhr.responseText || xhr.data);
                            error.call(this, this, data, xhr);
                        }
                    }.bind(this));

                return deferred;
            }
        },

        destroy: function(options) {
            this.lastDestroyOptions = _.clone(options);
            var moveIssuesTo = parseInt(options.data.moveIssuesTo);
            var collection = this.collection;

            // should not send JSON in request body, end point coming from projects config plugin expecting query params instead
            options.url = this.url() + '?' + $.param(options.data);
            delete options.data;

            // do not remove model immediately, wait for REST result
            options.wait = true;

            //if model isNew then destroy returns false
            var result = Backbone.Model.prototype.destroy.apply(this, arguments);
            if (result) {
                if (moveIssuesTo > 0) {
                    this.trigger("swapItemEvent");
                    result.done(function () {
                        var targetCmp = collection.get(moveIssuesTo);
                        if (targetCmp) {
                            targetCmp.detailsSync().always(function () {
                                logger.trace("jira.components.delete.finished");
                            });
                        } else {
                            logger.trace("jira.components.delete.finished");
                        }
                    }.bind(this));
                } else {
                    result.always(function () {
                        logger.trace("jira.components.delete.finished");
                    })
                }

                this.trigger("removeItemEvent");

                // remove components from all collections, it is refactored on other branch
                result.fail(function (xhr) {
                    this.set("state", SubmitModelStates.ERROR_DELETE);
                    this.set("errorMsg", AJS.escapeHtml(ajaxUtil.getErrorMessageFromXHR(xhr)));
                }.bind(this));
            }

            return result;
        },

        toDataJSON: function() {
            return _.pick(this.toJSON(), "project", "description", "leadUserName", "name", "assigneeType");
        },

        getRelatedIssueCount: function(options) {
            options.success(this.get("issuesCount"));
        },

        getSwapComponentsJSON: function() {
            var currentId = this.get('id');
            return this.collection.filter(function(model) {
                return !model.get('archived') && model.get('id') !== currentId;
            }).map(function(model) {
                return model.toJSON();
            });
        },

        _traceItemAdded: function () {
            logger.trace("jira.components.add.finished");
        }
    });

    return SubmitModel;
});
