define("jira/projects/abstract-list/lazy-loading-collection", [
    "jira-projects-backbone",
    "jquery",
    "jira/util/logger"
], function (Backbone,
             $,
             logger
) {
    "use strict";

    /**
     * Collection which is able to load another set of items via calling 'nextPage()'.
     * Elements optionally can be filtered using 'this.filter.elementMatchFilter' method.
     *
     * Current implementation assumes that all elements are pre-loaded and contained under 'this.originalCollection'.
     * All elements applicable to selected filter are contained under 'this.fullCollection'.
     * Currently visible elements are just accessible as regular collection.
     *
     */
    var LazyLoadingCollection = Backbone.Collection.extend({
        collectionClass: null,
        lazySpinnerContainerElement: $([]),
        versionsTableElement: $([]),
        elementsInProcessingCount: 0,

        _findAddingIndex: function (collection, addingElement) {
            return 0;
        },

        initialize: function initialize(items, options) {
            this.originalCollection = new this.collectionClass(options.data);
            this.fullCollection = new this.collectionClass(this.originalCollection.models);

            this.nextPage();
            this.listenTo(this, 'remove', function () {
                if (this.originalCollection.length <= 1) {
                    this.trigger('collectionChanged');
                }
            });
        },

        lazySpinnerContainer: function () {
            if (this.lazySpinnerContainerElement.length === 0) {
                this.lazySpinnerContainerElement = $('.lazy-spinner-container');
            }

            return this.lazySpinnerContainerElement;
        },

        versionsTable: function () {
            if (this.versionsTableElement.length === 0) {
                this.versionsTableElement = $('#versions-table');
            }

            return this.versionsTableElement;
        },

        addSlice: function (newslice) {
            this.add(newslice);
            this.elementsInProcessingCount = 0;

            this.lazySpinnerContainer().removeClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE);
            this.versionsTable().removeClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING);
        },

        callLoadProgress: function (collectionSlice) {
            var progressResult = this.loadProgress(collectionSlice);

            if (Array.isArray(progressResult)) {
                this.addSlice(progressResult);
            } else if (progressResult instanceof Promise) {
                progressResult
                    .then(this.addSlice.bind(this))
                    //we use such exotic notation because YUI (which we use as a JS parser) does not know '.catch' function
                    ['catch'](logger.warn);
            } else {
                logger.info('Result of load progress should be either array or Promise.');
            }
        },

        nextPage: function (areFiltersRefreshed) {
            if (this.isFullyLoaded() && !areFiltersRefreshed) {
                return false;
            }

            if (this.elementsInProcessingCount !== 0 && !areFiltersRefreshed) {
                return false;
            }

            var collectionSlice = this.fullCollection.slice(
                this.size(),
                Math.min(
                    this.size() + LazyLoadingCollection.PAGE_INCREMENT,
                    this.fullCollection.size()
                ));

            if (this.loadProgress) {
                this.lazySpinnerContainer().addClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE);
                this.versionsTable().toggleClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING, this.size() + collectionSlice.length !== 0);

                this.elementsInProcessingCount = collectionSlice.length;
                this.callLoadProgress(collectionSlice);
            } else {
                this.add(collectionSlice);
            }

            return !this.isFullyLoaded();
        },

        isFullyLoaded: function isFullyLoaded() {
            return this.size() + this.elementsInProcessingCount >= this.fullCollection.size();
        },

        updateElement: function (element) {
            var id = element.get('id');
            var elementJson = element.toJSON();

            var model = this.fullCollection.get(id);

            if (model) {
                model.set(elementJson);
            }

            this.originalCollection.filter(function (originalElement) {
                return originalElement.id == id;
            }).some(function (originalElement) {
                originalElement.set(elementJson);
            });

            return model;
        },

        addElement: function (addedElement) {
            this.originalCollection.add(addedElement, { at: this._findAddingIndex(this.originalCollection, addedElement) });

            if (this.filter.elementMatchFilter(addedElement)) {
                var index = this._findAddingIndex(this.fullCollection, addedElement);

                this.fullCollection.add(addedElement, { at: index });

                if (index <= this.size()) {
                    this.add(addedElement, { at: index });
                }
            }
            else {
                this.trigger('added-hidden-item', addedElement);
            }

            if (this.size() <= 1) {
                this.trigger('update');
            }

            this.trigger('collectionChanged');
        },

        resetFilter: function (filteredElements) {
            this.fullCollection.reset(filteredElements);
            this.reset();
            this.nextPage(true);
        },

        elementsLeft: function () {
            return {
                elementsTotal: this.fullCollection.size(),
                elementsLoaded: this.size() + this.elementsInProcessingCount,
                elementsLeft: this.fullCollection.size() - this.size() - this.elementsInProcessingCount
            };
        }
    });

    LazyLoadingCollection.PAGE_INCREMENT = 25;
    LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE = 'spinner-visible';
    LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING = 'background-loading';

    return LazyLoadingCollection;
});