AJS.test.require(["com.atlassian.jira.jira-projects-plugin:abstract-list-page"], function () {
    "use strict";

    var _ = require("underscore");
    var $ = require("jquery");
    var Backbone = require("jira-projects-backbone");
    var LazyLoadingCollection = require("jira/projects/abstract-list/lazy-loading-collection");

    var DummyModel = Backbone.Model.extend({});
    var DummyCollection = Backbone.Collection.extend({
        model: DummyModel
    });

    var TestLazyLoadingCollection = LazyLoadingCollection.extend({
        collectionClass: DummyCollection
    });

    function stubAddingMethods(collection) {
        collection.originalCollection.add = sinon.spy();
        collection.fullCollection.add = sinon.spy();
        collection.add = sinon.spy();

        collection.filter.elementMatchFilter = sinon.stub();
        collection._findAddingIndex = sinon.stub();
    }

    module('jira/projects/abstract-list/lazy-loading-collection', {
        setup: function () {
            this.stubModelDataJson = function stubModelDataJson(count) {
                return _.range(count).map(function (id) {
                    return {'id': id};
                });
            };
        },
        teardown: function () {
        }
    });

    test('Should add next slice from full collection on nextPage()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(30)
        });

        equal(collection.size(), 25);
        equal(collection.last().id, 24);

        collection.nextPage();

        equal(collection.size(), 30);
        equal(collection.last().id, 29);
    });

    test('Should add slice on addSlice()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(30)
        });

        equal(collection.size(), 25);
        equal(collection.last().id, 24);

        collection.addSlice(collection.originalCollection.slice(25, 30));

        equal(collection.size(), 30);
        equal(collection.last().id, 29);
    });

    test('When elementsInProcessingCount is not 0 and areFiltersRefreshed is false multiple calls on nextPage() should not cause any effect', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(101)
        });

        equal(collection.nextPage(false), true);
        equal(collection.size(), 50);

        collection.elementsInProcessingCount = 25;

        equal(collection.nextPage(false), false);
        equal(collection.size(), 50);
    });

    test('When elementsInProcessingCount is not 0 and areFiltersRefreshed is true multiple calls on nextPage() should fetch data from backend', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(101)
        });

        equal(collection.nextPage(false), true);
        equal(collection.size(), 50);

        collection.elementsInProcessingCount = 25;

        equal(collection.nextPage(true), true);
        equal(collection.size(), 75);
    });

    test('Should save elementsInProcessingCount value on nextPage when loadProgress() exist and different than null', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(101)
        });

        var lazySpinnerContainerMock = $('<div></div>');
        var versionsTableMock = $('<div></div>');

        collection.lazySpinnerContainer = function () {
            return lazySpinnerContainerMock;
        };
        collection.versionsTable = function () {
            return versionsTableMock;
        };

        equal(collection.nextPage(false), true);
        equal(collection.size(), 50);

        collection.loadProgress = function () {
        };

        collection.callLoadProgress = sinon.stub();
        collection.callLoadProgress.withArgs(collection.originalCollection.slice(50, 75)).returns(null);

        equal(collection.nextPage(false), true);
        equal(collection.elementsInProcessingCount, 25);
        equal(collection.callLoadProgress.callCount, 1);

        equal(collection.versionsTable().hasClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING), true);
        equal(collection.lazySpinnerContainer().hasClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE), true);
    });

    test('Should clear elementsInProcessingCount and add slice to collection on resolution of callLoadProgress()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(101)
        });

        var lazySpinnerContainerMock = $('<div></div>');
        var versionsTableMock = $('<div></div>');

        collection.lazySpinnerContainer = function () {
            return lazySpinnerContainerMock;
        };

        collection.versionsTable = function () {
            return versionsTableMock;
        };

        equal(collection.nextPage(false), true);
        equal(collection.size(), 50);

        var slice4ThisTest = collection.originalCollection.slice(50, 75);

        collection.loadProgress = sinon.stub();
        collection.loadProgress.withArgs(slice4ThisTest)
            .returns(Promise.resolve(slice4ThisTest));
        //We assign some value to elementsInProcessingCount, so we can see if it will be set back to 0
        collection.elementsInProcessingCount = 18;
        collection.callLoadProgress(slice4ThisTest);

        //setTimeout used to make sure asserts are executed after then as result of Promise resolution
        setTimeout(function () {
            equal(collection.elementsInProcessingCount, 0);
            equal(collection.size(), 75);

            equal(collection.versionsTable().hasClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING), false);
            equal(collection.lazySpinnerContainer().hasClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE), false);
        }, 0);
    });

    test('Should do nothing on rejection of callLoadProgress()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(101)
        });

        var lazySpinnerContainerMock = $('<div></div>');
        var versionsTableMock = $('<div></div>');

        collection.lazySpinnerContainer = function () {
            return lazySpinnerContainerMock;
        };

        collection.versionsTable = function () {
            return versionsTableMock;
        };

        //We add these classes to make sure they won't be removed by loadProgress function
        collection.versionsTable().addClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING);
        collection.lazySpinnerContainer().addClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE);

        equal(collection.nextPage(false), true);
        equal(collection.size(), 50);

        var slice4ThisTest = collection.originalCollection.slice(50, 75);

        collection.loadProgress = sinon.stub();
        collection.loadProgress.withArgs(slice4ThisTest)
            .returns(Promise.reject());
        //We assign some value to elementsInProcessingCount, so we can see if it will be unchanged
        collection.elementsInProcessingCount = 18;
        collection.callLoadProgress(slice4ThisTest);

        //setTimeout used to make sure asserts are executed after then as result of Promise resolution
        setTimeout(function () {
            equal(collection.elementsInProcessingCount, 18);
            equal(collection.size(), 50);

            equal(collection.versionsTable().hasClass(LazyLoadingCollection.CSS_CLASS_BACKGROUND_LOADING), true);
            equal(collection.lazySpinnerContainer().hasClass(LazyLoadingCollection.CSS_CLASS_SPINNER_VISIBLE), true);
        }, 0);
    });

    test('Should take elementsInProcessingCount into consideration on isFullyLoaded()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(30)
        });

        equal(collection.size(), 25);

        collection.elementsInProcessingCount = 0;
        equal(collection.isFullyLoaded(), false);

        collection.elementsInProcessingCount = 5;
        equal(collection.isFullyLoaded(), true);

        collection.elementsInProcessingCount = 10;
        equal(collection.isFullyLoaded(), true);
    });

    test('Should take elementsInProcessingCount into concideration on elementsLeft()', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(40)
        });

        equal(collection.size(), 25);

        collection.elementsInProcessingCount = 0;

        equal(collection.elementsLeft().elementsTotal, 40);
        equal(collection.elementsLeft().elementsLoaded, 25);
        equal(collection.elementsLeft().elementsLeft, 15);

        collection.elementsInProcessingCount = 10;

        equal(collection.elementsLeft().elementsTotal, 40);
        equal(collection.elementsLeft().elementsLoaded, 35);
        equal(collection.elementsLeft().elementsLeft, 5);
    });

    test('Should not add slice if elementsInProcessingCount is not 0 on nextPage and areFiltersRefreshed is false', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(40)
        });

        equal(collection.size(), 25);

        //we set elementsInProcessingCount to non-zero value
        collection.elementsInProcessingCount = 25;

        equal(collection.nextPage(false), false);
        equal(collection.size(), 25);
    });

    test('Should add slice if elementsInProcessingCount is not 0 on nextPage and areFiltersRefreshed is true', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(40)
        });

        equal(collection.size(), 25);

        //we set elementsInProcessingCount to non-zero value
        collection.elementsInProcessingCount = 25;

        equal(collection.nextPage(true), false);
        equal(collection.size(), 40);
    });

    test('When all elements loaded multiple calls on nextPage() should not cause any effect', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: this.stubModelDataJson(51)
        });

        equal(collection.nextPage(), true);
        equal(collection.size(), 50);

        equal(collection.nextPage(), false);
        equal(collection.size(), 51);
    });

    test('Should add new elements on matching indexes', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: []
        });

        stubAddingMethods(collection);

        var myModel = new DummyModel();
        collection.filter.elementMatchFilter.withArgs(myModel).returns(true);
        collection._findAddingIndex.withArgs(collection.originalCollection, myModel).returns(15);
        collection._findAddingIndex.withArgs(collection.fullCollection, myModel).returns(0);

        collection.addElement(myModel);

        sinon.assert.calledWith(collection.originalCollection.add, myModel, {at: 15});
        sinon.assert.calledWith(collection.fullCollection.add, myModel, {at: 0});
        sinon.assert.calledWith(collection.add, myModel, {at: 0});
    });

    test('Should not add to the fullCollection when element does not match filter', function () {
        var collection = new TestLazyLoadingCollection(null, {
            data: []
        });

        stubAddingMethods(collection);

        var myModel = new DummyModel();
        collection.filter.elementMatchFilter.withArgs(myModel).returns(false);
        collection._findAddingIndex.withArgs(collection.originalCollection, myModel).returns(13);

        collection.addElement(myModel);

        sinon.assert.calledWith(collection.originalCollection.add, myModel, {at: 13});
        sinon.assert.notCalled(collection.fullCollection.add);
        sinon.assert.notCalled(collection.add, myModel);
    });
});